"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const assert_1 = require("@aws-cdk/assert");
const core_1 = require("@aws-cdk/core");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'can add a vpn connection to a vpc with a vpn gateway'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    asn: 65001,
                    ip: '192.0.2.1',
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::CustomerGateway', {
            BgpAsn: 65001,
            IpAddress: '192.0.2.1',
            Type: 'ipsec.1',
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkVpnConnectionCustomerGateway8B56D9AF',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: false,
        }));
        test.done();
    },
    'with static routing'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                static: {
                    ip: '192.0.2.1',
                    staticRoutes: [
                        '192.168.10.0/24',
                        '192.168.20.0/24',
                    ],
                },
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkstaticCustomerGatewayAF2651CC',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: true,
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnectionRoute', {
            DestinationCidrBlock: '192.168.10.0/24',
            VpnConnectionId: {
                Ref: 'VpcNetworkstaticE33EA98C',
            },
        }));
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnectionRoute', {
            DestinationCidrBlock: '192.168.20.0/24',
            VpnConnectionId: {
                Ref: 'VpcNetworkstaticE33EA98C',
            },
        }));
        test.done();
    },
    'with tunnel options'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: 'secretkey1234',
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                    ],
                },
            },
        });
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::VPNConnection', {
            CustomerGatewayId: {
                Ref: 'VpcNetworkVpnConnectionCustomerGateway8B56D9AF',
            },
            Type: 'ipsec.1',
            VpnGatewayId: {
                Ref: 'VpcNetworkVpnGateway501295FA',
            },
            StaticRoutesOnly: false,
            VpnTunnelOptionsSpecifications: [
                {
                    PreSharedKey: 'secretkey1234',
                    TunnelInsideCidr: '169.254.10.0/30',
                },
            ],
        }));
        test.done();
    },
    'fails when ip is invalid'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.256',
                },
            },
        }), /`ip`.+IPv4/);
        test.done();
    },
    'fails when specifying more than two tunnel options'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: 'secretkey1234',
                        },
                        {
                            preSharedKey: 'secretkey1234',
                        },
                        {
                            preSharedKey: 'secretkey1234',
                        },
                    ],
                },
            },
        }), /two.+`tunnelOptions`/);
        test.done();
    },
    'fails with duplicate tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                        {
                            tunnelInsideCidr: '169.254.10.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+both tunnels/);
        test.done();
    },
    'fails when specifying an invalid pre-shared key'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            preSharedKey: '0invalid',
                        },
                    ],
                },
            },
        }), /`preSharedKey`/);
        test.done();
    },
    'fails when specifying a reserved tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.254.1.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+reserved/);
        test.done();
    },
    'fails when specifying an invalid tunnel inside cidr'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnConnections: {
                VpnConnection: {
                    ip: '192.0.2.1',
                    tunnelOptions: [
                        {
                            tunnelInsideCidr: '169.200.10.0/30',
                        },
                    ],
                },
            },
        }), /`tunnelInsideCidr`.+size/);
        test.done();
    },
    'can use metricTunnelState on a vpn connection'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnGateway: true,
        });
        const vpn = vpc.addVpnConnection('Vpn', {
            ip: '192.0.2.1',
        });
        // THEN
        test.deepEqual(stack.resolve(vpn.metricTunnelState()), {
            dimensions: { VpnId: { Ref: 'VpcNetworkVpnA476C58D' } },
            namespace: 'AWS/VPN',
            metricName: 'TunnelState',
            period: core_1.Duration.minutes(5),
            statistic: 'Average',
        });
        test.done();
    },
    'can use metricAllTunnelDataOut'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // THEN
        test.deepEqual(stack.resolve(lib_1.VpnConnection.metricAllTunnelDataOut()), {
            namespace: 'AWS/VPN',
            metricName: 'TunnelDataOut',
            period: core_1.Duration.minutes(5),
            statistic: 'Sum',
        });
        test.done();
    },
    'fails when enabling vpnGateway without having subnets'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        test.throws(() => new lib_1.Vpc(stack, 'VpcNetwork', {
            vpnGateway: true,
            subnetConfiguration: [],
        }), /VPN gateway/);
        test.done();
    },
    'can add a vpn connection later to a vpc that initially had no subnets'(test) {
        // GIVEN
        const stack = new core_1.Stack();
        // WHEN
        const vpc = new lib_1.Vpc(stack, 'VpcNetwork', {
            subnetConfiguration: [],
        });
        const subnet = new lib_1.PublicSubnet(stack, 'Subnet', {
            vpcId: vpc.vpcId,
            availabilityZone: 'eu-central-1a',
            cidrBlock: '10.0.0.0/28',
        });
        vpc.publicSubnets.push(subnet);
        vpc.addVpnConnection('VPNConnection', {
            ip: '1.2.3.4',
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::EC2::CustomerGateway', {
            Type: 'ipsec.1',
        }));
        test.done();
    },
});
//# sourceMappingURL=data:application/json;base64,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