"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const nodeunit_shim_1 = require("nodeunit-shim");
const lib_1 = require("../lib");
nodeunit_shim_1.nodeunitShim({
    'Vpc.fromLookup()': {
        'requires concrete values'(test) {
            // GIVEN
            const stack = new core_1.Stack();
            test.throws(() => {
                lib_1.Vpc.fromLookup(stack, 'Vpc', {
                    vpcId: core_1.Lazy.stringValue({ produce: () => 'some-id' }),
                });
            }, 'All arguments to Vpc.fromLookup() must be concrete');
            test.done();
        },
        'selecting subnets by name from a looked-up VPC does not throw'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, undefined, { env: { region: 'us-east-1', account: '123456789012' } });
            const vpc = lib_1.Vpc.fromLookup(stack, 'VPC', {
                vpcId: 'vpc-1234',
            });
            // WHEN
            vpc.selectSubnets({ subnetName: 'Bleep' });
            // THEN: no exception
            test.done();
        },
        'accepts asymmetric subnets'(test) {
            const previous = mockVpcContextProviderWith(test, {
                vpcId: 'vpc-1234',
                subnetGroups: [
                    {
                        name: 'Public',
                        type: cxapi.VpcSubnetGroupType.PUBLIC,
                        subnets: [
                            {
                                subnetId: 'pub-sub-in-us-east-1a',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pub-sub-in-us-east-1b',
                                availabilityZone: 'us-east-1b',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                    {
                        name: 'Private',
                        type: cxapi.VpcSubnetGroupType.PRIVATE,
                        subnets: [
                            {
                                subnetId: 'pri-sub-1-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-1-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                ],
            }, options => {
                test.deepEqual(options.filter, {
                    isDefault: 'true',
                });
                test.equal(options.subnetGroupNameTag, undefined);
            });
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromLookup(stack, 'Vpc', {
                isDefault: true,
            });
            test.deepEqual(vpc.availabilityZones, ['us-east-1a', 'us-east-1b', 'us-east-1c', 'us-east-1d']);
            test.equal(vpc.publicSubnets.length, 2);
            test.equal(vpc.privateSubnets.length, 4);
            test.equal(vpc.isolatedSubnets.length, 0);
            restoreContextProvider(previous);
            test.done();
        },
        'selectSubnets onePerAz works on imported VPC'(test) {
            const previous = mockVpcContextProviderWith(test, {
                vpcId: 'vpc-1234',
                subnetGroups: [
                    {
                        name: 'Public',
                        type: cxapi.VpcSubnetGroupType.PUBLIC,
                        subnets: [
                            {
                                subnetId: 'pub-sub-in-us-east-1a',
                                availabilityZone: 'us-east-1a',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pub-sub-in-us-east-1b',
                                availabilityZone: 'us-east-1b',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                    {
                        name: 'Private',
                        type: cxapi.VpcSubnetGroupType.PRIVATE,
                        subnets: [
                            {
                                subnetId: 'pri-sub-1-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1c',
                                availabilityZone: 'us-east-1c',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-1-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                            {
                                subnetId: 'pri-sub-2-in-us-east-1d',
                                availabilityZone: 'us-east-1d',
                                routeTableId: 'rt-123',
                            },
                        ],
                    },
                ],
            }, options => {
                test.deepEqual(options.filter, {
                    isDefault: 'true',
                });
                test.equal(options.subnetGroupNameTag, undefined);
            });
            const stack = new core_1.Stack();
            const vpc = lib_1.Vpc.fromLookup(stack, 'Vpc', {
                isDefault: true,
            });
            // WHEN
            const subnets = vpc.selectSubnets({ subnetType: lib_1.SubnetType.PRIVATE, onePerAz: true });
            // THEN: we got 2 subnets and not 4
            test.deepEqual(subnets.subnets.map(s => s.availabilityZone), ['us-east-1c', 'us-east-1d']);
            restoreContextProvider(previous);
            test.done();
        },
        'AZ in dummy lookup VPC matches AZ in Stack'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, 'MyTestStack', { env: { account: '1234567890', region: 'dummy' } });
            const vpc = lib_1.Vpc.fromLookup(stack, 'vpc', { isDefault: true });
            // WHEN
            const subnets = vpc.selectSubnets({
                availabilityZones: stack.availabilityZones,
            });
            // THEN
            test.equals(subnets.subnets.length, 2);
            test.done();
        },
        'don\'t crash when using subnetgroup name in lookup VPC'(test) {
            // GIVEN
            const stack = new core_1.Stack(undefined, 'MyTestStack', { env: { account: '1234567890', region: 'dummy' } });
            const vpc = lib_1.Vpc.fromLookup(stack, 'vpc', { isDefault: true });
            // WHEN
            new lib_1.Instance(stack, 'Instance', {
                vpc,
                instanceType: new lib_1.InstanceType('t2.large'),
                machineImage: new lib_1.GenericLinuxImage({ dummy: 'ami-1234' }),
                vpcSubnets: {
                    subnetGroupName: 'application_layer',
                },
            });
            // THEN -- no exception occurred
            test.done();
        },
    },
});
function mockVpcContextProviderWith(test, response, paramValidator) {
    const previous = core_1.ContextProvider.getValue;
    core_1.ContextProvider.getValue = (_scope, options) => {
        // do some basic sanity checks
        test.equal(options.provider, cxschema.ContextProvider.VPC_PROVIDER, `Expected provider to be: '${cxschema.ContextProvider.VPC_PROVIDER}', got: '${options.provider}'`);
        test.equal((options.props || {}).returnAsymmetricSubnets, true, `Expected options.props.returnAsymmetricSubnets to be true, got: '${(options.props || {}).returnAsymmetricSubnets}'`);
        if (paramValidator) {
            paramValidator(options.props);
        }
        return {
            value: {
                availabilityZones: [],
                isolatedSubnetIds: undefined,
                isolatedSubnetNames: undefined,
                isolatedSubnetRouteTableIds: undefined,
                privateSubnetIds: undefined,
                privateSubnetNames: undefined,
                privateSubnetRouteTableIds: undefined,
                publicSubnetIds: undefined,
                publicSubnetNames: undefined,
                publicSubnetRouteTableIds: undefined,
                ...response,
            },
        };
    };
    return previous;
}
function restoreContextProvider(previous) {
    core_1.ContextProvider.getValue = previous;
}
//# sourceMappingURL=data:application/json;base64,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