"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const path = require("path");
const assert_1 = require("@aws-cdk/assert");
require("@aws-cdk/assert/jest");
const iam = require("@aws-cdk/aws-iam");
const s3 = require("@aws-cdk/aws-s3");
const s3_assets = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const ec2 = require("../lib");
let app;
let stack;
let instanceRole;
let resource;
let linuxUserData;
beforeEach(() => {
    app = new core_1.App();
    stack = new core_1.Stack(app, 'Stack', {
        env: { account: '1234', region: 'testregion' },
    });
    instanceRole = new iam.Role(stack, 'InstanceRole', {
        assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
    });
    resource = new core_1.CfnResource(stack, 'Resource', {
        type: 'CDK::Test::Resource',
    });
    linuxUserData = ec2.UserData.forLinux();
});
test('whole config with restart handles', () => {
    // WHEN
    const handle = new ec2.InitServiceRestartHandle();
    const config = new ec2.InitConfig([
        ec2.InitFile.fromString('/etc/my.cnf', '[mysql]\ngo_fast=true', { serviceRestartHandles: [handle] }),
        ec2.InitSource.fromUrl('/tmp/foo', 'https://amazon.com/foo.zip', { serviceRestartHandles: [handle] }),
        ec2.InitPackage.yum('httpd', { serviceRestartHandles: [handle] }),
        ec2.InitCommand.argvCommand(['/bin/true'], { serviceRestartHandles: [handle] }),
        ec2.InitService.enable('httpd', { serviceRestartHandle: handle }),
    ]);
    // THEN
    expect(config._bind(stack, linuxOptions()).config).toEqual(expect.objectContaining({
        services: {
            sysvinit: {
                httpd: {
                    enabled: true,
                    ensureRunning: true,
                    commands: ['000'],
                    files: ['/etc/my.cnf'],
                    packages: {
                        yum: ['httpd'],
                    },
                    sources: ['/tmp/foo'],
                },
            },
        },
    }));
});
test('CloudFormationInit can be added to after instantiation', () => {
    // GIVEN
    const config = new ec2.InitConfig([]);
    const init = ec2.CloudFormationInit.fromConfig(config);
    // WHEN
    config.add(ec2.InitCommand.argvCommand(['/bin/true']));
    init._attach(resource, linuxOptions());
    // THEN
    expectMetadataLike({
        'AWS::CloudFormation::Init': {
            config: {
                commands: {
                    '000': { command: ['/bin/true'] },
                },
            },
        },
    });
});
test('empty configs are not rendered', () => {
    // GIVEN
    const config1 = new ec2.InitConfig([]);
    const config2 = new ec2.InitConfig([
        ec2.InitCommand.argvCommand(['/bin/true']),
    ]);
    // WHEN
    const init = ec2.CloudFormationInit.fromConfigSets({
        configSets: { default: ['config2', 'config1'] },
        configs: { config1, config2 },
    });
    init._attach(resource, linuxOptions());
    // THEN
    expectMetadataLike({
        'AWS::CloudFormation::Init': {
            configSets: {
                default: ['config2'],
            },
            config2: {
                commands: {
                    '000': { command: ['/bin/true'] },
                },
            },
        },
    });
});
describe('userdata', () => {
    let simpleInit;
    beforeEach(() => {
        simpleInit = ec2.CloudFormationInit.fromElements(ec2.InitCommand.argvCommand(['/bin/true']));
    });
    test('linux userdata contains right commands', () => {
        // WHEN
        simpleInit._attach(resource, linuxOptions());
        // THEN
        const lines = linuxUserData.render().split('\n');
        expectLine(lines, cmdArg('cfn-init', `--region ${core_1.Aws.REGION}`));
        expectLine(lines, cmdArg('cfn-init', `--stack ${core_1.Aws.STACK_NAME}`));
        expectLine(lines, cmdArg('cfn-init', `--resource ${resource.logicalId}`));
        expectLine(lines, cmdArg('cfn-init', '-c default'));
        expectLine(lines, cmdArg('cfn-signal', `--region ${core_1.Aws.REGION}`));
        expectLine(lines, cmdArg('cfn-signal', `--stack ${core_1.Aws.STACK_NAME}`));
        expectLine(lines, cmdArg('cfn-signal', `--resource ${resource.logicalId}`));
        expectLine(lines, cmdArg('cfn-signal', '-e $?'));
        expectLine(lines, cmdArg('cat', 'cfn-init.log'));
        expectLine(lines, /fingerprint/);
    });
    test('Windows userdata contains right commands', () => {
        // WHEN
        const windowsUserData = ec2.UserData.forWindows();
        simpleInit._attach(resource, {
            platform: ec2.OperatingSystemType.WINDOWS,
            instanceRole,
            userData: windowsUserData,
        });
        // THEN
        const lines = windowsUserData.render().split('\n');
        expectLine(lines, cmdArg('cfn-init', `--region ${core_1.Aws.REGION}`));
        expectLine(lines, cmdArg('cfn-init', `--stack ${core_1.Aws.STACK_NAME}`));
        expectLine(lines, cmdArg('cfn-init', `--resource ${resource.logicalId}`));
        expectLine(lines, cmdArg('cfn-init', '-c default'));
        expectLine(lines, cmdArg('cfn-signal', `--region ${core_1.Aws.REGION}`));
        expectLine(lines, cmdArg('cfn-signal', `--stack ${core_1.Aws.STACK_NAME}`));
        expectLine(lines, cmdArg('cfn-signal', `--resource ${resource.logicalId}`));
        expectLine(lines, cmdArg('cfn-signal', '-e $LASTEXITCODE'));
        expectLine(lines, cmdArg('type', 'cfn-init.log'));
        expectLine(lines, /fingerprint/);
    });
    test('ignoreFailures disables result code reporting', () => {
        // WHEN
        simpleInit._attach(resource, {
            ...linuxOptions(),
            ignoreFailures: true,
        });
        // THEN
        const lines = linuxUserData.render().split('\n');
        dontExpectLine(lines, cmdArg('cfn-signal', '-e $?'));
        expectLine(lines, cmdArg('cfn-signal', '-e 0'));
    });
    test('can disable log printing', () => {
        // WHEN
        simpleInit._attach(resource, {
            ...linuxOptions(),
            printLog: false,
        });
        // THEN
        const lines = linuxUserData.render().split('\n');
        dontExpectLine(lines, cmdArg('cat', 'cfn-init.log'));
    });
    test('can disable fingerprinting', () => {
        // WHEN
        simpleInit._attach(resource, {
            ...linuxOptions(),
            embedFingerprint: false,
        });
        // THEN
        const lines = linuxUserData.render().split('\n');
        dontExpectLine(lines, /fingerprint/);
    });
    test('can request multiple different configsets to be used', () => {
        // WHEN
        simpleInit._attach(resource, {
            ...linuxOptions(),
            configSets: ['banana', 'peach'],
        });
        // THEN
        const lines = linuxUserData.render().split('\n');
        expectLine(lines, cmdArg('cfn-init', '-c banana,peach'));
    });
});
const ASSET_STATEMENT = {
    Action: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
    Effect: 'Allow',
    Resource: [
        {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':s3:::',
                    { Ref: assert_1.stringLike('AssetParameter*S3Bucket*') },
                ]],
        },
        {
            'Fn::Join': ['', [
                    'arn:',
                    { Ref: 'AWS::Partition' },
                    ':s3:::',
                    { Ref: assert_1.stringLike('AssetParameter*S3Bucket*') },
                    '/*',
                ]],
        },
    ],
};
describe('assets n buckets', () => {
    test.each([
        ['Existing'],
        [''],
    ])('InitFile.from%sAsset', (existing) => {
        // GIVEN
        const asset = new s3_assets.Asset(stack, 'Asset', { path: __filename });
        const init = ec2.CloudFormationInit.fromElements(existing
            ? ec2.InitFile.fromExistingAsset('/etc/fun.js', asset)
            : ec2.InitFile.fromAsset('/etc/fun.js', __filename));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith(ASSET_STATEMENT),
                Version: '2012-10-17',
            },
        });
        expectMetadataLike({
            'AWS::CloudFormation::Init': {
                config: {
                    files: {
                        '/etc/fun.js': {
                            source: {
                                'Fn::Join': ['', [
                                        'https://s3.testregion.',
                                        { Ref: 'AWS::URLSuffix' },
                                        '/',
                                        { Ref: assert_1.stringLike('AssetParameters*') },
                                        '/',
                                        { 'Fn::Select': [0, { 'Fn::Split': ['||', { Ref: assert_1.stringLike('AssetParameters*') }] }] },
                                        { 'Fn::Select': [1, { 'Fn::Split': ['||', { Ref: assert_1.stringLike('AssetParameters*') }] }] },
                                    ]],
                            },
                        },
                    },
                },
            },
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: [
                        { Ref: assert_1.stringLike('AssetParameters*S3Bucket*') },
                    ],
                },
            },
        });
    });
    test.each([
        ['Existing'],
        [''],
    ])('InitSource.from%sAsset', (existing) => {
        // GIVEN
        const asset = new s3_assets.Asset(stack, 'Asset', { path: path.join(__dirname, 'asset-fixture') });
        const init = ec2.CloudFormationInit.fromElements(existing
            ? ec2.InitSource.fromExistingAsset('/etc/fun', asset)
            : ec2.InitSource.fromAsset('/etc/fun', path.join(__dirname, 'asset-fixture')));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith(ASSET_STATEMENT),
                Version: '2012-10-17',
            },
        });
        expectMetadataLike({
            'AWS::CloudFormation::Init': {
                config: {
                    sources: {
                        '/etc/fun': {
                            'Fn::Join': ['', [
                                    'https://s3.testregion.',
                                    { Ref: 'AWS::URLSuffix' },
                                    '/',
                                    { Ref: assert_1.stringLike('AssetParameters*') },
                                    '/',
                                    { 'Fn::Select': [0, { 'Fn::Split': ['||', { Ref: assert_1.stringLike('AssetParameters*') }] }] },
                                    { 'Fn::Select': [1, { 'Fn::Split': ['||', { Ref: assert_1.stringLike('AssetParameters*') }] }] },
                                ]],
                        },
                    },
                },
            },
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: [
                        { Ref: assert_1.stringLike('AssetParameters*S3Bucket*') },
                    ],
                },
            },
        });
    });
    test('InitFile.fromS3Object', () => {
        const bucket = s3.Bucket.fromBucketName(stack, 'Bucket', 'my-bucket');
        const init = ec2.CloudFormationInit.fromElements(ec2.InitFile.fromS3Object('/etc/fun.js', bucket, 'file.js'));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith({
                    Action: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket']] },
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket/file.js']] },
                    ],
                }),
                Version: '2012-10-17',
            },
        });
        expectMetadataLike({
            'AWS::CloudFormation::Init': {
                config: {
                    files: {
                        '/etc/fun.js': {
                            source: { 'Fn::Join': ['', ['https://s3.testregion.', { Ref: 'AWS::URLSuffix' }, '/my-bucket/file.js']] },
                        },
                    },
                },
            },
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: ['my-bucket'],
                },
            },
        });
    });
    test('InitSource.fromS3Object', () => {
        const bucket = s3.Bucket.fromBucketName(stack, 'Bucket', 'my-bucket');
        const init = ec2.CloudFormationInit.fromElements(ec2.InitSource.fromS3Object('/etc/fun', bucket, 'file.zip'));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expect(stack).toHaveResource('AWS::IAM::Policy', {
            PolicyDocument: {
                Statement: assert_1.arrayWith({
                    Action: ['s3:GetObject*', 's3:GetBucket*', 's3:List*'],
                    Effect: 'Allow',
                    Resource: [
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket']] },
                        { 'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':s3:::my-bucket/file.zip']] },
                    ],
                }),
                Version: '2012-10-17',
            },
        });
        expectMetadataLike({
            'AWS::CloudFormation::Init': {
                config: {
                    sources: {
                        '/etc/fun': { 'Fn::Join': ['', ['https://s3.testregion.', { Ref: 'AWS::URLSuffix' }, '/my-bucket/file.zip']] },
                    },
                },
            },
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: ['my-bucket'],
                },
            },
        });
    });
    test('no duplication of bucket names when using multiple assets', () => {
        // GIVEN
        const init = ec2.CloudFormationInit.fromElements(ec2.InitFile.fromAsset('/etc/fun.js', __filename), ec2.InitSource.fromAsset('/etc/fun', path.join(__dirname, 'asset-fixture')));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expectMetadataLike({
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: [
                        { Ref: assert_1.stringLike('AssetParameters*S3Bucket*') },
                    ],
                },
            },
        });
    });
    test('multiple buckets appear in the same auth block', () => {
        // GIVEN
        const bucket = s3.Bucket.fromBucketName(stack, 'Bucket', 'my-bucket');
        const init = ec2.CloudFormationInit.fromElements(ec2.InitFile.fromS3Object('/etc/fun.js', bucket, 'file.js'), ec2.InitSource.fromAsset('/etc/fun', path.join(__dirname, 'asset-fixture')));
        // WHEN
        init._attach(resource, linuxOptions());
        // THEN
        expectMetadataLike({
            'AWS::CloudFormation::Authentication': {
                S3AccessCreds: {
                    type: 'S3',
                    roleName: { Ref: 'InstanceRole3CCE2F1D' },
                    buckets: assert_1.arrayWith({ Ref: assert_1.stringLike('AssetParameters*S3Bucket*') }, 'my-bucket'),
                },
            },
        });
    });
});
function linuxOptions() {
    return {
        platform: ec2.OperatingSystemType.LINUX,
        instanceRole,
        userData: linuxUserData,
    };
}
function expectMetadataLike(pattern) {
    expect(stack).toHaveResourceLike('CDK::Test::Resource', {
        Metadata: pattern,
    }, assert_1.ResourcePart.CompleteDefinition);
}
function expectLine(lines, re) {
    for (const line of lines) {
        if (re.test(line)) {
            return;
        }
    }
    throw new Error(`None of the lines matched '${re}': ${lines.join('\n')}`);
}
function dontExpectLine(lines, re) {
    try {
        expectLine(lines, re);
    }
    catch (e) {
        return;
    }
    throw new Error(`Found unexpected line matching '${re}': ${lines.join('\n')}`);
}
function cmdArg(command, argument) {
    return new RegExp(`${escapeRegex(command)}(\.exe)? .*${escapeRegex(argument)}`);
}
function escapeRegex(s) {
    return s.replace(/[.*+?^${}()|[\]\\]/g, '\\$&'); // $& means the whole matched string
}
//# sourceMappingURL=data:application/json;base64,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