"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PrivateSubnet = exports.PublicSubnet = exports.RouterType = exports.Subnet = exports.Vpc = exports.DefaultInstanceTenancy = exports.SubnetType = void 0;
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const core_1 = require("@aws-cdk/core");
const cxapi = require("@aws-cdk/cx-api");
const ec2_generated_1 = require("./ec2.generated");
const nat_1 = require("./nat");
const network_acl_1 = require("./network-acl");
const network_util_1 = require("./network-util");
const util_1 = require("./util");
const vpc_endpoint_1 = require("./vpc-endpoint");
const vpc_flow_logs_1 = require("./vpc-flow-logs");
const vpn_1 = require("./vpn");
const VPC_SUBNET_SYMBOL = Symbol.for('@aws-cdk/aws-ec2.VpcSubnet');
/**
 * The type of Subnet
 */
var SubnetType;
(function (SubnetType) {
    /**
     * Isolated Subnets do not route traffic to the Internet (in this VPC).
     *
     * This can be good for subnets with RDS or Elasticache instances,
     * or which route Internet traffic through a peer VPC.
     */
    SubnetType["ISOLATED"] = "Isolated";
    /**
     * Subnet that routes to the internet, but not vice versa.
     *
     * Instances in a private subnet can connect to the Internet, but will not
     * allow connections to be initiated from the Internet. Internet traffic will
     * be routed via a NAT Gateway.
     *
     * Normally a Private subnet will use a NAT gateway in the same AZ, but
     * if `natGateways` is used to reduce the number of NAT gateways, a NAT
     * gateway from another AZ will be used instead.
     */
    SubnetType["PRIVATE"] = "Private";
    /**
     * Subnet connected to the Internet
     *
     * Instances in a Public subnet can connect to the Internet and can be
     * connected to from the Internet as long as they are launched with public
     * IPs (controlled on the AutoScalingGroup or other constructs that launch
     * instances).
     *
     * Public subnets route outbound traffic via an Internet Gateway.
     */
    SubnetType["PUBLIC"] = "Public";
})(SubnetType = exports.SubnetType || (exports.SubnetType = {}));
/**
 * A new or imported VPC
 */
class VpcBase extends core_1.Resource {
    constructor() {
        super(...arguments);
        /**
         * Dependencies for NAT connectivity
         *
         * @deprecated - This value is no longer used.
         */
        this.natDependencies = new Array();
        /**
         * If this is set to true, don't error out on trying to select subnets
         */
        this.incompleteSubnetDefinition = false;
    }
    /**
     * Returns IDs of selected subnets
     */
    selectSubnets(selection = {}) {
        const subnets = this.selectSubnetObjects(selection);
        const pubs = new Set(this.publicSubnets);
        return {
            subnetIds: subnets.map(s => s.subnetId),
            get availabilityZones() { return subnets.map(s => s.availabilityZone); },
            internetConnectivityEstablished: tap(new CompositeDependable(), d => subnets.forEach(s => d.add(s.internetConnectivityEstablished))),
            subnets,
            hasPublic: subnets.some(s => pubs.has(s)),
        };
    }
    /**
     * Adds a VPN Gateway to this VPC
     */
    enableVpnGateway(options) {
        var _a;
        if (this.vpnGatewayId) {
            throw new Error('The VPN Gateway has already been enabled.');
        }
        const vpnGateway = new vpn_1.VpnGateway(this, 'VpnGateway', {
            amazonSideAsn: options.amazonSideAsn,
            type: vpn_1.VpnConnectionType.IPSEC_1,
        });
        this._vpnGatewayId = vpnGateway.gatewayId;
        const attachment = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCVPNGW', {
            vpcId: this.vpcId,
            vpnGatewayId: this._vpnGatewayId,
        });
        // Propagate routes on route tables associated with the right subnets
        const vpnRoutePropagation = (_a = options.vpnRoutePropagation) !== null && _a !== void 0 ? _a : [{}];
        const routeTableIds = util_1.allRouteTableIds(util_1.flatten(vpnRoutePropagation.map(s => this.selectSubnets(s).subnets)));
        if (routeTableIds.length === 0) {
            this.node.addError(`enableVpnGateway: no subnets matching selection: '${JSON.stringify(vpnRoutePropagation)}'. Select other subnets to add routes to.`);
        }
        const routePropagation = new ec2_generated_1.CfnVPNGatewayRoutePropagation(this, 'RoutePropagation', {
            routeTableIds,
            vpnGatewayId: this._vpnGatewayId,
        });
        // The AWS::EC2::VPNGatewayRoutePropagation resource cannot use the VPN gateway
        // until it has successfully attached to the VPC.
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-ec2-vpn-gatewayrouteprop.html
        routePropagation.node.addDependency(attachment);
    }
    /**
     * Adds a new VPN connection to this VPC
     */
    addVpnConnection(id, options) {
        return new vpn_1.VpnConnection(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new interface endpoint to this VPC
     */
    addInterfaceEndpoint(id, options) {
        return new vpc_endpoint_1.InterfaceVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new gateway endpoint to this VPC
     */
    addGatewayEndpoint(id, options) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            vpc: this,
            ...options,
        });
    }
    /**
     * Adds a new flow log to this VPC
     */
    addFlowLog(id, options) {
        return new vpc_flow_logs_1.FlowLog(this, id, {
            resourceType: vpc_flow_logs_1.FlowLogResourceType.fromVpc(this),
            ...options,
        });
    }
    /**
     * Returns the id of the VPN Gateway (if enabled)
     */
    get vpnGatewayId() {
        return this._vpnGatewayId;
    }
    /**
     * Return the subnets appropriate for the placement strategy
     */
    selectSubnetObjects(selection = {}) {
        selection = this.reifySelectionDefaults(selection);
        if (selection.subnets !== undefined) {
            return selection.subnets;
        }
        let subnets;
        if (selection.subnetGroupName !== undefined) { // Select by name
            subnets = this.selectSubnetObjectsByName(selection.subnetGroupName);
        }
        else { // Or specify by type
            const type = selection.subnetType || SubnetType.PRIVATE;
            subnets = this.selectSubnetObjectsByType(type);
        }
        if (selection.availabilityZones !== undefined) { // Filter by AZs, if specified
            subnets = retainByAZ(subnets, selection.availabilityZones);
        }
        if (!!selection.onePerAz && subnets.length > 0) { // Ensure one per AZ if specified
            subnets = retainOnePerAz(subnets);
        }
        return subnets;
    }
    selectSubnetObjectsByName(groupName) {
        const allSubnets = [...this.publicSubnets, ...this.privateSubnets, ...this.isolatedSubnets];
        const subnets = allSubnets.filter(s => util_1.subnetGroupNameFromConstructId(s) === groupName);
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const names = Array.from(new Set(allSubnets.map(util_1.subnetGroupNameFromConstructId)));
            throw new Error(`There are no subnet groups with name '${groupName}' in this VPC. Available names: ${names}`);
        }
        return subnets;
    }
    selectSubnetObjectsByType(subnetType) {
        const allSubnets = {
            [SubnetType.ISOLATED]: this.isolatedSubnets,
            [SubnetType.PRIVATE]: this.privateSubnets,
            [SubnetType.PUBLIC]: this.publicSubnets,
        };
        const subnets = allSubnets[subnetType];
        // Force merge conflict here with https://github.com/aws/aws-cdk/pull/4089
        // see ImportedVpc
        if (subnets.length === 0 && !this.incompleteSubnetDefinition) {
            const availableTypes = Object.entries(allSubnets).filter(([_, subs]) => subs.length > 0).map(([typeName, _]) => typeName);
            throw new Error(`There are no '${subnetType}' subnet groups in this VPC. Available types: ${availableTypes}`);
        }
        return subnets;
    }
    /**
     * Validate the fields in a SubnetSelection object, and reify defaults if necessary
     *
     * In case of default selection, select the first type of PRIVATE, ISOLATED,
     * PUBLIC (in that order) that has any subnets.
     */
    reifySelectionDefaults(placement) {
        if (placement.subnetName !== undefined) {
            if (placement.subnetGroupName !== undefined) {
                throw new Error('Please use only \'subnetGroupName\' (\'subnetName\' is deprecated and has the same behavior)');
            }
            placement = { ...placement, subnetGroupName: placement.subnetName };
        }
        const exclusiveSelections = ['subnets', 'subnetType', 'subnetGroupName'];
        const providedSelections = exclusiveSelections.filter(key => placement[key] !== undefined);
        if (providedSelections.length > 1) {
            throw new Error(`Only one of '${providedSelections}' can be supplied to subnet selection.`);
        }
        if (placement.subnetType === undefined && placement.subnetGroupName === undefined && placement.subnets === undefined) {
            // Return default subnet type based on subnets that actually exist
            if (this.privateSubnets.length > 0) {
                return {
                    subnetType: SubnetType.PRIVATE,
                    onePerAz: placement.onePerAz,
                    availabilityZones: placement.availabilityZones,
                };
            }
            if (this.isolatedSubnets.length > 0) {
                return {
                    subnetType: SubnetType.ISOLATED,
                    onePerAz: placement.onePerAz,
                    availabilityZones: placement.availabilityZones,
                };
            }
            return {
                subnetType: SubnetType.PUBLIC,
                onePerAz: placement.onePerAz,
                availabilityZones: placement.availabilityZones,
            };
        }
        return placement;
    }
}
function retainByAZ(subnets, azs) {
    return subnets.filter(s => azs.includes(s.availabilityZone));
}
function retainOnePerAz(subnets) {
    const azsSeen = new Set();
    return subnets.filter(subnet => {
        if (azsSeen.has(subnet.availabilityZone)) {
            return false;
        }
        azsSeen.add(subnet.availabilityZone);
        return true;
    });
}
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * The default tenancy of instances launched into the VPC.
 */
var DefaultInstanceTenancy;
(function (DefaultInstanceTenancy) {
    /**
     * Instances can be launched with any tenancy.
     */
    DefaultInstanceTenancy["DEFAULT"] = "default";
    /**
     * Any instance launched into the VPC automatically has dedicated tenancy, unless you launch it with the default tenancy.
     */
    DefaultInstanceTenancy["DEDICATED"] = "dedicated";
})(DefaultInstanceTenancy = exports.DefaultInstanceTenancy || (exports.DefaultInstanceTenancy = {}));
/**
 * Define an AWS Virtual Private Cloud
 *
 * See the package-level documentation of this package for an overview
 * of the various dimensions in which you can configure your VPC.
 *
 * For example:
 *
 * ```ts
 * const vpc = new ec2.Vpc(this, 'TheVPC', {
 *   cidr: "10.0.0.0/16"
 * })
 *
 * // Iterate the private subnets
 * const selection = vpc.selectSubnets({
 *   subnetType: ec2.SubnetType.PRIVATE
 * });
 *
 * for (const subnet of selection.subnets) {
 *   // ...
 * }
 * ```
 *
 * @resource AWS::EC2::VPC
 */
class Vpc extends VpcBase {
    /**
     * Vpc creates a VPC that spans a whole region.
     * It will automatically divide the provided VPC CIDR range, and create public and private subnets per Availability Zone.
     * Network routing for the public subnets will be configured to allow outbound access directly via an Internet Gateway.
     * Network routing for the private subnets will be configured to allow outbound access via a set of resilient NAT Gateways (one per AZ).
     */
    constructor(scope, id, props = {}) {
        super(scope, id);
        /**
         * List of public subnets in this VPC
         */
        this.publicSubnets = [];
        /**
         * List of private subnets in this VPC
         */
        this.privateSubnets = [];
        /**
         * List of isolated subnets in this VPC
         */
        this.isolatedSubnets = [];
        /**
         * Subnet configurations for this VPC
         */
        this.subnetConfiguration = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        const stack = core_1.Stack.of(this);
        // Can't have enabledDnsHostnames without enableDnsSupport
        if (props.enableDnsHostnames && !props.enableDnsSupport) {
            throw new Error('To use DNS Hostnames, DNS Support must be enabled, however, it was explicitly disabled.');
        }
        const cidrBlock = ifUndefined(props.cidr, Vpc.DEFAULT_CIDR_RANGE);
        if (core_1.Token.isUnresolved(cidrBlock)) {
            throw new Error('\'cidr\' property must be a concrete CIDR string, got a Token (we need to parse it for automatic subdivision)');
        }
        this.networkBuilder = new network_util_1.NetworkBuilder(cidrBlock);
        this.dnsHostnamesEnabled = props.enableDnsHostnames == null ? true : props.enableDnsHostnames;
        this.dnsSupportEnabled = props.enableDnsSupport == null ? true : props.enableDnsSupport;
        const instanceTenancy = props.defaultInstanceTenancy || 'default';
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
        // Define a VPC using the provided CIDR range
        this.resource = new ec2_generated_1.CfnVPC(this, 'Resource', {
            cidrBlock,
            enableDnsHostnames: this.dnsHostnamesEnabled,
            enableDnsSupport: this.dnsSupportEnabled,
            instanceTenancy,
        });
        this.vpcDefaultNetworkAcl = this.resource.attrDefaultNetworkAcl;
        this.vpcCidrBlockAssociations = this.resource.attrCidrBlockAssociations;
        this.vpcCidrBlock = this.resource.attrCidrBlock;
        this.vpcDefaultSecurityGroup = this.resource.attrDefaultSecurityGroup;
        this.vpcIpv6CidrBlocks = this.resource.attrIpv6CidrBlocks;
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.availabilityZones = stack.availabilityZones;
        const maxAZs = props.maxAzs !== undefined ? props.maxAzs : 3;
        this.availabilityZones = this.availabilityZones.slice(0, maxAZs);
        this.vpcId = this.resource.ref;
        const defaultSubnet = props.natGateways === 0 ? Vpc.DEFAULT_SUBNETS_NO_NAT : Vpc.DEFAULT_SUBNETS;
        this.subnetConfiguration = ifUndefined(props.subnetConfiguration, defaultSubnet);
        const natGatewayPlacement = props.natGatewaySubnets || { subnetType: SubnetType.PUBLIC };
        const natGatewayCount = determineNatGatewayCount(props.natGateways, this.subnetConfiguration, this.availabilityZones.length);
        // subnetConfiguration must be set before calling createSubnets
        this.createSubnets();
        const allowOutbound = this.subnetConfiguration.filter(subnet => (subnet.subnetType !== SubnetType.ISOLATED)).length > 0;
        // Create an Internet Gateway and attach it if necessary
        if (allowOutbound) {
            const igw = new ec2_generated_1.CfnInternetGateway(this, 'IGW', {});
            this.internetGatewayId = igw.ref;
            this._internetConnectivityEstablished.add(igw);
            const att = new ec2_generated_1.CfnVPCGatewayAttachment(this, 'VPCGW', {
                internetGatewayId: igw.ref,
                vpcId: this.resource.ref,
            });
            this.publicSubnets.forEach(publicSubnet => {
                publicSubnet.addDefaultInternetRoute(igw.ref, att);
            });
            // if gateways are needed create them
            if (natGatewayCount > 0) {
                const provider = props.natGatewayProvider || nat_1.NatProvider.gateway();
                this.createNatGateways(provider, natGatewayCount, natGatewayPlacement);
            }
        }
        if (props.vpnGateway && this.publicSubnets.length === 0 && this.privateSubnets.length === 0 && this.isolatedSubnets.length === 0) {
            throw new Error('Can not enable the VPN gateway while the VPC has no subnets at all');
        }
        if ((props.vpnConnections || props.vpnGatewayAsn) && props.vpnGateway === false) {
            throw new Error('Cannot specify `vpnConnections` or `vpnGatewayAsn` when `vpnGateway` is set to false.');
        }
        if (props.vpnGateway || props.vpnConnections || props.vpnGatewayAsn) {
            this.enableVpnGateway({
                amazonSideAsn: props.vpnGatewayAsn,
                type: vpn_1.VpnConnectionType.IPSEC_1,
                vpnRoutePropagation: props.vpnRoutePropagation,
            });
            const vpnConnections = props.vpnConnections || {};
            for (const [connectionId, connection] of Object.entries(vpnConnections)) {
                this.addVpnConnection(connectionId, connection);
            }
        }
        // Allow creation of gateway endpoints on VPC instantiation as those can be
        // immediately functional without further configuration. This is not the case
        // for interface endpoints where the security group must be configured.
        if (props.gatewayEndpoints) {
            const gatewayEndpoints = props.gatewayEndpoints || {};
            for (const [endpointId, endpoint] of Object.entries(gatewayEndpoints)) {
                this.addGatewayEndpoint(endpointId, endpoint);
            }
        }
        // Add flow logs to the VPC
        if (props.flowLogs) {
            const flowLogs = props.flowLogs || {};
            for (const [flowLogId, flowLog] of Object.entries(flowLogs)) {
                this.addFlowLog(flowLogId, flowLog);
            }
        }
    }
    /**
     * Import an exported VPC
     */
    static fromVpcAttributes(scope, id, attrs) {
        return new ImportedVpc(scope, id, attrs, false);
    }
    /**
     * Import an existing VPC from by querying the AWS environment this stack is deployed to.
     *
     * This function only needs to be used to use VPCs not defined in your CDK
     * application. If you are looking to share a VPC between stacks, you can
     * pass the `Vpc` object between stacks and use it as normal.
     *
     * Calling this method will lead to a lookup when the CDK CLI is executed.
     * You can therefore not use any values that will only be available at
     * CloudFormation execution time (i.e., Tokens).
     *
     * The VPC information will be cached in `cdk.context.json` and the same VPC
     * will be used on future runs. To refresh the lookup, you will have to
     * evict the value from the cache using the `cdk context` command. See
     * https://docs.aws.amazon.com/cdk/latest/guide/context.html for more information.
     */
    static fromLookup(scope, id, options) {
        if (core_1.Token.isUnresolved(options.vpcId)
            || core_1.Token.isUnresolved(options.vpcName)
            || Object.values(options.tags || {}).some(core_1.Token.isUnresolved)
            || Object.keys(options.tags || {}).some(core_1.Token.isUnresolved)) {
            throw new Error('All arguments to Vpc.fromLookup() must be concrete (no Tokens)');
        }
        const filter = makeTagFilter(options.tags);
        // We give special treatment to some tags
        if (options.vpcId) {
            filter['vpc-id'] = options.vpcId;
        }
        if (options.vpcName) {
            filter['tag:Name'] = options.vpcName;
        }
        if (options.isDefault !== undefined) {
            filter.isDefault = options.isDefault ? 'true' : 'false';
        }
        const attributes = core_1.ContextProvider.getValue(scope, {
            provider: cxschema.ContextProvider.VPC_PROVIDER,
            props: {
                filter,
                returnAsymmetricSubnets: true,
                subnetGroupNameTag: options.subnetGroupNameTag,
            },
            dummyValue: undefined,
        }).value;
        return new LookedUpVpc(scope, id, attributes || DUMMY_VPC_PROPS, attributes === undefined);
        /**
         * Prefixes all keys in the argument with `tag:`.`
         */
        function makeTagFilter(tags) {
            const result = {};
            for (const [name, value] of Object.entries(tags || {})) {
                result[`tag:${name}`] = value;
            }
            return result;
        }
    }
    /**
     * Adds a new S3 gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addS3Endpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.S3,
            vpc: this,
            subnets,
        });
    }
    /**
     * Adds a new DynamoDB gateway endpoint to this VPC
     *
     * @deprecated use `addGatewayEndpoint()` instead
     */
    addDynamoDbEndpoint(id, subnets) {
        return new vpc_endpoint_1.GatewayVpcEndpoint(this, id, {
            service: vpc_endpoint_1.GatewayVpcEndpointAwsService.DYNAMODB,
            vpc: this,
            subnets,
        });
    }
    createNatGateways(provider, natCount, placement) {
        const natSubnets = this.selectSubnetObjects(placement);
        for (const sub of natSubnets) {
            if (this.publicSubnets.indexOf(sub) === -1) {
                throw new Error(`natGatewayPlacement ${placement} contains non public subnet ${sub}`);
            }
        }
        provider.configureNat({
            vpc: this,
            natSubnets: natSubnets.slice(0, natCount),
            privateSubnets: this.privateSubnets,
        });
    }
    /**
     * createSubnets creates the subnets specified by the subnet configuration
     * array or creates the `DEFAULT_SUBNETS` configuration
     */
    createSubnets() {
        const remainingSpaceSubnets = [];
        for (const subnet of this.subnetConfiguration) {
            if (subnet.cidrMask === undefined) {
                remainingSpaceSubnets.push(subnet);
                continue;
            }
            this.createSubnetResources(subnet, subnet.cidrMask);
        }
        const totalRemaining = remainingSpaceSubnets.length * this.availabilityZones.length;
        const cidrMaskForRemaing = this.networkBuilder.maskForRemainingSubnets(totalRemaining);
        for (const subnet of remainingSpaceSubnets) {
            this.createSubnetResources(subnet, cidrMaskForRemaing);
        }
    }
    createSubnetResources(subnetConfig, cidrMask) {
        this.availabilityZones.forEach((zone, index) => {
            if (subnetConfig.reserved === true) {
                // For reserved subnets, just allocate ip space but do not create any resources
                this.networkBuilder.addSubnet(cidrMask);
                return;
            }
            const name = util_1.subnetId(subnetConfig.name, index);
            const subnetProps = {
                availabilityZone: zone,
                vpcId: this.vpcId,
                cidrBlock: this.networkBuilder.addSubnet(cidrMask),
                mapPublicIpOnLaunch: (subnetConfig.subnetType === SubnetType.PUBLIC),
            };
            let subnet;
            switch (subnetConfig.subnetType) {
                case SubnetType.PUBLIC:
                    const publicSubnet = new PublicSubnet(this, name, subnetProps);
                    this.publicSubnets.push(publicSubnet);
                    subnet = publicSubnet;
                    break;
                case SubnetType.PRIVATE:
                    const privateSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.privateSubnets.push(privateSubnet);
                    subnet = privateSubnet;
                    break;
                case SubnetType.ISOLATED:
                    const isolatedSubnet = new PrivateSubnet(this, name, subnetProps);
                    this.isolatedSubnets.push(isolatedSubnet);
                    subnet = isolatedSubnet;
                    break;
                default:
                    throw new Error(`Unrecognized subnet type: ${subnetConfig.subnetType}`);
            }
            // These values will be used to recover the config upon provider import
            const includeResourceTypes = [ec2_generated_1.CfnSubnet.CFN_RESOURCE_TYPE_NAME];
            subnet.node.applyAspect(new core_1.Tag(SUBNETNAME_TAG, subnetConfig.name, { includeResourceTypes }));
            subnet.node.applyAspect(new core_1.Tag(SUBNETTYPE_TAG, subnetTypeTagValue(subnetConfig.subnetType), { includeResourceTypes }));
        });
    }
}
exports.Vpc = Vpc;
/**
 * The default CIDR range used when creating VPCs.
 * This can be overridden using VpcProps when creating a VPCNetwork resource.
 * e.g. new VpcResource(this, { cidr: '192.168.0.0./16' })
 */
Vpc.DEFAULT_CIDR_RANGE = '10.0.0.0/16';
/**
 * The default subnet configuration
 *
 * 1 Public and 1 Private subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.PRIVATE,
        name: util_1.defaultSubnetName(SubnetType.PRIVATE),
    },
];
/**
 * The default subnet configuration if natGateways specified to be 0
 *
 * 1 Public and 1 Isolated Subnet per AZ evenly split
 */
Vpc.DEFAULT_SUBNETS_NO_NAT = [
    {
        subnetType: SubnetType.PUBLIC,
        name: util_1.defaultSubnetName(SubnetType.PUBLIC),
    },
    {
        subnetType: SubnetType.ISOLATED,
        name: util_1.defaultSubnetName(SubnetType.ISOLATED),
    },
];
const SUBNETTYPE_TAG = 'aws-cdk:subnet-type';
const SUBNETNAME_TAG = 'aws-cdk:subnet-name';
function subnetTypeTagValue(type) {
    switch (type) {
        case SubnetType.PUBLIC: return 'Public';
        case SubnetType.PRIVATE: return 'Private';
        case SubnetType.ISOLATED: return 'Isolated';
    }
}
/**
 * Represents a new VPC subnet resource
 *
 * @resource AWS::EC2::Subnet
 */
class Subnet extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * Parts of this VPC subnet
         */
        this.dependencyElements = [];
        this._internetConnectivityEstablished = new core_1.ConcreteDependable();
        Object.defineProperty(this, VPC_SUBNET_SYMBOL, { value: true });
        this.node.applyAspect(new core_1.Tag(NAME_TAG, this.node.path));
        this.availabilityZone = props.availabilityZone;
        const subnet = new ec2_generated_1.CfnSubnet(this, 'Subnet', {
            vpcId: props.vpcId,
            cidrBlock: props.cidrBlock,
            availabilityZone: props.availabilityZone,
            mapPublicIpOnLaunch: props.mapPublicIpOnLaunch,
        });
        this.subnetId = subnet.ref;
        this.subnetVpcId = subnet.attrVpcId;
        this.subnetAvailabilityZone = subnet.attrAvailabilityZone;
        this.subnetIpv6CidrBlocks = subnet.attrIpv6CidrBlocks;
        // subnet.attrNetworkAclAssociationId is the default ACL after the subnet
        // was just created. However, the ACL can be replaced at a later time.
        this._networkAcl = network_acl_1.NetworkAcl.fromNetworkAclId(this, 'Acl', subnet.attrNetworkAclAssociationId);
        this.subnetNetworkAclAssociationId = core_1.Lazy.stringValue({ produce: () => this._networkAcl.networkAclId });
        this.node.defaultChild = subnet;
        const table = new ec2_generated_1.CfnRouteTable(this, 'RouteTable', {
            vpcId: props.vpcId,
        });
        this.routeTable = { routeTableId: table.ref };
        // Associate the public route table for this subnet, to this subnet
        new ec2_generated_1.CfnSubnetRouteTableAssociation(this, 'RouteTableAssociation', {
            subnetId: this.subnetId,
            routeTableId: table.ref,
        });
        this.internetConnectivityEstablished = this._internetConnectivityEstablished;
    }
    static isVpcSubnet(x) {
        return VPC_SUBNET_SYMBOL in x;
    }
    static fromSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    /**
     * Import existing subnet from id.
     */
    // eslint-disable-next-line no-shadow
    static fromSubnetId(scope, id, subnetId) {
        return this.fromSubnetAttributes(scope, id, { subnetId });
    }
    /**
     * Create a default route that points to a passed IGW, with a dependency
     * on the IGW's attachment to the VPC.
     *
     * @param gatewayId the logical ID (ref) of the gateway attached to your VPC
     * @param gatewayAttachment the gateway attachment construct to be added as a dependency
     */
    addDefaultInternetRoute(gatewayId, gatewayAttachment) {
        const route = new ec2_generated_1.CfnRoute(this, 'DefaultRoute', {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: '0.0.0.0/0',
            gatewayId,
        });
        route.node.addDependency(gatewayAttachment);
        // Since the 'route' depends on the gateway attachment, just
        // depending on the route is enough.
        this._internetConnectivityEstablished.add(route);
    }
    /**
     * Network ACL associated with this Subnet
     *
     * Upon creation, this is the default ACL which allows all traffic, except
     * explicit DENY entries that you add.
     *
     * You can replace it with a custom ACL which denies all traffic except
     * the explic it ALLOW entries that you add by creating a `NetworkAcl`
     * object and calling `associateNetworkAcl()`.
     */
    get networkAcl() {
        return this._networkAcl;
    }
    /**
     * Adds an entry to this subnets route table that points to the passed NATGatwayId
     * @param natGatewayId The ID of the NAT gateway
     */
    addDefaultNatRoute(natGatewayId) {
        this.addRoute('DefaultRoute', {
            routerType: RouterType.NAT_GATEWAY,
            routerId: natGatewayId,
            enablesInternetConnectivity: true,
        });
    }
    /**
     * Adds an entry to this subnets route table
     */
    addRoute(id, options) {
        if (options.destinationCidrBlock && options.destinationIpv6CidrBlock) {
            throw new Error('Cannot specify both \'destinationCidrBlock\' and \'destinationIpv6CidrBlock\'');
        }
        const route = new ec2_generated_1.CfnRoute(this, id, {
            routeTableId: this.routeTable.routeTableId,
            destinationCidrBlock: options.destinationCidrBlock || (options.destinationIpv6CidrBlock === undefined ? '0.0.0.0/0' : undefined),
            destinationIpv6CidrBlock: options.destinationIpv6CidrBlock,
            [routerTypeToPropName(options.routerType)]: options.routerId,
        });
        if (options.enablesInternetConnectivity) {
            this._internetConnectivityEstablished.add(route);
        }
    }
    associateNetworkAcl(id, networkAcl) {
        this._networkAcl = networkAcl;
        const scope = core_1.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_1.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
exports.Subnet = Subnet;
/**
 * Type of router used in route
 */
var RouterType;
(function (RouterType) {
    /**
     * Egress-only Internet Gateway
     */
    RouterType["EGRESS_ONLY_INTERNET_GATEWAY"] = "EgressOnlyInternetGateway";
    /**
     * Internet Gateway
     */
    RouterType["GATEWAY"] = "Gateway";
    /**
     * Instance
     */
    RouterType["INSTANCE"] = "Instance";
    /**
     * NAT Gateway
     */
    RouterType["NAT_GATEWAY"] = "NatGateway";
    /**
     * Network Interface
     */
    RouterType["NETWORK_INTERFACE"] = "NetworkInterface";
    /**
     * VPC peering connection
     */
    RouterType["VPC_PEERING_CONNECTION"] = "VpcPeeringConnection";
})(RouterType = exports.RouterType || (exports.RouterType = {}));
function routerTypeToPropName(routerType) {
    return ({
        [RouterType.EGRESS_ONLY_INTERNET_GATEWAY]: 'egressOnlyInternetGatewayId',
        [RouterType.GATEWAY]: 'gatewayId',
        [RouterType.INSTANCE]: 'instanceId',
        [RouterType.NAT_GATEWAY]: 'natGatewayId',
        [RouterType.NETWORK_INTERFACE]: 'networkInterfaceId',
        [RouterType.VPC_PEERING_CONNECTION]: 'vpcPeeringConnectionId',
    })[routerType];
}
/**
 * Represents a public VPC subnet resource
 */
class PublicSubnet extends Subnet {
    static fromPublicSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
    /**
     * Creates a new managed NAT gateway attached to this public subnet.
     * Also adds the EIP for the managed NAT.
     * @returns A ref to the the NAT Gateway ID
     */
    addNatGateway() {
        // Create a NAT Gateway in this public subnet
        const ngw = new ec2_generated_1.CfnNatGateway(this, 'NATGateway', {
            subnetId: this.subnetId,
            allocationId: new ec2_generated_1.CfnEIP(this, 'EIP', {
                domain: 'vpc',
            }).attrAllocationId,
        });
        return ngw;
    }
}
exports.PublicSubnet = PublicSubnet;
/**
 * Represents a private VPC subnet resource
 */
class PrivateSubnet extends Subnet {
    static fromPrivateSubnetAttributes(scope, id, attrs) {
        return new ImportedSubnet(scope, id, attrs);
    }
    constructor(scope, id, props) {
        super(scope, id, props);
    }
}
exports.PrivateSubnet = PrivateSubnet;
function ifUndefined(value, defaultValue) {
    return value !== undefined ? value : defaultValue;
}
class ImportedVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this.availabilityZones = props.availabilityZones;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        /* eslint-disable max-len */
        const pub = new util_1.ImportSubnetGroup(props.publicSubnetIds, props.publicSubnetNames, props.publicSubnetRouteTableIds, SubnetType.PUBLIC, this.availabilityZones, 'publicSubnetIds', 'publicSubnetNames', 'publicSubnetRouteTableIds');
        const priv = new util_1.ImportSubnetGroup(props.privateSubnetIds, props.privateSubnetNames, props.privateSubnetRouteTableIds, SubnetType.PRIVATE, this.availabilityZones, 'privateSubnetIds', 'privateSubnetNames', 'privateSubnetRouteTableIds');
        const iso = new util_1.ImportSubnetGroup(props.isolatedSubnetIds, props.isolatedSubnetNames, props.isolatedSubnetRouteTableIds, SubnetType.ISOLATED, this.availabilityZones, 'isolatedSubnetIds', 'isolatedSubnetNames', 'isolatedSubnetRouteTableIds');
        /* eslint-enable max-len */
        this.publicSubnets = pub.import(this);
        this.privateSubnets = priv.import(this);
        this.isolatedSubnets = iso.import(this);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not supplied when creating this VPC');
        }
        return this.cidr;
    }
}
class LookedUpVpc extends VpcBase {
    constructor(scope, id, props, isIncomplete) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        this.vpcId = props.vpcId;
        this.cidr = props.vpcCidrBlock;
        this._vpnGatewayId = props.vpnGatewayId;
        this.incompleteSubnetDefinition = isIncomplete;
        const subnetGroups = props.subnetGroups || [];
        const availabilityZones = Array.from(new Set(flatMap(subnetGroups, subnetGroup => {
            return subnetGroup.subnets.map(subnet => subnet.availabilityZone);
        })));
        availabilityZones.sort((az1, az2) => az1.localeCompare(az2));
        this.availabilityZones = availabilityZones;
        this.publicSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PUBLIC);
        this.privateSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.PRIVATE);
        this.isolatedSubnets = this.extractSubnetsOfType(subnetGroups, cxapi.VpcSubnetGroupType.ISOLATED);
    }
    get vpcCidrBlock() {
        if (this.cidr === undefined) {
            // Value might be cached from an old CLI version, so bumping the CX API protocol to
            // force the value to exist would not have helped.
            throw new Error('Cannot perform this operation: \'vpcCidrBlock\' was not found when looking up this VPC. Use a newer version of the CDK CLI and clear the old context value.');
        }
        return this.cidr;
    }
    extractSubnetsOfType(subnetGroups, subnetGroupType) {
        return flatMap(subnetGroups.filter(subnetGroup => subnetGroup.type === subnetGroupType), subnetGroup => this.subnetGroupToSubnets(subnetGroup));
    }
    subnetGroupToSubnets(subnetGroup) {
        const ret = new Array();
        for (let i = 0; i < subnetGroup.subnets.length; i++) {
            const vpcSubnet = subnetGroup.subnets[i];
            ret.push(Subnet.fromSubnetAttributes(this, `${subnetGroup.name}Subnet${i + 1}`, {
                availabilityZone: vpcSubnet.availabilityZone,
                subnetId: vpcSubnet.subnetId,
                routeTableId: vpcSubnet.routeTableId,
            }));
        }
        return ret;
    }
}
function flatMap(xs, fn) {
    const ret = new Array();
    for (const x of xs) {
        ret.push(...fn(x));
    }
    return ret;
}
class CompositeDependable {
    constructor() {
        this.dependables = new Array();
        const self = this;
        core_1.DependableTrait.implement(this, {
            get dependencyRoots() {
                const ret = new Array();
                for (const dep of self.dependables) {
                    ret.push(...core_1.DependableTrait.get(dep).dependencyRoots);
                }
                return ret;
            },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(dep) {
        this.dependables.push(dep);
    }
}
/**
 * Invoke a function on a value (for its side effect) and return the value
 */
function tap(x, fn) {
    fn(x);
    return x;
}
class ImportedSubnet extends core_1.Resource {
    constructor(scope, id, attrs) {
        super(scope, id);
        this.internetConnectivityEstablished = new core_1.ConcreteDependable();
        if (!attrs.routeTableId) {
            const ref = core_1.Token.isUnresolved(attrs.subnetId)
                ? `at '${scope.node.path}/${id}'`
                : `'${attrs.subnetId}'`;
            // eslint-disable-next-line max-len
            scope.node.addWarning(`No routeTableId was provided to the subnet ${ref}. Attempting to read its .routeTable.routeTableId will return null/undefined. (More info: https://github.com/aws/aws-cdk/pull/3171)`);
        }
        this._availabilityZone = attrs.availabilityZone;
        this.subnetId = attrs.subnetId;
        this.routeTable = {
            // Forcing routeTableId to pretend non-null to maintain backwards-compatibility. See https://github.com/aws/aws-cdk/pull/3171
            routeTableId: attrs.routeTableId,
        };
    }
    get availabilityZone() {
        if (!this._availabilityZone) {
            // eslint-disable-next-line max-len
            throw new Error("You cannot reference a Subnet's availability zone if it was not supplied. Add the availabilityZone when importing using Subnet.fromSubnetAttributes()");
        }
        return this._availabilityZone;
    }
    associateNetworkAcl(id, networkAcl) {
        const scope = core_1.Construct.isConstruct(networkAcl) ? networkAcl : this;
        const other = core_1.Construct.isConstruct(networkAcl) ? this : networkAcl;
        new network_acl_1.SubnetNetworkAclAssociation(scope, id + other.node.uniqueId, {
            networkAcl,
            subnet: this,
        });
    }
}
/**
 * Determine (and validate) the NAT gateway count w.r.t. the rest of the subnet configuration
 *
 * We have the following requirements:
 *
 * - NatGatewayCount = 0 ==> there are no private subnets
 * - NatGatewayCount > 0 ==> there must be public subnets
 *
 * Do we want to require that there are private subnets if there are NatGateways?
 * They seem pointless but I see no reason to prevent it.
 */
function determineNatGatewayCount(requestedCount, subnetConfig, azCount) {
    const hasPrivateSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PRIVATE && !c.reserved);
    const hasPublicSubnets = subnetConfig.some(c => c.subnetType === SubnetType.PUBLIC);
    const count = requestedCount !== undefined ? Math.min(requestedCount, azCount) : (hasPrivateSubnets ? azCount : 0);
    if (count === 0 && hasPrivateSubnets) {
        // eslint-disable-next-line max-len
        throw new Error('If you do not want NAT gateways (natGateways=0), make sure you don\'t configure any PRIVATE subnets in \'subnetConfiguration\' (make them PUBLIC or ISOLATED instead)');
    }
    if (count > 0 && !hasPublicSubnets) {
        // eslint-disable-next-line max-len
        throw new Error(`If you configure PRIVATE subnets in 'subnetConfiguration', you must also configure PUBLIC subnets to put the NAT gateways into (got ${JSON.stringify(subnetConfig)}.`);
    }
    return count;
}
/**
 * There are returned when the provider has not supplied props yet
 *
 * It's only used for testing and on the first run-through.
 */
const DUMMY_VPC_PROPS = {
    availabilityZones: [],
    vpcCidrBlock: '1.2.3.4/5',
    isolatedSubnetIds: undefined,
    isolatedSubnetNames: undefined,
    isolatedSubnetRouteTableIds: undefined,
    privateSubnetIds: undefined,
    privateSubnetNames: undefined,
    privateSubnetRouteTableIds: undefined,
    publicSubnetIds: undefined,
    publicSubnetNames: undefined,
    publicSubnetRouteTableIds: undefined,
    subnetGroups: [
        {
            name: 'Public',
            type: cxapi.VpcSubnetGroupType.PUBLIC,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 's-12345',
                    routeTableId: 'rtb-12345s',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 's-67890',
                    routeTableId: 'rtb-67890s',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
        {
            name: 'Private',
            type: cxapi.VpcSubnetGroupType.PRIVATE,
            subnets: [
                {
                    availabilityZone: 'dummy1a',
                    subnetId: 'p-12345',
                    routeTableId: 'rtb-12345p',
                    cidr: '1.2.3.4/5',
                },
                {
                    availabilityZone: 'dummy1b',
                    subnetId: 'p-67890',
                    routeTableId: 'rtb-57890p',
                    cidr: '1.2.3.4/5',
                },
            ],
        },
    ],
    vpcId: 'vpc-12345',
};
//# sourceMappingURL=data:application/json;base64,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