"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SubnetNetworkAclAssociation = exports.NetworkAclEntry = exports.TrafficDirection = exports.Action = exports.NetworkAcl = void 0;
const core_1 = require("@aws-cdk/core");
const ec2_generated_1 = require("./ec2.generated");
/**
 * A NetworkAclBase that is not created in this template
 *
 * @experimental
 */
class NetworkAclBase extends core_1.Resource {
    /**
     * Add a new entry to the ACL
     */
    addEntry(id, options) {
        return new NetworkAclEntry(this, id, {
            networkAcl: this,
            ...options,
        });
    }
}
/**
 * Define a new custom network ACL
 *
 * By default, will deny all inbound and outbound traffic unless entries are
 * added explicitly allowing it.
 *
 * @experimental
 */
class NetworkAcl extends NetworkAclBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclName,
        });
        this.vpc = props.vpc;
        this.networkAcl = new ec2_generated_1.CfnNetworkAcl(this, 'Resource', {
            vpcId: props.vpc.vpcId,
        });
        this.networkAclId = this.networkAcl.ref;
        this.networkAclVpcId = this.networkAcl.vpcId;
        if (props.subnetSelection !== undefined) {
            this.associateWithSubnet('DefaultAssociation', props.subnetSelection);
        }
    }
    /**
     * Import an existing NetworkAcl into this app.
     */
    static fromNetworkAclId(scope, id, networkAclId) {
        class Import extends NetworkAclBase {
            constructor() {
                super(...arguments);
                this.networkAclId = networkAclId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Associate the ACL with a given set of subnets
     */
    associateWithSubnet(id, selection) {
        const subnets = this.vpc.selectSubnets(selection);
        for (const subnet of subnets.subnets) {
            subnet.associateNetworkAcl(id, this);
        }
    }
}
exports.NetworkAcl = NetworkAcl;
/**
 * What action to apply to traffic matching the ACL
 *
 * @experimental
 */
var Action;
(function (Action) {
    /**
     * Allow the traffic
     */
    Action["ALLOW"] = "allow";
    /**
     * Deny the traffic
     */
    Action["DENY"] = "deny";
})(Action = exports.Action || (exports.Action = {}));
/**
 * Base class for NetworkAclEntries
 *
 * @experimental
 */
class NetworkAclEntryBase extends core_1.Resource {
}
/**
 * Direction of traffic the AclEntry applies to
 *
 * @experimental
 */
var TrafficDirection;
(function (TrafficDirection) {
    /**
     * Traffic leaving the subnet
     */
    TrafficDirection[TrafficDirection["EGRESS"] = 0] = "EGRESS";
    /**
     * Traffic entering the subnet
     */
    TrafficDirection[TrafficDirection["INGRESS"] = 1] = "INGRESS";
})(TrafficDirection = exports.TrafficDirection || (exports.TrafficDirection = {}));
/**
 * Define an entry in a Network ACL table
 *
 * @experimental
 */
class NetworkAclEntry extends NetworkAclEntryBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.networkAclEntryName,
        });
        this.networkAcl = props.networkAcl;
        new ec2_generated_1.CfnNetworkAclEntry(this, 'Resource', {
            networkAclId: this.networkAcl.networkAclId,
            ruleNumber: props.ruleNumber,
            ruleAction: props.ruleAction !== undefined ? props.ruleAction : Action.ALLOW,
            egress: props.direction !== undefined ? props.direction === TrafficDirection.EGRESS : undefined,
            ...props.traffic.toTrafficConfig(),
            ...props.cidr.toCidrConfig(),
        });
    }
}
exports.NetworkAclEntry = NetworkAclEntry;
/**
 * Associate a network ACL with a subnet
 *
 * @experimental
 */
class SubnetNetworkAclAssociationBase extends core_1.Resource {
}
class SubnetNetworkAclAssociation extends SubnetNetworkAclAssociationBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.subnetNetworkAclAssociationName,
        });
        this.association = new ec2_generated_1.CfnSubnetNetworkAclAssociation(this, 'Resource', {
            networkAclId: props.networkAcl.networkAclId,
            subnetId: props.subnet.subnetId,
        });
        this.networkAcl = props.networkAcl;
        this.subnet = props.subnet;
        this.subnetNetworkAclAssociationAssociationId = this.association.attrAssociationId;
    }
    static fromSubnetNetworkAclAssociationAssociationId(scope, id, subnetNetworkAclAssociationAssociationId) {
        class Import extends SubnetNetworkAclAssociationBase {
            constructor() {
                super(...arguments);
                this.subnetNetworkAclAssociationAssociationId = subnetNetworkAclAssociationAssociationId;
            }
        }
        return new Import(scope, id);
    }
}
exports.SubnetNetworkAclAssociation = SubnetNetworkAclAssociation;
//# sourceMappingURL=data:application/json;base64,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