"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Instance = void 0;
const crypto = require("crypto");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const connections_1 = require("./connections");
const ec2_generated_1 = require("./ec2.generated");
const security_group_1 = require("./security-group");
const volume_1 = require("./volume");
const vpc_1 = require("./vpc");
/**
 * Name tag constant
 */
const NAME_TAG = 'Name';
/**
 * This represents a single EC2 instance
 */
class Instance extends core_1.Resource {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        this.securityGroups = [];
        if (props.initOptions && !props.init) {
            throw new Error('Setting \'initOptions\' requires that \'init\' is also set');
        }
        if (props.securityGroup) {
            this.securityGroup = props.securityGroup;
        }
        else {
            this.securityGroup = new security_group_1.SecurityGroup(this, 'InstanceSecurityGroup', {
                vpc: props.vpc,
                allowAllOutbound: props.allowAllOutbound !== false,
            });
        }
        this.connections = new connections_1.Connections({ securityGroups: [this.securityGroup] });
        this.securityGroups.push(this.securityGroup);
        core_1.Tag.add(this, NAME_TAG, props.instanceName || this.node.path);
        this.role = props.role || new iam.Role(this, 'InstanceRole', {
            assumedBy: new iam.ServicePrincipal('ec2.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const iamProfile = new iam.CfnInstanceProfile(this, 'InstanceProfile', {
            roles: [this.role.roleName],
        });
        // use delayed evaluation
        const imageConfig = props.machineImage.getImage(this);
        this.userData = (_a = props.userData) !== null && _a !== void 0 ? _a : imageConfig.userData;
        const userDataToken = core_1.Lazy.stringValue({ produce: () => core_1.Fn.base64(this.userData.render()) });
        const securityGroupsToken = core_1.Lazy.listValue({ produce: () => this.securityGroups.map(sg => sg.securityGroupId) });
        const { subnets } = props.vpc.selectSubnets(props.vpcSubnets);
        let subnet;
        if (props.availabilityZone) {
            const selected = subnets.filter(sn => sn.availabilityZone === props.availabilityZone);
            if (selected.length === 1) {
                subnet = selected[0];
            }
            else {
                this.node.addError(`Need exactly 1 subnet to match AZ '${props.availabilityZone}', found ${selected.length}. Use a different availabilityZone.`);
            }
        }
        else {
            if (subnets.length > 0) {
                subnet = subnets[0];
            }
            else {
                this.node.addError(`Did not find any subnets matching '${JSON.stringify(props.vpcSubnets)}', please use a different selection.`);
            }
        }
        if (!subnet) {
            // We got here and we don't have a subnet because of validation errors.
            // Invent one on the spot so the code below doesn't fail.
            subnet = vpc_1.Subnet.fromSubnetAttributes(this, 'DummySubnet', {
                subnetId: 's-notfound',
                availabilityZone: 'az-notfound',
            });
        }
        this.instance = new ec2_generated_1.CfnInstance(this, 'Resource', {
            imageId: imageConfig.imageId,
            keyName: props.keyName,
            instanceType: props.instanceType.toString(),
            securityGroupIds: securityGroupsToken,
            iamInstanceProfile: iamProfile.ref,
            userData: userDataToken,
            subnetId: subnet.subnetId,
            availabilityZone: subnet.availabilityZone,
            sourceDestCheck: props.sourceDestCheck,
            blockDeviceMappings: props.blockDevices !== undefined ? volume_1.synthesizeBlockDeviceMappings(this, props.blockDevices) : undefined,
            privateIpAddress: props.privateIpAddress,
        });
        this.instance.node.addDependency(this.role);
        this.osType = imageConfig.osType;
        this.node.defaultChild = this.instance;
        this.instanceId = this.instance.ref;
        this.instanceAvailabilityZone = this.instance.attrAvailabilityZone;
        this.instancePrivateDnsName = this.instance.attrPrivateDnsName;
        this.instancePrivateIp = this.instance.attrPrivateIp;
        this.instancePublicDnsName = this.instance.attrPublicDnsName;
        this.instancePublicIp = this.instance.attrPublicIp;
        if (props.init) {
            this.applyCloudFormationInit(props.init, props.initOptions);
        }
        this.applyUpdatePolicies(props);
        // Trigger replacement (via new logical ID) on user data change, if specified or cfn-init is being used.
        const originalLogicalId = core_1.Stack.of(this).getLogicalId(this.instance);
        this.instance.overrideLogicalId(core_1.Lazy.stringValue({
            produce: () => {
                var _a;
                let logicalId = originalLogicalId;
                if ((_a = props.userDataCausesReplacement) !== null && _a !== void 0 ? _a : props.initOptions) {
                    const md5 = crypto.createHash('md5');
                    md5.update(this.userData.render());
                    logicalId += md5.digest('hex').substr(0, 16);
                }
                return logicalId;
            },
        }));
    }
    /**
     * Add the security group to the instance.
     *
     * @param securityGroup: The security group to add
     */
    addSecurityGroup(securityGroup) {
        this.securityGroups.push(securityGroup);
    }
    /**
     * Add command to the startup script of the instance.
     * The command must be in the scripting language supported by the instance's OS (i.e. Linux/Windows).
     */
    addUserData(...commands) {
        this.userData.addCommands(...commands);
    }
    /**
     * Adds a statement to the IAM role assumed by the instance.
     */
    addToRolePolicy(statement) {
        this.role.addToPolicy(statement);
    }
    /**
     * Use a CloudFormation Init configuration at instance startup
     *
     * This does the following:
     *
     * - Attaches the CloudFormation Init metadata to the Instance resource.
     * - Add commands to the instance UserData to run `cfn-init` and `cfn-signal`.
     * - Update the instance's CreationPolicy to wait for the `cfn-signal` commands.
     */
    applyCloudFormationInit(init, options = {}) {
        var _a;
        init._attach(this.instance, {
            platform: this.osType,
            instanceRole: this.role,
            userData: this.userData,
            configSets: options.configSets,
            embedFingerprint: options.embedFingerprint,
            printLog: options.printLog,
            ignoreFailures: options.ignoreFailures,
        });
        this.waitForResourceSignal((_a = options.timeout) !== null && _a !== void 0 ? _a : core_1.Duration.minutes(5));
    }
    /**
     * Wait for a single additional resource signal
     *
     * Add 1 to the current ResourceSignal Count and add the given timeout to the current timeout.
     *
     * Use this to pause the CloudFormation deployment to wait for the instances
     * in the AutoScalingGroup to report successful startup during
     * creation and updates. The UserData script needs to invoke `cfn-signal`
     * with a success or failure code after it is done setting up the instance.
     */
    waitForResourceSignal(timeout) {
        var _a, _b;
        const oldResourceSignal = (_a = this.instance.cfnOptions.creationPolicy) === null || _a === void 0 ? void 0 : _a.resourceSignal;
        this.instance.cfnOptions.creationPolicy = {
            ...this.instance.cfnOptions.creationPolicy,
            resourceSignal: {
                count: ((_b = oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.count) !== null && _b !== void 0 ? _b : 0) + 1,
                timeout: ((oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout) ? core_1.Duration.parse(oldResourceSignal === null || oldResourceSignal === void 0 ? void 0 : oldResourceSignal.timeout).plus(timeout) : timeout).toIsoString(),
            },
        };
    }
    /**
     * Apply CloudFormation update policies for the instance
     */
    applyUpdatePolicies(props) {
        if (props.resourceSignalTimeout !== undefined) {
            this.instance.cfnOptions.creationPolicy = {
                ...this.instance.cfnOptions.creationPolicy,
                resourceSignal: {
                    timeout: props.resourceSignalTimeout && props.resourceSignalTimeout.toISOString(),
                },
            };
        }
    }
}
exports.Instance = Instance;
//# sourceMappingURL=data:application/json;base64,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