# OpenTelemetry Instrumentation for Discord.py

This library provides OpenTelemetry instrumentation for bots built with `discord.py`, making it easy to collect traces, metrics, and logs to analyze the performance and reliability of your Discord bots.

## Features

- **Automatic Instrumentation**: Automatically instruments your Discord.py bot to trace key events and operations with minimal setup.
- **Manual Instrumentation**: Provides decorators for fine-grained control over tracing specific commands and event handlers.
- **Custom Event Support**: Register your own custom events for tracing, beyond the standard Discord.py events.
- **Comprehensive Coverage**: Instruments event listeners, message operations, commands, HTTP requests, and voice operations.
- **Detailed Context**: Captures rich contextual information such as guild IDs, channel names, message content lengths, and more.
- **Flexible Configuration**: Allows you to enable or disable instrumentation for specific components based on your needs.
- **Performance Monitoring**: Track execution times, identify bottlenecks, and monitor the health of your Discord bot.
- **Error Tracking**: Automatically captures exceptions and errors in your bot's operations.
- **Seamless Integration**: Works with any OpenTelemetry exporter (Console, OTLP, Jaeger, etc.).

## Features Overview

The OpenTelemetry Instrumentation for Discord.py enhances the observability of your Discord bots, providing detailed insights into their operations and interactions. Here are the key components that are instrumented:

### Event Listeners
- **on_message**: Traces message processing and the bot's responsiveness to user messages.
- **on_ready**: Monitors bot startup and initialization.
- **Other Events**: Automatically traces all Discord.py events.

### Message Operations
- **send_message**: Traces message sending operations, including execution times and potential bottlenecks.
- **edit_message**: Captures information about message edit operations.
- **delete_message**: Monitors message deletion operations.

### Commands
- **Command Invocation**: Traces command execution, including parsing and handling.
- **Command Errors**: Captures and traces command errors and exceptions.

### HTTP Operations
- **API Requests**: Traces Discord API requests made by the bot.
- **Rate Limiting**: Monitors rate limit encounters and handling.

### Voice Operations
- **Voice Connections**: Traces voice channel connections and disconnections.
- **Audio Playback**: Monitors audio streaming and playback operations.

## Installation

Install this package with pip:

```
pip install opentelemetry-instrumentation-discordpy
```

## Usage

### Automatic Instrumentation

To automatically instrument your Discord bot, simply initialize the instrumentation at the start of your bot's code:

```
from opentelemetry_instrumentation_discordpy import DiscordPyInstrumentor
DiscordPyInstrumentor().instrument()
```

### Manual Instrumentation

For more fine-grained control, use the provided decorators to instrument specific commands or event handlers:

```python
from opentelemetry_instrumentation_discordpy.decorators import trace_command, trace_event
from opentelemetry import trace
from opentelemetry.trace import SpanKind

# Get a tracer for custom spans
tracer = trace.get_tracer("my-discord-bot")

@bot.event
@trace_event(attributes={"custom.attribute": "value"})
async def on_message(message):
    # Your code here...
    await bot.process_commands(message)

@bot.command()
@trace_command(attributes={"command.type": "utility"})
async def ping(ctx):
    # Create a custom span for measuring latency
    with tracer.start_as_current_span(
        "ping_latency",
        kind=SpanKind.INTERNAL,
        attributes={"discord.channel.id": str(ctx.channel.id)}
    ) as span:
        # Measure latency
        message = await ctx.send("Pinging...")
        latency = bot.latency * 1000
        
        # Record latency in span
        span.set_attribute("ping.latency_ms", latency)
        
        await message.edit(content=f"Pong! Latency: {latency:.2f}ms")
```

### Examples

For complete examples of setting up a bot with OpenTelemetry instrumentation:

- [Basic Bot Example](examples/basic_bot.py): A simple bot with automatic instrumentation
- [Advanced Bot Example](examples/advanced_bot.py): A more complex bot with manual instrumentation and custom spans

## Configuration

### Configuring the Instrumentor

You can configure which components of discord.py to instrument:

```python
# Instrument only specific components
DiscordPyInstrumentor().instrument(
    event_listeners=True,    # Instrument event listeners (on_message, on_ready, etc.)
    message_operations=True, # Instrument message operations (send, edit, delete)
    commands=True,           # Instrument command handling
    http=True,               # Instrument HTTP requests
    voice=True               # Instrument voice operations
)
```

### Setting Up Exporters

The library works with any OpenTelemetry exporter:

```python
# Console exporter (for development)
from opentelemetry.sdk.trace.export import ConsoleSpanExporter, SimpleSpanProcessor
tracer_provider.add_span_processor(SimpleSpanProcessor(ConsoleSpanExporter()))

# OTLP exporter (for production)
from opentelemetry.exporter.otlp.proto.grpc.trace_exporter import OTLPSpanExporter
from opentelemetry.sdk.trace.export import BatchSpanProcessor
otlp_exporter = OTLPSpanExporter(endpoint="your-otlp-endpoint:4317")
tracer_provider.add_span_processor(BatchSpanProcessor(otlp_exporter))
```

For more detailed configuration options and advanced usage, refer to the [documentation](./docs/usage.md).

## Contributing

Contributions are welcome! See [CONTRIBUTING.md](https://github.com/Idegrity/opentelemetry-instrumentation-discordpy/blob/main/docs/CONTRIBUTING.md) for how to get started.

## License

This library is licensed under the [BSD License](https://github.com/Idegrity/opentelemetry-instrumentation-discordpy/blob/main/LICENSE).

## Support

If you encounter any issues or have questions, please file an issue on GitHub.
