"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Placeholders which can be used manifests
 *
 * These can occur both in the Asset Manifest as well as the general
 * Cloud Assembly manifest.
 */
class EnvironmentPlaceholders {
    /**
     * Replace the environment placeholders in all strings found in a complex object.
     *
     * Duplicated between cdk-assets and aws-cdk CLI because we don't have a good single place to put it
     * (they're nominally independent tools).
     */
    static replace(object, values) {
        return this.recurse(object, value => {
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_REGION, values.region);
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_ACCOUNT, values.accountId);
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_PARTITION, values.partition);
            return value;
        });
    }
    /**
     * Like 'replace', but asynchronous
     */
    static async replaceAsync(object, provider) {
        let needRegion = false;
        let needAccountId = false;
        let needPartition = false;
        this.recurse(object, value => {
            if (value.indexOf(EnvironmentPlaceholders.CURRENT_REGION) > 1) {
                needRegion = true;
            }
            if (value.indexOf(EnvironmentPlaceholders.CURRENT_ACCOUNT) > 1) {
                needAccountId = true;
            }
            if (value.indexOf(EnvironmentPlaceholders.CURRENT_PARTITION) > 1) {
                needPartition = true;
            }
            return value;
        });
        const region = needRegion ? await provider.region() : undefined;
        const accountId = needAccountId ? await provider.accountId() : undefined;
        const partition = needPartition ? await provider.partition() : undefined;
        return this.recurse(object, value => {
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_REGION, region !== null && region !== void 0 ? region : 'WONTHAPPEN');
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_ACCOUNT, accountId !== null && accountId !== void 0 ? accountId : 'WONTHAPPEN');
            value = replaceAll(value, EnvironmentPlaceholders.CURRENT_PARTITION, partition !== null && partition !== void 0 ? partition : 'WONTHAPPEN');
            return value;
        });
    }
    static recurse(value, cb) {
        if (typeof value === 'string') {
            return cb(value);
        }
        if (typeof value !== 'object' || value === null) {
            return value;
        }
        if (Array.isArray(value)) {
            return value.map(x => this.recurse(x, cb));
        }
        const ret = {};
        for (const [key, inner] of Object.entries(value)) {
            ret[key] = this.recurse(inner, cb);
        }
        return ret;
    }
}
exports.EnvironmentPlaceholders = EnvironmentPlaceholders;
/**
 * Insert this into the destination fields to be replaced with the current region
 */
EnvironmentPlaceholders.CURRENT_REGION = '${AWS::Region}';
/**
 * Insert this into the destination fields to be replaced with the current account
 */
EnvironmentPlaceholders.CURRENT_ACCOUNT = '${AWS::AccountId}';
/**
 * Insert this into the destination fields to be replaced with the current partition
 */
EnvironmentPlaceholders.CURRENT_PARTITION = '${AWS::Partition}';
/**
 * A "replace-all" function that doesn't require us escaping a literal string to a regex
 */
function replaceAll(s, search, replace) {
    return s.split(search).join(replace);
}
//# sourceMappingURL=data:application/json;base64,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