"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("../../cloud-assembly-schema"); // Automatically re-written from '@aws-cdk/cloud-assembly-schema'
const metadata_1 = require("./metadata");
/**
 * Represents an artifact within a cloud assembly.
 */
class CloudArtifact {
    constructor(assembly, id, manifest) {
        this.assembly = assembly;
        this.id = id;
        this.manifest = manifest;
        this.messages = this.renderMessages();
        this._dependencyIDs = manifest.dependencies || [];
    }
    /**
     * Returns a subclass of `CloudArtifact` based on the artifact type defined in the artifact manifest.
     * @param assembly The cloud assembly from which to load the artifact
     * @param id The artifact ID
     * @param artifact The artifact manifest
     * @returns the `CloudArtifact` that matches the artifact type or `undefined` if it's an artifact type that is unrecognized by this module.
     */
    static fromManifest(assembly, id, artifact) {
        switch (artifact.type) {
            case cxschema.ArtifactType.AWS_CLOUDFORMATION_STACK:
                return new cloudformation_artifact_1.CloudFormationStackArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.CDK_TREE:
                return new tree_cloud_artifact_1.TreeCloudArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.ASSET_MANIFEST:
                return new asset_manifest_artifact_1.AssetManifestArtifact(assembly, id, artifact);
            case cxschema.ArtifactType.NESTED_CLOUD_ASSEMBLY:
                return new nested_cloud_assembly_artifact_1.NestedCloudAssemblyArtifact(assembly, id, artifact);
            default:
                return undefined;
        }
    }
    /**
     * Returns all the artifacts that this artifact depends on.
     */
    get dependencies() {
        if (this._deps) {
            return this._deps;
        }
        this._deps = this._dependencyIDs.map(id => {
            const dep = this.assembly.tryGetArtifact(id);
            if (!dep) {
                throw new Error(`Artifact ${this.id} depends on non-existing artifact ${id}`);
            }
            return dep;
        });
        return this._deps;
    }
    /**
     * @returns all the metadata entries of a specific type in this artifact.
     * @param type
     */
    findMetadataByType(type) {
        const result = new Array();
        for (const path of Object.keys(this.manifest.metadata || {})) {
            for (const entry of (this.manifest.metadata || {})[path]) {
                if (entry.type === type) {
                    result.push({ path, ...entry });
                }
            }
        }
        return result;
    }
    renderMessages() {
        const messages = new Array();
        for (const [id, metadata] of Object.entries(this.manifest.metadata || {})) {
            for (const entry of metadata) {
                let level;
                switch (entry.type) {
                    case cxschema.ArtifactMetadataEntryType.WARN:
                        level = metadata_1.SynthesisMessageLevel.WARNING;
                        break;
                    case cxschema.ArtifactMetadataEntryType.ERROR:
                        level = metadata_1.SynthesisMessageLevel.ERROR;
                        break;
                    case cxschema.ArtifactMetadataEntryType.INFO:
                        level = metadata_1.SynthesisMessageLevel.INFO;
                        break;
                    default:
                        continue;
                }
                messages.push({ level, entry, id });
            }
        }
        return messages;
    }
}
exports.CloudArtifact = CloudArtifact;
// needs to be defined at the end to avoid a cyclic dependency
const asset_manifest_artifact_1 = require("./artifacts/asset-manifest-artifact");
const cloudformation_artifact_1 = require("./artifacts/cloudformation-artifact");
const nested_cloud_assembly_artifact_1 = require("./artifacts/nested-cloud-assembly-artifact");
const tree_cloud_artifact_1 = require("./artifacts/tree-cloud-artifact");
//# sourceMappingURL=data:application/json;base64,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