"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const fs = require("fs");
const path = require("path");
// don't use "require" since the typescript compiler emits errors since this
// file is not listed in tsconfig.json.
const metadata = JSON.parse(fs.readFileSync(path.join(__dirname, 'sdk-api-metadata.json'), 'utf-8'));
const awsSdkMetadata = metadata;
/**
 * Physical ID of the custom resource.
 */
class PhysicalResourceId {
    /**
     * @param responsePath Path to a response data element to be used as the physical id.
     * @param id Literal string to be used as the physical id.
     */
    constructor(responsePath, id) {
        this.responsePath = responsePath;
        this.id = id;
    }
    /**
     * Extract the physical resource id from the path (dot notation) to the data in the API call response.
     */
    static fromResponse(responsePath) {
        return new PhysicalResourceId(responsePath, undefined);
    }
    /**
     * Explicit physical resource id.
     */
    static of(id) {
        return new PhysicalResourceId(undefined, id);
    }
}
exports.PhysicalResourceId = PhysicalResourceId;
/**
 * The IAM Policy that will be applied to the different calls.
 */
class AwsCustomResourcePolicy {
    /**
     * @param statements statements for explicit policy.
     * @param resources resources for auto-generated from SDK calls.
     */
    constructor(statements, resources) {
        this.statements = statements;
        this.resources = resources;
    }
    /**
     * Explicit IAM Policy Statements.
     *
     * @param statements the statements to propagate to the SDK calls.
     */
    static fromStatements(statements) {
        return new AwsCustomResourcePolicy(statements, undefined);
    }
    /**
     * Generate IAM Policy Statements from the configured SDK calls.
     *
     * Each SDK call with be translated to an IAM Policy Statement in the form of: `call.service:call.action` (e.g `s3:PutObject`).
     *
     * @param options options for the policy generation
     */
    static fromSdkCalls(options) {
        return new AwsCustomResourcePolicy([], options.resources);
    }
}
exports.AwsCustomResourcePolicy = AwsCustomResourcePolicy;
/**
 * Use this constant to configure access to any resource.
 */
AwsCustomResourcePolicy.ANY_RESOURCE = ['*'];
/**
 * Defines a custom resource that is materialized using specific AWS API calls.
 *
 * Use this to bridge any gap that might exist in the CloudFormation Coverage.
 * You can specify exactly which calls are invoked for the 'CREATE', 'UPDATE' and 'DELETE' life cycle events.
 *
 */
class AwsCustomResource extends cdk.Construct {
    // 'props' cannot be optional, even though all its properties are optional.
    // this is because at least one sdk call must be provided.
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        if (!props.onCreate && !props.onUpdate && !props.onDelete) {
            throw new Error('At least `onCreate`, `onUpdate` or `onDelete` must be specified.');
        }
        for (const call of [props.onCreate, props.onUpdate]) {
            if (call && !call.physicalResourceId) {
                throw new Error('`physicalResourceId` must be specified for onCreate and onUpdate calls.');
            }
        }
        for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
            if ((_a = call === null || call === void 0 ? void 0 : call.physicalResourceId) === null || _a === void 0 ? void 0 : _a.responsePath) {
                AwsCustomResource.breakIgnoreErrorsCircuit([call], 'PhysicalResourceId.fromResponse');
            }
        }
        this.props = props;
        const provider = new lambda.SingletonFunction(this, 'Provider', {
            code: lambda.Code.fromAsset(path.join(__dirname, 'runtime')),
            runtime: lambda.Runtime.NODEJS_12_X,
            handler: 'index.handler',
            uuid: '679f53fa-c002-430c-b0da-5b7982bd2287',
            lambdaPurpose: 'AWS',
            timeout: props.timeout || cdk.Duration.minutes(2),
            role: props.role,
            logRetention: props.logRetention,
        });
        this.grantPrincipal = provider.grantPrincipal;
        if (props.policy.statements.length !== 0) {
            // Use custom statements provided by the user
            for (const statement of props.policy.statements) {
                provider.addToRolePolicy(statement);
            }
        }
        else {
            // Derive statements from AWS SDK calls
            for (const call of [props.onCreate, props.onUpdate, props.onDelete]) {
                if (call) {
                    provider.addToRolePolicy(new iam.PolicyStatement({
                        actions: [awsSdkToIamAction(call.service, call.action)],
                        resources: props.policy.resources,
                    }));
                }
            }
        }
        const create = props.onCreate || props.onUpdate;
        this.customResource = new cdk.CustomResource(this, 'Resource', {
            resourceType: props.resourceType || 'Custom::AWS',
            serviceToken: provider.functionArn,
            pascalCaseProperties: true,
            properties: {
                create: create && encodeBooleans(create),
                update: props.onUpdate && encodeBooleans(props.onUpdate),
                delete: props.onDelete && encodeBooleans(props.onDelete),
            },
        });
    }
    static breakIgnoreErrorsCircuit(sdkCalls, caller) {
        for (const call of sdkCalls) {
            if (call === null || call === void 0 ? void 0 : call.ignoreErrorCodesMatching) {
                throw new Error(`\`${caller}\`` + ' cannot be called along with `ignoreErrorCodesMatching`.');
            }
        }
    }
    /**
     * Returns response data for the AWS SDK call.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Use `Token.asXxx` to encode the returned `Reference` as a specific type or
     * use the convenience `getDataString` for string attributes.
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseFieldReference(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getData');
        return this.customResource.getAtt(dataPath);
    }
    /**
     * Returns response data for the AWS SDK call as string.
     *
     * Example for S3 / listBucket : 'Buckets.0.Name'
     *
     * Note that you cannot use this method if `ignoreErrorCodesMatching`
     * is configured for any of the SDK calls. This is because in such a case,
     * the response data might not exist, and will cause a CloudFormation deploy time error.
     *
     * @param dataPath the path to the data
     */
    getResponseField(dataPath) {
        AwsCustomResource.breakIgnoreErrorsCircuit([this.props.onCreate, this.props.onUpdate], 'getDataString');
        return this.customResource.getAttString(dataPath);
    }
}
exports.AwsCustomResource = AwsCustomResource;
/**
 * Transform SDK service/action to IAM action using metadata from aws-sdk module.
 * Example: CloudWatchLogs with putRetentionPolicy => logs:PutRetentionPolicy
 *
 * TODO: is this mapping correct for all services?
 */
function awsSdkToIamAction(service, action) {
    const srv = service.toLowerCase();
    const iamService = (awsSdkMetadata[srv] && awsSdkMetadata[srv].prefix) || srv;
    const iamAction = action.charAt(0).toUpperCase() + action.slice(1);
    return `${iamService}:${iamAction}`;
}
/**
 * Encodes booleans as special strings
 */
function encodeBooleans(object) {
    return JSON.parse(JSON.stringify(object), (_k, v) => {
        switch (v) {
            case true:
                return 'TRUE:BOOLEAN';
            case false:
                return 'FALSE:BOOLEAN';
            default:
                return v;
        }
    });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXdzLWN1c3RvbS1yZXNvdXJjZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImF3cy1jdXN0b20tcmVzb3VyY2UudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx3Q0FBd0MsQ0FBQyxtREFBbUQ7QUFDNUYsOENBQThDLENBQUMsc0RBQXNEO0FBRXJHLHFDQUFxQyxDQUFDLGdEQUFnRDtBQUN0Rix5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLDRFQUE0RTtBQUM1RSx1Q0FBdUM7QUFDdkMsTUFBTSxRQUFRLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLHVCQUF1QixDQUFDLEVBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQztBQU9yRyxNQUFNLGNBQWMsR0FBbUIsUUFBUSxDQUFDO0FBQ2hEOztHQUVHO0FBQ0gsTUFBYSxrQkFBa0I7SUFhM0I7OztPQUdHO0lBQ0gsWUFBb0MsWUFBcUIsRUFBa0IsRUFBVztRQUFsRCxpQkFBWSxHQUFaLFlBQVksQ0FBUztRQUFrQixPQUFFLEdBQUYsRUFBRSxDQUFTO0lBQUksQ0FBQztJQWhCM0Y7O09BRUc7SUFDSSxNQUFNLENBQUMsWUFBWSxDQUFDLFlBQW9CO1FBQzNDLE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxZQUFZLEVBQUUsU0FBUyxDQUFDLENBQUM7SUFDM0QsQ0FBQztJQUNEOztPQUVHO0lBQ0ksTUFBTSxDQUFDLEVBQUUsQ0FBQyxFQUFVO1FBQ3ZCLE9BQU8sSUFBSSxrQkFBa0IsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUM7SUFDakQsQ0FBQztDQU1KO0FBbEJELGdEQWtCQztBQWlGRDs7R0FFRztBQUNILE1BQWEsdUJBQXVCO0lBdUJoQzs7O09BR0c7SUFDSCxZQUFvQyxVQUFpQyxFQUFrQixTQUFvQjtRQUF2RSxlQUFVLEdBQVYsVUFBVSxDQUF1QjtRQUFrQixjQUFTLEdBQVQsU0FBUyxDQUFXO0lBQUksQ0FBQztJQXRCaEg7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxjQUFjLENBQUMsVUFBaUM7UUFDMUQsT0FBTyxJQUFJLHVCQUF1QixDQUFDLFVBQVUsRUFBRSxTQUFTLENBQUMsQ0FBQztJQUM5RCxDQUFDO0lBQ0Q7Ozs7OztPQU1HO0lBQ0ksTUFBTSxDQUFDLFlBQVksQ0FBQyxPQUE4QjtRQUNyRCxPQUFPLElBQUksdUJBQXVCLENBQUMsRUFBRSxFQUFFLE9BQU8sQ0FBQyxTQUFTLENBQUMsQ0FBQztJQUM5RCxDQUFDOztBQXRCTCwwREE0QkM7QUEzQkc7O0dBRUc7QUFDb0Isb0NBQVksR0FBRyxDQUFDLEdBQUcsQ0FBQyxDQUFDO0FBNkZoRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLGlCQUFrQixTQUFRLEdBQUcsQ0FBQyxTQUFTO0lBV2hELDJFQUEyRTtJQUMzRSwwREFBMEQ7SUFDMUQsWUFBWSxLQUFvQixFQUFFLEVBQVUsRUFBRSxLQUE2Qjs7UUFDdkUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFO1lBQ3ZELE1BQU0sSUFBSSxLQUFLLENBQUMsa0VBQWtFLENBQUMsQ0FBQztTQUN2RjtRQUNELEtBQUssTUFBTSxJQUFJLElBQUksQ0FBQyxLQUFLLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUNqRCxJQUFJLElBQUksSUFBSSxDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRTtnQkFDbEMsTUFBTSxJQUFJLEtBQUssQ0FBQyx5RUFBeUUsQ0FBQyxDQUFDO2FBQzlGO1NBQ0o7UUFDRCxLQUFLLE1BQU0sSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsRUFBRTtZQUNqRSxVQUFJLElBQUksYUFBSixJQUFJLHVCQUFKLElBQUksQ0FBRSxrQkFBa0IsMENBQUUsWUFBWSxFQUFFO2dCQUN4QyxpQkFBaUIsQ0FBQyx3QkFBd0IsQ0FBQyxDQUFDLElBQUksQ0FBQyxFQUFFLGlDQUFpQyxDQUFDLENBQUM7YUFDekY7U0FDSjtRQUNELElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO1FBQ25CLE1BQU0sUUFBUSxHQUFHLElBQUksTUFBTSxDQUFDLGlCQUFpQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDNUQsSUFBSSxFQUFFLE1BQU0sQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFNBQVMsQ0FBQyxDQUFDO1lBQzVELE9BQU8sRUFBRSxNQUFNLENBQUMsT0FBTyxDQUFDLFdBQVc7WUFDbkMsT0FBTyxFQUFFLGVBQWU7WUFDeEIsSUFBSSxFQUFFLHNDQUFzQztZQUM1QyxhQUFhLEVBQUUsS0FBSztZQUNwQixPQUFPLEVBQUUsS0FBSyxDQUFDLE9BQU8sSUFBSSxHQUFHLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUM7WUFDakQsSUFBSSxFQUFFLEtBQUssQ0FBQyxJQUFJO1lBQ2hCLFlBQVksRUFBRSxLQUFLLENBQUMsWUFBWTtTQUNuQyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsY0FBYyxHQUFHLFFBQVEsQ0FBQyxjQUFjLENBQUM7UUFDOUMsSUFBSSxLQUFLLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3RDLDZDQUE2QztZQUM3QyxLQUFLLE1BQU0sU0FBUyxJQUFJLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFO2dCQUM3QyxRQUFRLENBQUMsZUFBZSxDQUFDLFNBQVMsQ0FBQyxDQUFDO2FBQ3ZDO1NBQ0o7YUFDSTtZQUNELHVDQUF1QztZQUN2QyxLQUFLLE1BQU0sSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUMsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRLENBQUMsRUFBRTtnQkFDakUsSUFBSSxJQUFJLEVBQUU7b0JBQ04sUUFBUSxDQUFDLGVBQWUsQ0FBQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7d0JBQzdDLE9BQU8sRUFBRSxDQUFDLGlCQUFpQixDQUFDLElBQUksQ0FBQyxPQUFPLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxDQUFDO3dCQUN2RCxTQUFTLEVBQUUsS0FBSyxDQUFDLE1BQU0sQ0FBQyxTQUFTO3FCQUNwQyxDQUFDLENBQUMsQ0FBQztpQkFDUDthQUNKO1NBQ0o7UUFDRCxNQUFNLE1BQU0sR0FBRyxLQUFLLENBQUMsUUFBUSxJQUFJLEtBQUssQ0FBQyxRQUFRLENBQUM7UUFDaEQsSUFBSSxDQUFDLGNBQWMsR0FBRyxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLFVBQVUsRUFBRTtZQUMzRCxZQUFZLEVBQUUsS0FBSyxDQUFDLFlBQVksSUFBSSxhQUFhO1lBQ2pELFlBQVksRUFBRSxRQUFRLENBQUMsV0FBVztZQUNsQyxvQkFBb0IsRUFBRSxJQUFJO1lBQzFCLFVBQVUsRUFBRTtnQkFDUixNQUFNLEVBQUUsTUFBTSxJQUFJLGNBQWMsQ0FBQyxNQUFNLENBQUM7Z0JBQ3hDLE1BQU0sRUFBRSxLQUFLLENBQUMsUUFBUSxJQUFJLGNBQWMsQ0FBQyxLQUFLLENBQUMsUUFBUSxDQUFDO2dCQUN4RCxNQUFNLEVBQUUsS0FBSyxDQUFDLFFBQVEsSUFBSSxjQUFjLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQzthQUMzRDtTQUNKLENBQUMsQ0FBQztJQUNQLENBQUM7SUFuRU8sTUFBTSxDQUFDLHdCQUF3QixDQUFDLFFBQXVDLEVBQUUsTUFBYztRQUMzRixLQUFLLE1BQU0sSUFBSSxJQUFJLFFBQVEsRUFBRTtZQUN6QixJQUFJLElBQUksYUFBSixJQUFJLHVCQUFKLElBQUksQ0FBRSx3QkFBd0IsRUFBRTtnQkFDaEMsTUFBTSxJQUFJLEtBQUssQ0FBQyxLQUFLLE1BQU0sSUFBSSxHQUFHLDBEQUEwRCxDQUFDLENBQUM7YUFDakc7U0FDSjtJQUNMLENBQUM7SUE4REQ7Ozs7Ozs7Ozs7Ozs7T0FhRztJQUNJLHlCQUF5QixDQUFDLFFBQWdCO1FBQzdDLGlCQUFpQixDQUFDLHdCQUF3QixDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsRUFBRSxTQUFTLENBQUMsQ0FBQztRQUNsRyxPQUFPLElBQUksQ0FBQyxjQUFjLENBQUMsTUFBTSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ2hELENBQUM7SUFDRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksZ0JBQWdCLENBQUMsUUFBZ0I7UUFDcEMsaUJBQWlCLENBQUMsd0JBQXdCLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxFQUFFLGVBQWUsQ0FBQyxDQUFDO1FBQ3hHLE9BQU8sSUFBSSxDQUFDLGNBQWMsQ0FBQyxZQUFZLENBQUMsUUFBUSxDQUFDLENBQUM7SUFDdEQsQ0FBQztDQUNKO0FBdEdELDhDQXNHQztBQUNEOzs7OztHQUtHO0FBQ0gsU0FBUyxpQkFBaUIsQ0FBQyxPQUFlLEVBQUUsTUFBYztJQUN0RCxNQUFNLEdBQUcsR0FBRyxPQUFPLENBQUMsV0FBVyxFQUFFLENBQUM7SUFDbEMsTUFBTSxVQUFVLEdBQUcsQ0FBQyxjQUFjLENBQUMsR0FBRyxDQUFDLElBQUksY0FBYyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxJQUFJLEdBQUcsQ0FBQztJQUM5RSxNQUFNLFNBQVMsR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsRUFBRSxHQUFHLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDbkUsT0FBTyxHQUFHLFVBQVUsSUFBSSxTQUFTLEVBQUUsQ0FBQztBQUN4QyxDQUFDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFTLGNBQWMsQ0FBQyxNQUFjO0lBQ2xDLE9BQU8sSUFBSSxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsRUFBRSxFQUFFO1FBQ2hELFFBQVEsQ0FBQyxFQUFFO1lBQ1AsS0FBSyxJQUFJO2dCQUNMLE9BQU8sY0FBYyxDQUFDO1lBQzFCLEtBQUssS0FBSztnQkFDTixPQUFPLGVBQWUsQ0FBQztZQUMzQjtnQkFDSSxPQUFPLENBQUMsQ0FBQztTQUNoQjtJQUNMLENBQUMsQ0FBQyxDQUFDO0FBQ1AsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFwiLi4vLi4vLi4vYXdzLWxhbWJkYVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSdcbmltcG9ydCAqIGFzIGxvZ3MgZnJvbSBcIi4uLy4uLy4uL2F3cy1sb2dzXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtbG9ncydcbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCAqIGFzIGZzIGZyb20gJ2ZzJztcbmltcG9ydCAqIGFzIHBhdGggZnJvbSAncGF0aCc7XG4vLyBkb24ndCB1c2UgXCJyZXF1aXJlXCIgc2luY2UgdGhlIHR5cGVzY3JpcHQgY29tcGlsZXIgZW1pdHMgZXJyb3JzIHNpbmNlIHRoaXNcbi8vIGZpbGUgaXMgbm90IGxpc3RlZCBpbiB0c2NvbmZpZy5qc29uLlxuY29uc3QgbWV0YWRhdGEgPSBKU09OLnBhcnNlKGZzLnJlYWRGaWxlU3luYyhwYXRoLmpvaW4oX19kaXJuYW1lLCAnc2RrLWFwaS1tZXRhZGF0YS5qc29uJyksICd1dGYtOCcpKTtcbi8qKlxuICogQVdTIFNESyBzZXJ2aWNlIG1ldGFkYXRhLlxuICovXG5leHBvcnQgdHlwZSBBd3NTZGtNZXRhZGF0YSA9IHtcbiAgICBba2V5OiBzdHJpbmddOiBhbnk7XG59O1xuY29uc3QgYXdzU2RrTWV0YWRhdGE6IEF3c1Nka01ldGFkYXRhID0gbWV0YWRhdGE7XG4vKipcbiAqIFBoeXNpY2FsIElEIG9mIHRoZSBjdXN0b20gcmVzb3VyY2UuXG4gKi9cbmV4cG9ydCBjbGFzcyBQaHlzaWNhbFJlc291cmNlSWQge1xuICAgIC8qKlxuICAgICAqIEV4dHJhY3QgdGhlIHBoeXNpY2FsIHJlc291cmNlIGlkIGZyb20gdGhlIHBhdGggKGRvdCBub3RhdGlvbikgdG8gdGhlIGRhdGEgaW4gdGhlIEFQSSBjYWxsIHJlc3BvbnNlLlxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVJlc3BvbnNlKHJlc3BvbnNlUGF0aDogc3RyaW5nKTogUGh5c2ljYWxSZXNvdXJjZUlkIHtcbiAgICAgICAgcmV0dXJuIG5ldyBQaHlzaWNhbFJlc291cmNlSWQocmVzcG9uc2VQYXRoLCB1bmRlZmluZWQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBFeHBsaWNpdCBwaHlzaWNhbCByZXNvdXJjZSBpZC5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIG9mKGlkOiBzdHJpbmcpOiBQaHlzaWNhbFJlc291cmNlSWQge1xuICAgICAgICByZXR1cm4gbmV3IFBoeXNpY2FsUmVzb3VyY2VJZCh1bmRlZmluZWQsIGlkKTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQHBhcmFtIHJlc3BvbnNlUGF0aCBQYXRoIHRvIGEgcmVzcG9uc2UgZGF0YSBlbGVtZW50IHRvIGJlIHVzZWQgYXMgdGhlIHBoeXNpY2FsIGlkLlxuICAgICAqIEBwYXJhbSBpZCBMaXRlcmFsIHN0cmluZyB0byBiZSB1c2VkIGFzIHRoZSBwaHlzaWNhbCBpZC5cbiAgICAgKi9cbiAgICBwcml2YXRlIGNvbnN0cnVjdG9yKHB1YmxpYyByZWFkb25seSByZXNwb25zZVBhdGg/OiBzdHJpbmcsIHB1YmxpYyByZWFkb25seSBpZD86IHN0cmluZykgeyB9XG59XG4vKipcbiAqIEFuIEFXUyBTREsgY2FsbC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBd3NTZGtDYWxsIHtcbiAgICAvKipcbiAgICAgKiBUaGUgc2VydmljZSB0byBjYWxsXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NKYXZhU2NyaXB0U0RLL2xhdGVzdC9pbmRleC5odG1sXG4gICAgICovXG4gICAgcmVhZG9ubHkgc2VydmljZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBzZXJ2aWNlIGFjdGlvbiB0byBjYWxsXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NKYXZhU2NyaXB0U0RLL2xhdGVzdC9pbmRleC5odG1sXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWN0aW9uOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIHBhcmFtZXRlcnMgZm9yIHRoZSBzZXJ2aWNlIGFjdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBubyBwYXJhbXRlcnNcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BV1NKYXZhU2NyaXB0U0RLL2xhdGVzdC9pbmRleC5odG1sXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGFyYW1ldGVycz86IGFueTtcbiAgICAvKipcbiAgICAgKiBUaGUgcGh5c2ljYWwgcmVzb3VyY2UgaWQgb2YgdGhlIGN1c3RvbSByZXNvdXJjZSBmb3IgdGhpcyBjYWxsLlxuICAgICAqIE1hbmRhdG9yeSBmb3Igb25DcmVhdGUgb3Igb25VcGRhdGUgY2FsbHMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG5vIHBoeXNpY2FsIHJlc291cmNlIGlkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGh5c2ljYWxSZXNvdXJjZUlkPzogUGh5c2ljYWxSZXNvdXJjZUlkO1xuICAgIC8qKlxuICAgICAqIFRoZSByZWdleCBwYXR0ZXJuIHRvIHVzZSB0byBjYXRjaCBBUEkgZXJyb3JzLiBUaGUgYGNvZGVgIHByb3BlcnR5IG9mIHRoZVxuICAgICAqIGBFcnJvcmAgb2JqZWN0IHdpbGwgYmUgdGVzdGVkIGFnYWluc3QgdGhpcyBwYXR0ZXJuLiBJZiB0aGVyZSBpcyBhIG1hdGNoIGFuXG4gICAgICogZXJyb3Igd2lsbCBub3QgYmUgdGhyb3duLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBkbyBub3QgY2F0Y2ggZXJyb3JzXG4gICAgICovXG4gICAgcmVhZG9ubHkgaWdub3JlRXJyb3JDb2Rlc01hdGNoaW5nPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEFQSSB2ZXJzaW9uIHRvIHVzZSBmb3IgdGhlIHNlcnZpY2VcbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3Nkay1mb3ItamF2YXNjcmlwdC92Mi9kZXZlbG9wZXItZ3VpZGUvbG9ja2luZy1hcGktdmVyc2lvbnMuaHRtbFxuICAgICAqIEBkZWZhdWx0IC0gdXNlIGxhdGVzdCBhdmFpbGFibGUgQVBJIHZlcnNpb25cbiAgICAgKi9cbiAgICByZWFkb25seSBhcGlWZXJzaW9uPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSByZWdpb24gdG8gc2VuZCBzZXJ2aWNlIHJlcXVlc3RzIHRvLlxuICAgICAqICoqTm90ZTogQ3Jvc3MtcmVnaW9uIG9wZXJhdGlvbnMgYXJlIGdlbmVyYWxseSBjb25zaWRlcmVkIGFuIGFudGktcGF0dGVybi4qKlxuICAgICAqICoqQ29uc2lkZXIgZmlyc3QgZGVwbG95aW5nIGEgc3RhY2sgaW4gdGhhdCByZWdpb24uKipcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gdGhlIHJlZ2lvbiB3aGVyZSB0aGlzIGN1c3RvbSByZXNvdXJjZSBpcyBkZXBsb3llZFxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlZ2lvbj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBSZXN0cmljdCB0aGUgZGF0YSByZXR1cm5lZCBieSB0aGUgY3VzdG9tIHJlc291cmNlIHRvIGEgc3BlY2lmaWMgcGF0aCBpblxuICAgICAqIHRoZSBBUEkgcmVzcG9uc2UuIFVzZSB0aGlzIHRvIGxpbWl0IHRoZSBkYXRhIHJldHVybmVkIGJ5IHRoZSBjdXN0b21cbiAgICAgKiByZXNvdXJjZSBpZiB3b3JraW5nIHdpdGggQVBJIGNhbGxzIHRoYXQgY291bGQgcG90ZW50aWFsbHkgcmVzdWx0IGluIGN1c3RvbVxuICAgICAqIHJlc3BvbnNlIG9iamVjdHMgZXhjZWVkaW5nIHRoZSBoYXJkIGxpbWl0IG9mIDQwOTYgYnl0ZXMuXG4gICAgICpcbiAgICAgKiBFeGFtcGxlIGZvciBFQ1MgLyB1cGRhdGVTZXJ2aWNlOiAnc2VydmljZS5kZXBsb3ltZW50Q29uZmlndXJhdGlvbi5tYXhpbXVtUGVyY2VudCdcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gcmV0dXJuIGFsbCBkYXRhXG4gICAgICovXG4gICAgcmVhZG9ubHkgb3V0cHV0UGF0aD86IHN0cmluZztcbn1cbi8qKlxuICogT3B0aW9ucyBmb3IgdGhlIGF1dG8tZ2VuZXJhdGlvbiBvZiBwb2xpY2llcyBiYXNlZCBvbiB0aGUgY29uZmlndXJlZCBTREsgY2FsbHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2RrQ2FsbHNQb2xpY3lPcHRpb25zIHtcbiAgICAvKipcbiAgICAgKiBUaGUgcmVzb3VyY2VzIHRoYXQgdGhlIGNhbGxzIHdpbGwgaGF2ZSBhY2Nlc3MgdG8uXG4gICAgICpcbiAgICAgKiBJdCBpcyBiZXN0IHRvIHVzZSBzcGVjaWZpYyByZXNvdXJjZSBBUk4ncyB3aGVuIHBvc3NpYmxlLiBIb3dldmVyLCB5b3UgY2FuIGFsc28gdXNlIGBBd3NDdXN0b21SZXNvdXJjZVBvbGljeS5BTllfUkVTT1VSQ0VgXG4gICAgICogdG8gYWxsb3cgYWNjZXNzIHRvIGFsbCByZXNvdXJjZXMuIEZvciBleGFtcGxlLCB3aGVuIGBvbkNyZWF0ZWAgaXMgdXNlZCB0byBjcmVhdGUgYSByZXNvdXJjZSB3aGljaCB5b3UgZG9uJ3RcbiAgICAgKiBrbm93IHRoZSBwaHlzaWNhbCBuYW1lIG9mIGluIGFkdmFuY2UuXG4gICAgICpcbiAgICAgKiBOb3RlIHRoYXQgd2lsbCBhcHBseSB0byBBTEwgU0RLIGNhbGxzLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJlc291cmNlczogc3RyaW5nW107XG59XG4vKipcbiAqIFRoZSBJQU0gUG9saWN5IHRoYXQgd2lsbCBiZSBhcHBsaWVkIHRvIHRoZSBkaWZmZXJlbnQgY2FsbHMuXG4gKi9cbmV4cG9ydCBjbGFzcyBBd3NDdXN0b21SZXNvdXJjZVBvbGljeSB7XG4gICAgLyoqXG4gICAgICogVXNlIHRoaXMgY29uc3RhbnQgdG8gY29uZmlndXJlIGFjY2VzcyB0byBhbnkgcmVzb3VyY2UuXG4gICAgICovXG4gICAgcHVibGljIHN0YXRpYyByZWFkb25seSBBTllfUkVTT1VSQ0UgPSBbJyonXTtcbiAgICAvKipcbiAgICAgKiBFeHBsaWNpdCBJQU0gUG9saWN5IFN0YXRlbWVudHMuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gc3RhdGVtZW50cyB0aGUgc3RhdGVtZW50cyB0byBwcm9wYWdhdGUgdG8gdGhlIFNESyBjYWxscy5cbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21TdGF0ZW1lbnRzKHN0YXRlbWVudHM6IGlhbS5Qb2xpY3lTdGF0ZW1lbnRbXSkge1xuICAgICAgICByZXR1cm4gbmV3IEF3c0N1c3RvbVJlc291cmNlUG9saWN5KHN0YXRlbWVudHMsIHVuZGVmaW5lZCk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEdlbmVyYXRlIElBTSBQb2xpY3kgU3RhdGVtZW50cyBmcm9tIHRoZSBjb25maWd1cmVkIFNESyBjYWxscy5cbiAgICAgKlxuICAgICAqIEVhY2ggU0RLIGNhbGwgd2l0aCBiZSB0cmFuc2xhdGVkIHRvIGFuIElBTSBQb2xpY3kgU3RhdGVtZW50IGluIHRoZSBmb3JtIG9mOiBgY2FsbC5zZXJ2aWNlOmNhbGwuYWN0aW9uYCAoZS5nIGBzMzpQdXRPYmplY3RgKS5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBvcHRpb25zIG9wdGlvbnMgZm9yIHRoZSBwb2xpY3kgZ2VuZXJhdGlvblxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVNka0NhbGxzKG9wdGlvbnM6IFNka0NhbGxzUG9saWN5T3B0aW9ucykge1xuICAgICAgICByZXR1cm4gbmV3IEF3c0N1c3RvbVJlc291cmNlUG9saWN5KFtdLCBvcHRpb25zLnJlc291cmNlcyk7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEBwYXJhbSBzdGF0ZW1lbnRzIHN0YXRlbWVudHMgZm9yIGV4cGxpY2l0IHBvbGljeS5cbiAgICAgKiBAcGFyYW0gcmVzb3VyY2VzIHJlc291cmNlcyBmb3IgYXV0by1nZW5lcmF0ZWQgZnJvbSBTREsgY2FsbHMuXG4gICAgICovXG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgc3RhdGVtZW50czogaWFtLlBvbGljeVN0YXRlbWVudFtdLCBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VzPzogc3RyaW5nW10pIHsgfVxufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIGZvciBBd3NDdXN0b21SZXNvdXJjZS5cbiAqXG4gKiBOb3RlIHRoYXQgYXQgbGVhc3Qgb25DcmVhdGUsIG9uVXBkYXRlIG9yIG9uRGVsZXRlIG11c3QgYmUgc3BlY2lmaWVkLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF3c0N1c3RvbVJlc291cmNlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIENsb3VkZm9ybWF0aW9uIFJlc291cmNlIHR5cGUuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEN1c3RvbTo6QVdTXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmVzb3VyY2VUeXBlPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSBBV1MgU0RLIGNhbGwgdG8gbWFrZSB3aGVuIHRoZSByZXNvdXJjZSBpcyBjcmVhdGVkLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSB0aGUgY2FsbCB3aGVuIHRoZSByZXNvdXJjZSBpcyB1cGRhdGVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgb25DcmVhdGU/OiBBd3NTZGtDYWxsO1xuICAgIC8qKlxuICAgICAqIFRoZSBBV1MgU0RLIGNhbGwgdG8gbWFrZSB3aGVuIHRoZSByZXNvdXJjZSBpcyB1cGRhdGVkXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIG5vIGNhbGxcbiAgICAgKi9cbiAgICByZWFkb25seSBvblVwZGF0ZT86IEF3c1Nka0NhbGw7XG4gICAgLyoqXG4gICAgICogVGhlIEFXUyBTREsgY2FsbCB0byBtYWtlIHdoZW4gdGhlIHJlc291cmNlIGlzIGRlbGV0ZWRcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gbm8gY2FsbFxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9uRGVsZXRlPzogQXdzU2RrQ2FsbDtcbiAgICAvKipcbiAgICAgKiBUaGUgcG9saWN5IHRoYXQgd2lsbCBiZSBhZGRlZCB0byB0aGUgZXhlY3V0aW9uIHJvbGUgb2YgdGhlIExhbWJkYVxuICAgICAqIGZ1bmN0aW9uIGltcGxlbWVudGluZyB0aGlzIGN1c3RvbSByZXNvdXJjZSBwcm92aWRlci5cbiAgICAgKlxuICAgICAqIFRoZSBjdXN0b20gcmVzb3VyY2UgYWxzbyBpbXBsZW1lbnRzIGBpYW0uSUdyYW50YWJsZWAsIG1ha2luZyBpdCBwb3NzaWJsZVxuICAgICAqIHRvIHVzZSB0aGUgYGdyYW50WHh4KClgIG1ldGhvZHMuXG4gICAgICpcbiAgICAgKiBBcyB0aGlzIGN1c3RvbSByZXNvdXJjZSB1c2VzIGEgc2luZ2xldG9uIExhbWJkYSBmdW5jdGlvbiwgaXQncyBpbXBvcnRhbnRcbiAgICAgKiB0byBub3RlIHRoZSB0aGF0IGZ1bmN0aW9uJ3Mgcm9sZSB3aWxsIGV2ZW50dWFsbHkgYWNjdW11bGF0ZSB0aGVcbiAgICAgKiBwZXJtaXNzaW9ucy9ncmFudHMgZnJvbSBhbGwgcmVzb3VyY2VzLlxuICAgICAqXG4gICAgICogQHNlZSBQb2xpY3kuZnJvbVN0YXRlbWVudHNcbiAgICAgKiBAc2VlIFBvbGljeS5mcm9tU2RrQ2FsbHNcbiAgICAgKi9cbiAgICByZWFkb25seSBwb2xpY3k6IEF3c0N1c3RvbVJlc291cmNlUG9saWN5O1xuICAgIC8qKlxuICAgICAqIFRoZSBleGVjdXRpb24gcm9sZSBmb3IgdGhlIExhbWJkYSBmdW5jdGlvbiBpbXBsZW1lbnRpbmcgdGhpcyBjdXN0b21cbiAgICAgKiByZXNvdXJjZSBwcm92aWRlci4gVGhpcyByb2xlIHdpbGwgYXBwbHkgdG8gYWxsIGBBd3NDdXN0b21SZXNvdXJjZWBcbiAgICAgKiBpbnN0YW5jZXMgaW4gdGhlIHN0YWNrLiBUaGUgcm9sZSBtdXN0IGJlIGFzc3VtYWJsZSBieSB0aGVcbiAgICAgKiBgbGFtYmRhLmFtYXpvbmF3cy5jb21gIHNlcnZpY2UgcHJpbmNpcGFsLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBhIG5ldyByb2xlIGlzIGNyZWF0ZWRcbiAgICAgKi9cbiAgICByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgIC8qKlxuICAgICAqIFRoZSB0aW1lb3V0IGZvciB0aGUgTGFtYmRhIGZ1bmN0aW9uIGltcGxlbWVudGluZyB0aGlzIGN1c3RvbSByZXNvdXJjZS5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IER1cmF0aW9uLm1pbnV0ZXMoMilcbiAgICAgKi9cbiAgICByZWFkb25seSB0aW1lb3V0PzogY2RrLkR1cmF0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBudW1iZXIgb2YgZGF5cyBsb2cgZXZlbnRzIG9mIHRoZSBMYW1iZGEgZnVuY3Rpb24gaW1wbGVtZW50aW5nXG4gICAgICogdGhpcyBjdXN0b20gcmVzb3VyY2UgYXJlIGtlcHQgaW4gQ2xvdWRXYXRjaCBMb2dzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgbG9ncy5SZXRlbnRpb25EYXlzLklORklOSVRFXG4gICAgICovXG4gICAgcmVhZG9ubHkgbG9nUmV0ZW50aW9uPzogbG9ncy5SZXRlbnRpb25EYXlzO1xufVxuLyoqXG4gKiBEZWZpbmVzIGEgY3VzdG9tIHJlc291cmNlIHRoYXQgaXMgbWF0ZXJpYWxpemVkIHVzaW5nIHNwZWNpZmljIEFXUyBBUEkgY2FsbHMuXG4gKlxuICogVXNlIHRoaXMgdG8gYnJpZGdlIGFueSBnYXAgdGhhdCBtaWdodCBleGlzdCBpbiB0aGUgQ2xvdWRGb3JtYXRpb24gQ292ZXJhZ2UuXG4gKiBZb3UgY2FuIHNwZWNpZnkgZXhhY3RseSB3aGljaCBjYWxscyBhcmUgaW52b2tlZCBmb3IgdGhlICdDUkVBVEUnLCAnVVBEQVRFJyBhbmQgJ0RFTEVURScgbGlmZSBjeWNsZSBldmVudHMuXG4gKlxuICovXG5leHBvcnQgY2xhc3MgQXdzQ3VzdG9tUmVzb3VyY2UgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IGltcGxlbWVudHMgaWFtLklHcmFudGFibGUge1xuICAgIHByaXZhdGUgc3RhdGljIGJyZWFrSWdub3JlRXJyb3JzQ2lyY3VpdChzZGtDYWxsczogQXJyYXk8QXdzU2RrQ2FsbCB8IHVuZGVmaW5lZD4sIGNhbGxlcjogc3RyaW5nKSB7XG4gICAgICAgIGZvciAoY29uc3QgY2FsbCBvZiBzZGtDYWxscykge1xuICAgICAgICAgICAgaWYgKGNhbGw/Lmlnbm9yZUVycm9yQ29kZXNNYXRjaGluZykge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcihgXFxgJHtjYWxsZXJ9XFxgYCArICcgY2Fubm90IGJlIGNhbGxlZCBhbG9uZyB3aXRoIGBpZ25vcmVFcnJvckNvZGVzTWF0Y2hpbmdgLicpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgfVxuICAgIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbDogaWFtLklQcmluY2lwYWw7XG4gICAgcHJpdmF0ZSByZWFkb25seSBjdXN0b21SZXNvdXJjZTogY2RrLkN1c3RvbVJlc291cmNlO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IEF3c0N1c3RvbVJlc291cmNlUHJvcHM7XG4gICAgLy8gJ3Byb3BzJyBjYW5ub3QgYmUgb3B0aW9uYWwsIGV2ZW4gdGhvdWdoIGFsbCBpdHMgcHJvcGVydGllcyBhcmUgb3B0aW9uYWwuXG4gICAgLy8gdGhpcyBpcyBiZWNhdXNlIGF0IGxlYXN0IG9uZSBzZGsgY2FsbCBtdXN0IGJlIHByb3ZpZGVkLlxuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXdzQ3VzdG9tUmVzb3VyY2VQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICBpZiAoIXByb3BzLm9uQ3JlYXRlICYmICFwcm9wcy5vblVwZGF0ZSAmJiAhcHJvcHMub25EZWxldGUpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3QgYG9uQ3JlYXRlYCwgYG9uVXBkYXRlYCBvciBgb25EZWxldGVgIG11c3QgYmUgc3BlY2lmaWVkLicpO1xuICAgICAgICB9XG4gICAgICAgIGZvciAoY29uc3QgY2FsbCBvZiBbcHJvcHMub25DcmVhdGUsIHByb3BzLm9uVXBkYXRlXSkge1xuICAgICAgICAgICAgaWYgKGNhbGwgJiYgIWNhbGwucGh5c2ljYWxSZXNvdXJjZUlkKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdgcGh5c2ljYWxSZXNvdXJjZUlkYCBtdXN0IGJlIHNwZWNpZmllZCBmb3Igb25DcmVhdGUgYW5kIG9uVXBkYXRlIGNhbGxzLicpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGZvciAoY29uc3QgY2FsbCBvZiBbcHJvcHMub25DcmVhdGUsIHByb3BzLm9uVXBkYXRlLCBwcm9wcy5vbkRlbGV0ZV0pIHtcbiAgICAgICAgICAgIGlmIChjYWxsPy5waHlzaWNhbFJlc291cmNlSWQ/LnJlc3BvbnNlUGF0aCkge1xuICAgICAgICAgICAgICAgIEF3c0N1c3RvbVJlc291cmNlLmJyZWFrSWdub3JlRXJyb3JzQ2lyY3VpdChbY2FsbF0sICdQaHlzaWNhbFJlc291cmNlSWQuZnJvbVJlc3BvbnNlJyk7XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy5wcm9wcyA9IHByb3BzO1xuICAgICAgICBjb25zdCBwcm92aWRlciA9IG5ldyBsYW1iZGEuU2luZ2xldG9uRnVuY3Rpb24odGhpcywgJ1Byb3ZpZGVyJywge1xuICAgICAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KHBhdGguam9pbihfX2Rpcm5hbWUsICdydW50aW1lJykpLFxuICAgICAgICAgICAgcnVudGltZTogbGFtYmRhLlJ1bnRpbWUuTk9ERUpTXzEyX1gsXG4gICAgICAgICAgICBoYW5kbGVyOiAnaW5kZXguaGFuZGxlcicsXG4gICAgICAgICAgICB1dWlkOiAnNjc5ZjUzZmEtYzAwMi00MzBjLWIwZGEtNWI3OTgyYmQyMjg3JyxcbiAgICAgICAgICAgIGxhbWJkYVB1cnBvc2U6ICdBV1MnLFxuICAgICAgICAgICAgdGltZW91dDogcHJvcHMudGltZW91dCB8fCBjZGsuRHVyYXRpb24ubWludXRlcygyKSxcbiAgICAgICAgICAgIHJvbGU6IHByb3BzLnJvbGUsXG4gICAgICAgICAgICBsb2dSZXRlbnRpb246IHByb3BzLmxvZ1JldGVudGlvbixcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuZ3JhbnRQcmluY2lwYWwgPSBwcm92aWRlci5ncmFudFByaW5jaXBhbDtcbiAgICAgICAgaWYgKHByb3BzLnBvbGljeS5zdGF0ZW1lbnRzLmxlbmd0aCAhPT0gMCkge1xuICAgICAgICAgICAgLy8gVXNlIGN1c3RvbSBzdGF0ZW1lbnRzIHByb3ZpZGVkIGJ5IHRoZSB1c2VyXG4gICAgICAgICAgICBmb3IgKGNvbnN0IHN0YXRlbWVudCBvZiBwcm9wcy5wb2xpY3kuc3RhdGVtZW50cykge1xuICAgICAgICAgICAgICAgIHByb3ZpZGVyLmFkZFRvUm9sZVBvbGljeShzdGF0ZW1lbnQpO1xuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIGVsc2Uge1xuICAgICAgICAgICAgLy8gRGVyaXZlIHN0YXRlbWVudHMgZnJvbSBBV1MgU0RLIGNhbGxzXG4gICAgICAgICAgICBmb3IgKGNvbnN0IGNhbGwgb2YgW3Byb3BzLm9uQ3JlYXRlLCBwcm9wcy5vblVwZGF0ZSwgcHJvcHMub25EZWxldGVdKSB7XG4gICAgICAgICAgICAgICAgaWYgKGNhbGwpIHtcbiAgICAgICAgICAgICAgICAgICAgcHJvdmlkZXIuYWRkVG9Sb2xlUG9saWN5KG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGFjdGlvbnM6IFthd3NTZGtUb0lhbUFjdGlvbihjYWxsLnNlcnZpY2UsIGNhbGwuYWN0aW9uKV0sXG4gICAgICAgICAgICAgICAgICAgICAgICByZXNvdXJjZXM6IHByb3BzLnBvbGljeS5yZXNvdXJjZXMsXG4gICAgICAgICAgICAgICAgICAgIH0pKTtcbiAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICB9XG4gICAgICAgIH1cbiAgICAgICAgY29uc3QgY3JlYXRlID0gcHJvcHMub25DcmVhdGUgfHwgcHJvcHMub25VcGRhdGU7XG4gICAgICAgIHRoaXMuY3VzdG9tUmVzb3VyY2UgPSBuZXcgY2RrLkN1c3RvbVJlc291cmNlKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIHJlc291cmNlVHlwZTogcHJvcHMucmVzb3VyY2VUeXBlIHx8ICdDdXN0b206OkFXUycsXG4gICAgICAgICAgICBzZXJ2aWNlVG9rZW46IHByb3ZpZGVyLmZ1bmN0aW9uQXJuLFxuICAgICAgICAgICAgcGFzY2FsQ2FzZVByb3BlcnRpZXM6IHRydWUsXG4gICAgICAgICAgICBwcm9wZXJ0aWVzOiB7XG4gICAgICAgICAgICAgICAgY3JlYXRlOiBjcmVhdGUgJiYgZW5jb2RlQm9vbGVhbnMoY3JlYXRlKSxcbiAgICAgICAgICAgICAgICB1cGRhdGU6IHByb3BzLm9uVXBkYXRlICYmIGVuY29kZUJvb2xlYW5zKHByb3BzLm9uVXBkYXRlKSxcbiAgICAgICAgICAgICAgICBkZWxldGU6IHByb3BzLm9uRGVsZXRlICYmIGVuY29kZUJvb2xlYW5zKHByb3BzLm9uRGVsZXRlKSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIHJlc3BvbnNlIGRhdGEgZm9yIHRoZSBBV1MgU0RLIGNhbGwuXG4gICAgICpcbiAgICAgKiBFeGFtcGxlIGZvciBTMyAvIGxpc3RCdWNrZXQgOiAnQnVja2V0cy4wLk5hbWUnXG4gICAgICpcbiAgICAgKiBVc2UgYFRva2VuLmFzWHh4YCB0byBlbmNvZGUgdGhlIHJldHVybmVkIGBSZWZlcmVuY2VgIGFzIGEgc3BlY2lmaWMgdHlwZSBvclxuICAgICAqIHVzZSB0aGUgY29udmVuaWVuY2UgYGdldERhdGFTdHJpbmdgIGZvciBzdHJpbmcgYXR0cmlidXRlcy5cbiAgICAgKlxuICAgICAqIE5vdGUgdGhhdCB5b3UgY2Fubm90IHVzZSB0aGlzIG1ldGhvZCBpZiBgaWdub3JlRXJyb3JDb2Rlc01hdGNoaW5nYFxuICAgICAqIGlzIGNvbmZpZ3VyZWQgZm9yIGFueSBvZiB0aGUgU0RLIGNhbGxzLiBUaGlzIGlzIGJlY2F1c2UgaW4gc3VjaCBhIGNhc2UsXG4gICAgICogdGhlIHJlc3BvbnNlIGRhdGEgbWlnaHQgbm90IGV4aXN0LCBhbmQgd2lsbCBjYXVzZSBhIENsb3VkRm9ybWF0aW9uIGRlcGxveSB0aW1lIGVycm9yLlxuICAgICAqXG4gICAgICogQHBhcmFtIGRhdGFQYXRoIHRoZSBwYXRoIHRvIHRoZSBkYXRhXG4gICAgICovXG4gICAgcHVibGljIGdldFJlc3BvbnNlRmllbGRSZWZlcmVuY2UoZGF0YVBhdGg6IHN0cmluZykge1xuICAgICAgICBBd3NDdXN0b21SZXNvdXJjZS5icmVha0lnbm9yZUVycm9yc0NpcmN1aXQoW3RoaXMucHJvcHMub25DcmVhdGUsIHRoaXMucHJvcHMub25VcGRhdGVdLCAnZ2V0RGF0YScpO1xuICAgICAgICByZXR1cm4gdGhpcy5jdXN0b21SZXNvdXJjZS5nZXRBdHQoZGF0YVBhdGgpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXR1cm5zIHJlc3BvbnNlIGRhdGEgZm9yIHRoZSBBV1MgU0RLIGNhbGwgYXMgc3RyaW5nLlxuICAgICAqXG4gICAgICogRXhhbXBsZSBmb3IgUzMgLyBsaXN0QnVja2V0IDogJ0J1Y2tldHMuMC5OYW1lJ1xuICAgICAqXG4gICAgICogTm90ZSB0aGF0IHlvdSBjYW5ub3QgdXNlIHRoaXMgbWV0aG9kIGlmIGBpZ25vcmVFcnJvckNvZGVzTWF0Y2hpbmdgXG4gICAgICogaXMgY29uZmlndXJlZCBmb3IgYW55IG9mIHRoZSBTREsgY2FsbHMuIFRoaXMgaXMgYmVjYXVzZSBpbiBzdWNoIGEgY2FzZSxcbiAgICAgKiB0aGUgcmVzcG9uc2UgZGF0YSBtaWdodCBub3QgZXhpc3QsIGFuZCB3aWxsIGNhdXNlIGEgQ2xvdWRGb3JtYXRpb24gZGVwbG95IHRpbWUgZXJyb3IuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gZGF0YVBhdGggdGhlIHBhdGggdG8gdGhlIGRhdGFcbiAgICAgKi9cbiAgICBwdWJsaWMgZ2V0UmVzcG9uc2VGaWVsZChkYXRhUGF0aDogc3RyaW5nKTogc3RyaW5nIHtcbiAgICAgICAgQXdzQ3VzdG9tUmVzb3VyY2UuYnJlYWtJZ25vcmVFcnJvcnNDaXJjdWl0KFt0aGlzLnByb3BzLm9uQ3JlYXRlLCB0aGlzLnByb3BzLm9uVXBkYXRlXSwgJ2dldERhdGFTdHJpbmcnKTtcbiAgICAgICAgcmV0dXJuIHRoaXMuY3VzdG9tUmVzb3VyY2UuZ2V0QXR0U3RyaW5nKGRhdGFQYXRoKTtcbiAgICB9XG59XG4vKipcbiAqIFRyYW5zZm9ybSBTREsgc2VydmljZS9hY3Rpb24gdG8gSUFNIGFjdGlvbiB1c2luZyBtZXRhZGF0YSBmcm9tIGF3cy1zZGsgbW9kdWxlLlxuICogRXhhbXBsZTogQ2xvdWRXYXRjaExvZ3Mgd2l0aCBwdXRSZXRlbnRpb25Qb2xpY3kgPT4gbG9nczpQdXRSZXRlbnRpb25Qb2xpY3lcbiAqXG4gKiBUT0RPOiBpcyB0aGlzIG1hcHBpbmcgY29ycmVjdCBmb3IgYWxsIHNlcnZpY2VzP1xuICovXG5mdW5jdGlvbiBhd3NTZGtUb0lhbUFjdGlvbihzZXJ2aWNlOiBzdHJpbmcsIGFjdGlvbjogc3RyaW5nKTogc3RyaW5nIHtcbiAgICBjb25zdCBzcnYgPSBzZXJ2aWNlLnRvTG93ZXJDYXNlKCk7XG4gICAgY29uc3QgaWFtU2VydmljZSA9IChhd3NTZGtNZXRhZGF0YVtzcnZdICYmIGF3c1Nka01ldGFkYXRhW3Nydl0ucHJlZml4KSB8fCBzcnY7XG4gICAgY29uc3QgaWFtQWN0aW9uID0gYWN0aW9uLmNoYXJBdCgwKS50b1VwcGVyQ2FzZSgpICsgYWN0aW9uLnNsaWNlKDEpO1xuICAgIHJldHVybiBgJHtpYW1TZXJ2aWNlfToke2lhbUFjdGlvbn1gO1xufVxuLyoqXG4gKiBFbmNvZGVzIGJvb2xlYW5zIGFzIHNwZWNpYWwgc3RyaW5nc1xuICovXG5mdW5jdGlvbiBlbmNvZGVCb29sZWFucyhvYmplY3Q6IG9iamVjdCkge1xuICAgIHJldHVybiBKU09OLnBhcnNlKEpTT04uc3RyaW5naWZ5KG9iamVjdCksIChfaywgdikgPT4ge1xuICAgICAgICBzd2l0Y2ggKHYpIHtcbiAgICAgICAgICAgIGNhc2UgdHJ1ZTpcbiAgICAgICAgICAgICAgICByZXR1cm4gJ1RSVUU6Qk9PTEVBTic7XG4gICAgICAgICAgICBjYXNlIGZhbHNlOlxuICAgICAgICAgICAgICAgIHJldHVybiAnRkFMU0U6Qk9PTEVBTic7XG4gICAgICAgICAgICBkZWZhdWx0OlxuICAgICAgICAgICAgICAgIHJldHVybiB2O1xuICAgICAgICB9XG4gICAgfSk7XG59XG4iXX0=