"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const intrinsic_1 = require("./private/intrinsic");
const stack_1 = require("./stack");
/**
 * Given an object, converts all keys to PascalCase given they are currently in camel case.
 * @param obj The object.
 */
function capitalizePropertyNames(construct, obj) {
    const stack = stack_1.Stack.of(construct);
    obj = stack.resolve(obj);
    if (typeof (obj) !== 'object') {
        return obj;
    }
    if (Array.isArray(obj)) {
        return obj.map(x => capitalizePropertyNames(construct, x));
    }
    const newObj = {};
    for (const key of Object.keys(obj)) {
        const value = obj[key];
        const first = key.charAt(0).toUpperCase();
        const newKey = first + key.slice(1);
        newObj[newKey] = capitalizePropertyNames(construct, value);
    }
    return newObj;
}
exports.capitalizePropertyNames = capitalizePropertyNames;
/**
 * Turns empty arrays/objects to undefined (after evaluating tokens).
 */
function ignoreEmpty(obj) {
    return new PostResolveToken(obj, o => {
        // undefined/null
        if (o == null) {
            return o;
        }
        if (Array.isArray(o) && o.length === 0) {
            return undefined;
        }
        if (typeof (o) === 'object' && Object.keys(o).length === 0) {
            return undefined;
        }
        return o;
    });
}
exports.ignoreEmpty = ignoreEmpty;
/**
 * Returns a copy of `obj` without `undefined` (or `null`) values in maps or arrays.
 */
function filterUndefined(obj) {
    if (Array.isArray(obj)) {
        return obj.filter(x => x != null).map(x => filterUndefined(x));
    }
    if (typeof (obj) === 'object') {
        const ret = {};
        for (const [key, value] of Object.entries(obj)) {
            if (value == null) {
                continue;
            }
            ret[key] = filterUndefined(value);
        }
        return ret;
    }
    return obj;
}
exports.filterUndefined = filterUndefined;
/**
 * A Token that applies a function AFTER resolve resolution
 */
class PostResolveToken extends intrinsic_1.Intrinsic {
    constructor(value, processor) {
        super(value);
        this.processor = processor;
    }
    resolve(context) {
        context.registerPostProcessor(this);
        return super.resolve(context);
    }
    postProcess(o, _context) {
        return this.processor(o);
    }
}
exports.PostResolveToken = PostResolveToken;
/**
 * @returns the list of stacks that lead from the top-level stack (non-nested) all the way to a nested stack.
 */
function pathToTopLevelStack(s) {
    if (s.nestedStackParent) {
        return [...pathToTopLevelStack(s.nestedStackParent), s];
    }
    else {
        return [s];
    }
}
exports.pathToTopLevelStack = pathToTopLevelStack;
/**
 * Given two arrays, returns the last common element or `undefined` if there
 * isn't (arrays are foriegn).
 */
function findLastCommonElement(path1, path2) {
    let i = 0;
    while (i < path1.length && i < path2.length) {
        if (path1[i] !== path2[i]) {
            break;
        }
        i++;
    }
    return path1[i - 1];
}
exports.findLastCommonElement = findLastCommonElement;
//# sourceMappingURL=data:application/json;base64,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