"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const construct_compat_1 = require("./construct-compat");
const runtime_info_1 = require("./private/runtime-info");
const synthesis_1 = require("./private/synthesis");
/**
 * An abstract application modeling unit consisting of Stacks that should be
 * deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 */
class Stage extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        if (id !== '' && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        this.parentStage = Stage.of(this);
        this.region = (_b = (_a = props.env) === null || _a === void 0 ? void 0 : _a.region) !== null && _b !== void 0 ? _b : (_c = this.parentStage) === null || _c === void 0 ? void 0 : _c.region;
        this.account = (_e = (_d = props.env) === null || _d === void 0 ? void 0 : _d.account) !== null && _e !== void 0 ? _e : (_f = this.parentStage) === null || _f === void 0 ? void 0 : _f.account;
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [(_g = this.parentStage) === null || _g === void 0 ? void 0 : _g.stageName, id].filter(x => x).join('-');
    }
    /**
     * Return the stage this construct is contained with, if available. If called
     * on a nested stage, returns its parent.
     *
     * @experimental
     */
    static of(construct) {
        return construct.node.scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * Test whether the given construct is a stage.
     *
     * @experimental
     */
    static isStage(x) {
        return x !== null && x instanceof Stage;
    }
    /**
     * Artifact ID of the assembly if it is a nested stage. The root stage (app)
     * will return an empty string.
     *
     * Derived from the construct path.
     *
     * @experimental
     */
    get artifactId() {
        if (!this.node.path) {
            return '';
        }
        return `assembly-${this.node.path.replace(/\//g, '-').replace(/^-+|-+$/g, '')}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     */
    synth(options = {}) {
        if (!this.assembly) {
            const runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? undefined : runtime_info_1.collectRuntimeInformation();
            this.assembly = synthesis_1.synthesize(this, {
                skipValidation: options.skipValidation,
                runtimeInfo,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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