"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cfn_dynamic_reference_1 = require("./cfn-dynamic-reference");
const intrinsic_1 = require("./private/intrinsic");
/**
 * Work with secret values in the CDK
 *
 * Secret values in the CDK (such as those retrieved from SecretsManager) are
 * represented as regular strings, just like other values that are only
 * available at deployment time.
 *
 * To help you avoid accidental mistakes which would lead to you putting your
 * secret values directly into a CloudFormation template, constructs that take
 * secret values will not allow you to pass in a literal secret value. They do
 * so by calling `Secret.assertSafeSecret()`.
 *
 * You can escape the check by calling `Secret.plainText()`, but doing
 * so is highly discouraged.
 */
class SecretValue extends intrinsic_1.Intrinsic {
    /**
     * Construct a literal secret value for use with secret-aware constructs
     *
     * *Do not use this method for any secrets that you care about.*
     *
     * The only reasonable use case for using this method is when you are testing.
     */
    static plainText(secret) {
        return new SecretValue(secret);
    }
    /**
     * Creates a `SecretValue` with a value which is dynamically loaded from AWS Secrets Manager.
     * @param secretId The ID or ARN of the secret
     * @param options Options
     */
    static secretsManager(secretId, options = {}) {
        if (!secretId) {
            throw new Error('secretId cannot be empty');
        }
        if (!secretId.startsWith('arn:') && secretId.includes(':')) {
            throw new Error(`secret id "${secretId}" is not an ARN but contains ":"`);
        }
        const parts = [
            secretId,
            'SecretString',
            options.jsonField || '',
            options.versionStage || '',
            options.versionId || '',
        ];
        const dyref = new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SECRETS_MANAGER, parts.join(':'));
        return this.cfnDynamicReference(dyref);
    }
    /**
     * Use a secret value stored from a Systems Manager (SSM) parameter.
     *
     * @param parameterName The name of the parameter in the Systems Manager
     * Parameter Store. The parameter name is case-sensitive.
     *
     * @param version An integer that specifies the version of the parameter to
     * use. You must specify the exact version. You cannot currently specify that
     * AWS CloudFormation use the latest version of a parameter.
     */
    static ssmSecure(parameterName, version) {
        const parts = [parameterName, version];
        return this.cfnDynamicReference(new cfn_dynamic_reference_1.CfnDynamicReference(cfn_dynamic_reference_1.CfnDynamicReferenceService.SSM_SECURE, parts.join(':')));
    }
    /**
     * Obtain the secret value through a CloudFormation dynamic reference.
     *
     * If possible, use `SecretValue.ssmSecure` or `SecretValue.secretsManager` directly.
     *
     * @param ref The dynamic reference to use.
     */
    static cfnDynamicReference(ref) {
        return new SecretValue(ref);
    }
    /**
     * Obtain the secret value through a CloudFormation parameter.
     *
     * Generally, this is not a recommended approach. AWS Secrets Manager is the
     * recommended way to reference secrets.
     *
     * @param param The CloudFormation parameter to use.
     */
    static cfnParameter(param) {
        if (!param.noEcho) {
            throw new Error('CloudFormation parameter must be configured with "NoEcho"');
        }
        return new SecretValue(param.value);
    }
}
exports.SecretValue = SecretValue;
//# sourceMappingURL=data:application/json;base64,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