"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
function identity(x) {
    return x;
}
exports.stringToCloudFormation = identity;
exports.booleanToCloudFormation = identity;
exports.objectToCloudFormation = identity;
exports.numberToCloudFormation = identity;
/**
 * The date needs to be formatted as an ISO date in UTC
 *
 * Some usage sites require a date, some require a timestamp. We'll
 * always output a timestamp and hope the parser on the other end
 * is smart enough to ignore the time part... (?)
 */
function dateToCloudFormation(x) {
    if (!x) {
        return undefined;
    }
    // tslint:disable-next-line:max-line-length
    return `${x.getUTCFullYear()}-${pad(x.getUTCMonth() + 1)}-${pad(x.getUTCDate())}T${pad(x.getUTCHours())}:${pad(x.getUTCMinutes())}:${pad(x.getUTCSeconds())}`;
}
exports.dateToCloudFormation = dateToCloudFormation;
/**
 * Pad a number to 2 decimal places
 */
function pad(x) {
    if (x < 10) {
        return '0' + x.toString();
    }
    return x.toString();
}
/**
 * Turn a tag object into the proper CloudFormation representation
 */
function cfnTagToCloudFormation(x) {
    return {
        Key: x.key,
        Value: x.value,
    };
}
exports.cfnTagToCloudFormation = cfnTagToCloudFormation;
function listMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        return x.map(elementMapper);
    };
}
exports.listMapper = listMapper;
function hashMapper(elementMapper) {
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        const ret = {};
        Object.keys(x).forEach((key) => {
            ret[key] = elementMapper(x[key]);
        });
        return ret;
    };
}
exports.hashMapper = hashMapper;
/**
 * Return a union mapper
 *
 * Takes a list of validators and a list of mappers, which should correspond pairwise.
 *
 * The mapper of the first successful validator will be called.
 */
function unionMapper(validators, mappers) {
    if (validators.length !== mappers.length) {
        throw Error('Not the same amount of validators and mappers passed to unionMapper()');
    }
    return (x) => {
        if (!canInspect(x)) {
            return x;
        }
        for (let i = 0; i < validators.length; i++) {
            if (validators[i](x).isSuccess) {
                return mappers[i](x);
            }
        }
        // Should not be possible because the union must have passed validation before this function
        // will be called, but catch it anyway.
        throw new TypeError('No validators matched in the union()');
    };
}
exports.unionMapper = unionMapper;
// ----------------------------------------------------------------------
// VALIDATORS
//
// These are used while checking that supplied property bags match the expected schema
//
// We have a couple of datatypes that model validation errors and collections of validation
// errors (together forming a tree of errors so that we can trace validation errors through
// an object graph), and validators.
//
// Validators are simply functions that take a value and return a validation results. Then
// we have some combinators to turn primitive validators into more complex validators.
//
/**
 * Representation of validation results
 *
 * Models a tree of validation errors so that we have as much information as possible
 * about the failure that occurred.
 */
class ValidationResult {
    constructor(errorMessage = '', results = new ValidationResults()) {
        this.errorMessage = errorMessage;
        this.results = results;
    }
    get isSuccess() {
        return !this.errorMessage && this.results.isSuccess;
    }
    /**
     * Turn a failed validation into an exception
     */
    assertSuccess() {
        if (!this.isSuccess) {
            let message = this.errorTree();
            // The first letter will be lowercase, so uppercase it for a nicer error message
            message = message.substr(0, 1).toUpperCase() + message.substr(1);
            throw new CfnSynthesisError(message);
        }
    }
    /**
     * Return a string rendering of the tree of validation failures
     */
    errorTree() {
        const childMessages = this.results.errorTreeList();
        return this.errorMessage + (childMessages.length ? `\n  ${childMessages.replace(/\n/g, '\n  ')}` : '');
    }
    /**
     * Wrap this result with an error message, if it concerns an error
     */
    prefix(message) {
        if (this.isSuccess) {
            return this;
        }
        return new ValidationResult(`${message}: ${this.errorMessage}`, this.results);
    }
}
exports.ValidationResult = ValidationResult;
/**
 * A collection of validation results
 */
class ValidationResults {
    constructor(results = []) {
        this.results = results;
    }
    collect(result) {
        // Only collect failures
        if (!result.isSuccess) {
            this.results.push(result);
        }
    }
    get isSuccess() {
        return this.results.every(x => x.isSuccess);
    }
    errorTreeList() {
        return this.results.map(child => child.errorTree()).join('\n');
    }
    /**
     * Wrap up all validation results into a single tree node
     *
     * If there are failures in the collection, add a message, otherwise
     * return a success.
     */
    wrap(message) {
        if (this.isSuccess) {
            return exports.VALIDATION_SUCCESS;
        }
        return new ValidationResult(message, this);
    }
}
exports.ValidationResults = ValidationResults;
// Singleton object to save on allocations
exports.VALIDATION_SUCCESS = new ValidationResult();
/**
 * Return whether this object can be validated at all
 *
 * True unless it's undefined or a CloudFormation intrinsic
 */
function canInspect(x) {
    // Note: using weak equality on purpose, we also want to catch undefined
    return (x != null && !isCloudFormationIntrinsic(x));
}
exports.canInspect = canInspect;
// CloudFormation validators for primitive types
function validateString(x) {
    if (canInspect(x) && typeof x !== 'string') {
        return new ValidationResult(`${JSON.stringify(x)} should be a string`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateString = validateString;
function validateNumber(x) {
    if (canInspect(x) && typeof x !== 'number') {
        return new ValidationResult(`${JSON.stringify(x)} should be a number`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateNumber = validateNumber;
function validateBoolean(x) {
    if (canInspect(x) && typeof x !== 'boolean') {
        return new ValidationResult(`${JSON.stringify(x)} should be a boolean`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateBoolean = validateBoolean;
function validateDate(x) {
    if (canInspect(x) && !(x instanceof Date)) {
        return new ValidationResult(`${JSON.stringify(x)} should be a Date`);
    }
    if (x !== undefined && isNaN(x.getTime())) {
        return new ValidationResult('got an unparseable Date');
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateDate = validateDate;
function validateObject(x) {
    if (canInspect(x) && typeof x !== 'object') {
        return new ValidationResult(`${JSON.stringify(x)} should be an 'object'`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateObject = validateObject;
function validateCfnTag(x) {
    if (!canInspect(x)) {
        return exports.VALIDATION_SUCCESS;
    }
    if (x.key == null || x.value == null) {
        return new ValidationResult(`${JSON.stringify(x)} should have a 'key' and a 'value' property`);
    }
    return exports.VALIDATION_SUCCESS;
}
exports.validateCfnTag = validateCfnTag;
/**
 * Return a list validator based on the given element validator
 */
function listValidator(elementValidator) {
    return (x) => {
        if (!canInspect(x)) {
            return exports.VALIDATION_SUCCESS;
        }
        if (!x.forEach) {
            return new ValidationResult(`${JSON.stringify(x)} should be a list`);
        }
        for (let i = 0; i < x.length; i++) {
            const element = x[i];
            const result = elementValidator(element);
            if (!result.isSuccess) {
                return result.prefix(`element ${i}`);
            }
        }
        return exports.VALIDATION_SUCCESS;
    };
}
exports.listValidator = listValidator;
/**
 * Return a hash validator based on the given element validator
 */
function hashValidator(elementValidator) {
    return (x) => {
        if (!canInspect(x)) {
            return exports.VALIDATION_SUCCESS;
        }
        for (const key of Object.keys(x)) {
            const result = elementValidator(x[key]);
            if (!result.isSuccess) {
                return result.prefix(`element '${key}'`);
            }
        }
        return exports.VALIDATION_SUCCESS;
    };
}
exports.hashValidator = hashValidator;
/**
 * Decorate a validator with a message clarifying the property the failure is for.
 */
function propertyValidator(propName, validator) {
    return (x) => {
        return validator(x).prefix(propName);
    };
}
exports.propertyValidator = propertyValidator;
/**
 * Return a validator that will fail if the passed property is not present
 *
 * Does not distinguish between the property actually not being present, vs being present but 'null'
 * or 'undefined' (courtesy of JavaScript), which is generally the behavior that we want.
 *
 * Empty strings are considered "present"--don't know if this agrees with how CloudFormation looks
 * at the world.
 */
function requiredValidator(x) {
    if (x == null) {
        return new ValidationResult('required but missing');
    }
    return exports.VALIDATION_SUCCESS;
}
exports.requiredValidator = requiredValidator;
/**
 * Require a property from a property bag.
 *
 * @param props  the property bag from which a property is required.
 * @param name   the name of the required property.
 * @param typeName the name of the construct type that requires the property
 *
 * @returns the value of ``props[name]``
 *
 * @throws if the property ``name`` is not present in ``props``.
 */
function requireProperty(props, name, context) {
    const value = props[name];
    if (value == null) {
        throw new Error(`${context.toString()} is missing required property: ${name}`);
    }
    // Possibly add type-checking here...
    return value;
}
exports.requireProperty = requireProperty;
/**
 * Validates if any of the given validators matches
 *
 * We add either/or words to the front of the error mesages so that they read
 * more nicely. Example:
 *
 *   Properties not correct for 'FunctionProps'
 *     codeUri: not one of the possible types
 *       either: properties not correct for 'S3LocationProperty'
 *         bucket: required but missing
 *         key: required but missing
 *         version: required but missing
 *       or: '3' should be a 'string'
 *
 */
function unionValidator(...validators) {
    return (x) => {
        const results = new ValidationResults();
        let eitherOr = 'either';
        for (const validator of validators) {
            const result = validator(x);
            if (result.isSuccess) {
                return result;
            }
            results.collect(result.prefix(eitherOr));
            eitherOr = 'or';
        }
        return results.wrap('not one of the possible types');
    };
}
exports.unionValidator = unionValidator;
/**
 * Return whether the indicated value represents a CloudFormation intrinsic.
 *
 * CloudFormation intrinsics are modeled as objects with a single key, which
 * look like: { "Fn::GetAtt": [...] } or similar.
 */
function isCloudFormationIntrinsic(x) {
    if (!(typeof x === 'object')) {
        return false;
    }
    const keys = Object.keys(x);
    if (keys.length !== 1) {
        return false;
    }
    return keys[0] === 'Ref' || keys[0].substr(0, 4) === 'Fn::';
}
// Cannot be public because JSII gets confused about es5.d.ts
class CfnSynthesisError extends Error {
    constructor() {
        super(...arguments);
        this.type = 'CfnSynthesisError';
    }
}
//# sourceMappingURL=data:application/json;base64,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