"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const encoding_1 = require("./private/encoding");
const token_map_1 = require("./private/token-map");
/**
 * Converts all fragments to strings and concats those
 *
 * Drops 'undefined's.
 */
class StringConcat {
    join(left, right) {
        if (left === undefined) {
            return right !== undefined ? `${right}` : undefined;
        }
        if (right === undefined) {
            return `${left}`;
        }
        return `${left}${right}`;
    }
}
exports.StringConcat = StringConcat;
/**
 * Default resolver implementation
 *
 * @experimental
 */
class DefaultTokenResolver {
    constructor(concat) {
        this.concat = concat;
    }
    /**
     * Default Token resolution
     *
     * Resolve the Token, recurse into whatever it returns,
     * then finally post-process it.
     */
    resolveToken(t, context, postProcessor) {
        try {
            let resolved = t.resolve(context);
            // The token might have returned more values that need resolving, recurse
            resolved = context.resolve(resolved);
            resolved = postProcessor.postProcess(resolved, context);
            return resolved;
        }
        catch (e) {
            let message = `Resolution error: ${e.message}.`;
            if (t.creationStack && t.creationStack.length > 0) {
                message += `\nObject creation stack:\n  at ${t.creationStack.join('\n  at ')}`;
            }
            e.message = message;
            throw e;
        }
    }
    /**
     * Resolve string fragments to Tokens
     */
    resolveString(fragments, context) {
        return fragments.mapTokens({ mapToken: context.resolve }).join(this.concat);
    }
    resolveList(xs, context) {
        // Must be a singleton list token, because concatenation is not allowed.
        if (xs.length !== 1) {
            throw new Error(`Cannot add elements to list token, got: ${xs}`);
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const tokenMap = token_map_1.TokenMap.instance();
        const fragments = str.split(tokenMap.lookupToken.bind(tokenMap));
        if (fragments.length !== 1) {
            throw new Error(`Cannot concatenate strings in a tokenized string array, got: ${xs[0]}`);
        }
        return fragments.mapTokens({ mapToken: context.resolve }).firstValue;
    }
}
exports.DefaultTokenResolver = DefaultTokenResolver;
//# sourceMappingURL=data:application/json;base64,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