"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const token_1 = require("../token");
const encoding_1 = require("./encoding");
const glob = global;
const STRING_SYMBOL = Symbol.for('@aws-cdk/core.TokenMap.STRING');
const LIST_SYMBOL = Symbol.for('@aws-cdk/core.TokenMap.LIST');
const NUMBER_SYMBOL = Symbol.for('@aws-cdk/core.TokenMap.NUMBER');
/**
 * Central place where we keep a mapping from Tokens to their String representation
 *
 * The string representation is used to embed token into strings,
 * and stored to be able to reverse that mapping.
 *
 * All instances of TokenStringMap share the same storage, so that this process
 * works even when different copies of the library are loaded.
 */
class TokenMap {
    constructor() {
        this.stringTokenMap = new Map();
        this.numberTokenMap = new Map();
        this.tokenCounter = 0;
    }
    /**
     * Singleton instance of the token string map
     */
    static instance() {
        if (!glob.__cdkTokenMap) {
            glob.__cdkTokenMap = new TokenMap();
        }
        return glob.__cdkTokenMap;
    }
    /**
     * Generate a unique string for this Token, returning a key
     *
     * Every call for the same Token will produce a new unique string, no
     * attempt is made to deduplicate. Token objects should cache the
     * value themselves, if required.
     *
     * The token can choose (part of) its own representation string with a
     * hint. This may be used to produce aesthetically pleasing and
     * recognizable token representations for humans.
     */
    registerString(token, displayHint) {
        return cachedValue(token, STRING_SYMBOL, () => {
            const key = this.registerStringKey(token, displayHint);
            return `${encoding_1.BEGIN_STRING_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`;
        });
    }
    /**
     * Generate a unique string for this Token, returning a key
     */
    registerList(token, displayHint) {
        return cachedValue(token, LIST_SYMBOL, () => {
            const key = this.registerStringKey(token, displayHint);
            return [`${encoding_1.BEGIN_LIST_TOKEN_MARKER}${key}${encoding_1.END_TOKEN_MARKER}`];
        });
    }
    /**
     * Create a unique number representation for this Token and return it
     */
    registerNumber(token) {
        return cachedValue(token, NUMBER_SYMBOL, () => {
            return this.registerNumberKey(token);
        });
    }
    /**
     * Lookup a token from an encoded value
     */
    tokenFromEncoding(x) {
        if (typeof x === 'string') {
            return this.lookupString(x);
        }
        if (Array.isArray(x)) {
            return this.lookupList(x);
        }
        if (token_1.Token.isUnresolved(x)) {
            return x;
        }
        return undefined;
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupString(s) {
        const fragments = this.splitString(s);
        if (fragments.tokens.length > 0 && fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Reverse a string representation into a Token object
     */
    lookupList(xs) {
        if (xs.length !== 1) {
            return undefined;
        }
        const str = encoding_1.TokenString.forListToken(xs[0]);
        const fragments = str.split(this.lookupToken.bind(this));
        if (fragments.length === 1) {
            return fragments.firstToken;
        }
        return undefined;
    }
    /**
     * Split a string into literals and Tokens
     */
    splitString(s) {
        const str = encoding_1.TokenString.forString(s);
        return str.split(this.lookupToken.bind(this));
    }
    /**
     * Reverse a number encoding into a Token, or undefined if the number wasn't a Token
     */
    lookupNumberToken(x) {
        const tokenIndex = encoding_1.extractTokenDouble(x);
        if (tokenIndex === undefined) {
            return undefined;
        }
        const t = this.numberTokenMap.get(tokenIndex);
        if (t === undefined) {
            throw new Error('Encoded representation of unknown number Token found');
        }
        return t;
    }
    /**
     * Find a Token by key.
     *
     * This excludes the token markers.
     */
    lookupToken(key) {
        const token = this.stringTokenMap.get(key);
        if (!token) {
            throw new Error(`Unrecognized token key: ${key}`);
        }
        return token;
    }
    registerStringKey(token, displayHint) {
        const counter = this.tokenCounter++;
        const representation = (displayHint || 'TOKEN').replace(new RegExp(`[^${encoding_1.VALID_KEY_CHARS}]`, 'g'), '.');
        const key = `${representation}.${counter}`;
        this.stringTokenMap.set(key, token);
        return key;
    }
    registerNumberKey(token) {
        const counter = this.tokenCounter++;
        this.numberTokenMap.set(counter, token);
        return encoding_1.createTokenDouble(counter);
    }
}
exports.TokenMap = TokenMap;
/**
 * Get a cached value for an object, storing it on the object in a symbol
 */
function cachedValue(x, sym, prod) {
    let cached = x[sym];
    if (cached === undefined) {
        cached = prod();
        Object.defineProperty(x, sym, { value: cached });
    }
    return cached;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidG9rZW4tbWFwLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsidG9rZW4tbWFwLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBRUEsb0NBQWlDO0FBQ2pDLHlDQUF1SztBQUN2SyxNQUFNLElBQUksR0FBRyxNQUFhLENBQUM7QUFDM0IsTUFBTSxhQUFhLEdBQUcsTUFBTSxDQUFDLEdBQUcsQ0FBQywrQkFBK0IsQ0FBQyxDQUFDO0FBQ2xFLE1BQU0sV0FBVyxHQUFHLE1BQU0sQ0FBQyxHQUFHLENBQUMsNkJBQTZCLENBQUMsQ0FBQztBQUM5RCxNQUFNLGFBQWEsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLCtCQUErQixDQUFDLENBQUM7QUFDbEU7Ozs7Ozs7O0dBUUc7QUFDSCxNQUFhLFFBQVE7SUFBckI7UUFVcUIsbUJBQWMsR0FBRyxJQUFJLEdBQUcsRUFBdUIsQ0FBQztRQUNoRCxtQkFBYyxHQUFHLElBQUksR0FBRyxFQUF1QixDQUFDO1FBQ3pELGlCQUFZLEdBQUcsQ0FBQyxDQUFDO0lBdUg3QixDQUFDO0lBbElHOztPQUVHO0lBQ0ksTUFBTSxDQUFDLFFBQVE7UUFDbEIsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLEVBQUU7WUFDckIsSUFBSSxDQUFDLGFBQWEsR0FBRyxJQUFJLFFBQVEsRUFBRSxDQUFDO1NBQ3ZDO1FBQ0QsT0FBTyxJQUFJLENBQUMsYUFBYSxDQUFDO0lBQzlCLENBQUM7SUFJRDs7Ozs7Ozs7OztPQVVHO0lBQ0ksY0FBYyxDQUFDLEtBQWtCLEVBQUUsV0FBb0I7UUFDMUQsT0FBTyxXQUFXLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7WUFDMUMsTUFBTSxHQUFHLEdBQUcsSUFBSSxDQUFDLGlCQUFpQixDQUFDLEtBQUssRUFBRSxXQUFXLENBQUMsQ0FBQztZQUN2RCxPQUFPLEdBQUcsb0NBQXlCLEdBQUcsR0FBRyxHQUFHLDJCQUFnQixFQUFFLENBQUM7UUFDbkUsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxZQUFZLENBQUMsS0FBa0IsRUFBRSxXQUFvQjtRQUN4RCxPQUFPLFdBQVcsQ0FBQyxLQUFLLEVBQUUsV0FBVyxFQUFFLEdBQUcsRUFBRTtZQUN4QyxNQUFNLEdBQUcsR0FBRyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxFQUFFLFdBQVcsQ0FBQyxDQUFDO1lBQ3ZELE9BQU8sQ0FBQyxHQUFHLGtDQUF1QixHQUFHLEdBQUcsR0FBRywyQkFBZ0IsRUFBRSxDQUFDLENBQUM7UUFDbkUsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxjQUFjLENBQUMsS0FBa0I7UUFDcEMsT0FBTyxXQUFXLENBQUMsS0FBSyxFQUFFLGFBQWEsRUFBRSxHQUFHLEVBQUU7WUFDMUMsT0FBTyxJQUFJLENBQUMsaUJBQWlCLENBQUMsS0FBSyxDQUFDLENBQUM7UUFDekMsQ0FBQyxDQUFDLENBQUM7SUFDUCxDQUFDO0lBQ0Q7O09BRUc7SUFDSSxpQkFBaUIsQ0FBQyxDQUFNO1FBQzNCLElBQUksT0FBTyxDQUFDLEtBQUssUUFBUSxFQUFFO1lBQ3ZCLE9BQU8sSUFBSSxDQUFDLFlBQVksQ0FBQyxDQUFDLENBQUMsQ0FBQztTQUMvQjtRQUNELElBQUksS0FBSyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRTtZQUNsQixPQUFPLElBQUksQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLENBQUM7U0FDN0I7UUFDRCxJQUFJLGFBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLEVBQUU7WUFDdkIsT0FBTyxDQUFDLENBQUM7U0FDWjtRQUNELE9BQU8sU0FBUyxDQUFDO0lBQ3JCLENBQUM7SUFDRDs7T0FFRztJQUNJLFlBQVksQ0FBQyxDQUFTO1FBQ3pCLE1BQU0sU0FBUyxHQUFHLElBQUksQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDdEMsSUFBSSxTQUFTLENBQUMsTUFBTSxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksU0FBUyxDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDdkQsT0FBTyxTQUFTLENBQUMsVUFBVSxDQUFDO1NBQy9CO1FBQ0QsT0FBTyxTQUFTLENBQUM7SUFDckIsQ0FBQztJQUNEOztPQUVHO0lBQ0ksVUFBVSxDQUFDLEVBQVk7UUFDMUIsSUFBSSxFQUFFLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNqQixPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE1BQU0sR0FBRyxHQUFHLHNCQUFXLENBQUMsWUFBWSxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQzVDLE1BQU0sU0FBUyxHQUFHLEdBQUcsQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUN6RCxJQUFJLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ3hCLE9BQU8sU0FBUyxDQUFDLFVBQVUsQ0FBQztTQUMvQjtRQUNELE9BQU8sU0FBUyxDQUFDO0lBQ3JCLENBQUM7SUFDRDs7T0FFRztJQUNJLFdBQVcsQ0FBQyxDQUFTO1FBQ3hCLE1BQU0sR0FBRyxHQUFHLHNCQUFXLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO1FBQ3JDLE9BQU8sR0FBRyxDQUFDLEtBQUssQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO0lBQ2xELENBQUM7SUFDRDs7T0FFRztJQUNJLGlCQUFpQixDQUFDLENBQVM7UUFDOUIsTUFBTSxVQUFVLEdBQUcsNkJBQWtCLENBQUMsQ0FBQyxDQUFDLENBQUM7UUFDekMsSUFBSSxVQUFVLEtBQUssU0FBUyxFQUFFO1lBQzFCLE9BQU8sU0FBUyxDQUFDO1NBQ3BCO1FBQ0QsTUFBTSxDQUFDLEdBQUcsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsVUFBVSxDQUFDLENBQUM7UUFDOUMsSUFBSSxDQUFDLEtBQUssU0FBUyxFQUFFO1lBQ2pCLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQztTQUMzRTtRQUNELE9BQU8sQ0FBQyxDQUFDO0lBQ2IsQ0FBQztJQUNEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsR0FBVztRQUMxQixNQUFNLEtBQUssR0FBRyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsQ0FBQztRQUMzQyxJQUFJLENBQUMsS0FBSyxFQUFFO1lBQ1IsTUFBTSxJQUFJLEtBQUssQ0FBQywyQkFBMkIsR0FBRyxFQUFFLENBQUMsQ0FBQztTQUNyRDtRQUNELE9BQU8sS0FBSyxDQUFDO0lBQ2pCLENBQUM7SUFDTyxpQkFBaUIsQ0FBQyxLQUFrQixFQUFFLFdBQW9CO1FBQzlELE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUNwQyxNQUFNLGNBQWMsR0FBRyxDQUFDLFdBQVcsSUFBSSxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxNQUFNLENBQUMsS0FBSywwQkFBZSxHQUFHLEVBQUUsR0FBRyxDQUFDLEVBQUUsR0FBRyxDQUFDLENBQUM7UUFDdkcsTUFBTSxHQUFHLEdBQUcsR0FBRyxjQUFjLElBQUksT0FBTyxFQUFFLENBQUM7UUFDM0MsSUFBSSxDQUFDLGNBQWMsQ0FBQyxHQUFHLENBQUMsR0FBRyxFQUFFLEtBQUssQ0FBQyxDQUFDO1FBQ3BDLE9BQU8sR0FBRyxDQUFDO0lBQ2YsQ0FBQztJQUNPLGlCQUFpQixDQUFDLEtBQWtCO1FBQ3hDLE1BQU0sT0FBTyxHQUFHLElBQUksQ0FBQyxZQUFZLEVBQUUsQ0FBQztRQUNwQyxJQUFJLENBQUMsY0FBYyxDQUFDLEdBQUcsQ0FBQyxPQUFPLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFDeEMsT0FBTyw0QkFBaUIsQ0FBQyxPQUFPLENBQUMsQ0FBQztJQUN0QyxDQUFDO0NBQ0o7QUFuSUQsNEJBbUlDO0FBQ0Q7O0dBRUc7QUFDSCxTQUFTLFdBQVcsQ0FBc0IsQ0FBSSxFQUFFLEdBQVcsRUFBRSxJQUFhO0lBQ3RFLElBQUksTUFBTSxHQUFJLENBQVMsQ0FBQyxHQUFVLENBQUMsQ0FBQztJQUNwQyxJQUFJLE1BQU0sS0FBSyxTQUFTLEVBQUU7UUFDdEIsTUFBTSxHQUFHLElBQUksRUFBRSxDQUFDO1FBQ2hCLE1BQU0sQ0FBQyxjQUFjLENBQUMsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLEtBQUssRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDO0tBQ3BEO0lBQ0QsT0FBTyxNQUFNLENBQUM7QUFDbEIsQ0FBQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IElSZXNvbHZhYmxlIH0gZnJvbSAnLi4vcmVzb2x2YWJsZSc7XG5pbXBvcnQgeyBUb2tlbml6ZWRTdHJpbmdGcmFnbWVudHMgfSBmcm9tICcuLi9zdHJpbmctZnJhZ21lbnRzJztcbmltcG9ydCB7IFRva2VuIH0gZnJvbSAnLi4vdG9rZW4nO1xuaW1wb3J0IHsgQkVHSU5fTElTVF9UT0tFTl9NQVJLRVIsIEJFR0lOX1NUUklOR19UT0tFTl9NQVJLRVIsIGNyZWF0ZVRva2VuRG91YmxlLCBFTkRfVE9LRU5fTUFSS0VSLCBleHRyYWN0VG9rZW5Eb3VibGUsIFRva2VuU3RyaW5nLCBWQUxJRF9LRVlfQ0hBUlMgfSBmcm9tICcuL2VuY29kaW5nJztcbmNvbnN0IGdsb2IgPSBnbG9iYWwgYXMgYW55O1xuY29uc3QgU1RSSU5HX1NZTUJPTCA9IFN5bWJvbC5mb3IoJ0Bhd3MtY2RrL2NvcmUuVG9rZW5NYXAuU1RSSU5HJyk7XG5jb25zdCBMSVNUX1NZTUJPTCA9IFN5bWJvbC5mb3IoJ0Bhd3MtY2RrL2NvcmUuVG9rZW5NYXAuTElTVCcpO1xuY29uc3QgTlVNQkVSX1NZTUJPTCA9IFN5bWJvbC5mb3IoJ0Bhd3MtY2RrL2NvcmUuVG9rZW5NYXAuTlVNQkVSJyk7XG4vKipcbiAqIENlbnRyYWwgcGxhY2Ugd2hlcmUgd2Uga2VlcCBhIG1hcHBpbmcgZnJvbSBUb2tlbnMgdG8gdGhlaXIgU3RyaW5nIHJlcHJlc2VudGF0aW9uXG4gKlxuICogVGhlIHN0cmluZyByZXByZXNlbnRhdGlvbiBpcyB1c2VkIHRvIGVtYmVkIHRva2VuIGludG8gc3RyaW5ncyxcbiAqIGFuZCBzdG9yZWQgdG8gYmUgYWJsZSB0byByZXZlcnNlIHRoYXQgbWFwcGluZy5cbiAqXG4gKiBBbGwgaW5zdGFuY2VzIG9mIFRva2VuU3RyaW5nTWFwIHNoYXJlIHRoZSBzYW1lIHN0b3JhZ2UsIHNvIHRoYXQgdGhpcyBwcm9jZXNzXG4gKiB3b3JrcyBldmVuIHdoZW4gZGlmZmVyZW50IGNvcGllcyBvZiB0aGUgbGlicmFyeSBhcmUgbG9hZGVkLlxuICovXG5leHBvcnQgY2xhc3MgVG9rZW5NYXAge1xuICAgIC8qKlxuICAgICAqIFNpbmdsZXRvbiBpbnN0YW5jZSBvZiB0aGUgdG9rZW4gc3RyaW5nIG1hcFxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgaW5zdGFuY2UoKTogVG9rZW5NYXAge1xuICAgICAgICBpZiAoIWdsb2IuX19jZGtUb2tlbk1hcCkge1xuICAgICAgICAgICAgZ2xvYi5fX2Nka1Rva2VuTWFwID0gbmV3IFRva2VuTWFwKCk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIGdsb2IuX19jZGtUb2tlbk1hcDtcbiAgICB9XG4gICAgcHJpdmF0ZSByZWFkb25seSBzdHJpbmdUb2tlbk1hcCA9IG5ldyBNYXA8c3RyaW5nLCBJUmVzb2x2YWJsZT4oKTtcbiAgICBwcml2YXRlIHJlYWRvbmx5IG51bWJlclRva2VuTWFwID0gbmV3IE1hcDxudW1iZXIsIElSZXNvbHZhYmxlPigpO1xuICAgIHByaXZhdGUgdG9rZW5Db3VudGVyID0gMDtcbiAgICAvKipcbiAgICAgKiBHZW5lcmF0ZSBhIHVuaXF1ZSBzdHJpbmcgZm9yIHRoaXMgVG9rZW4sIHJldHVybmluZyBhIGtleVxuICAgICAqXG4gICAgICogRXZlcnkgY2FsbCBmb3IgdGhlIHNhbWUgVG9rZW4gd2lsbCBwcm9kdWNlIGEgbmV3IHVuaXF1ZSBzdHJpbmcsIG5vXG4gICAgICogYXR0ZW1wdCBpcyBtYWRlIHRvIGRlZHVwbGljYXRlLiBUb2tlbiBvYmplY3RzIHNob3VsZCBjYWNoZSB0aGVcbiAgICAgKiB2YWx1ZSB0aGVtc2VsdmVzLCBpZiByZXF1aXJlZC5cbiAgICAgKlxuICAgICAqIFRoZSB0b2tlbiBjYW4gY2hvb3NlIChwYXJ0IG9mKSBpdHMgb3duIHJlcHJlc2VudGF0aW9uIHN0cmluZyB3aXRoIGFcbiAgICAgKiBoaW50LiBUaGlzIG1heSBiZSB1c2VkIHRvIHByb2R1Y2UgYWVzdGhldGljYWxseSBwbGVhc2luZyBhbmRcbiAgICAgKiByZWNvZ25pemFibGUgdG9rZW4gcmVwcmVzZW50YXRpb25zIGZvciBodW1hbnMuXG4gICAgICovXG4gICAgcHVibGljIHJlZ2lzdGVyU3RyaW5nKHRva2VuOiBJUmVzb2x2YWJsZSwgZGlzcGxheUhpbnQ/OiBzdHJpbmcpOiBzdHJpbmcge1xuICAgICAgICByZXR1cm4gY2FjaGVkVmFsdWUodG9rZW4sIFNUUklOR19TWU1CT0wsICgpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IGtleSA9IHRoaXMucmVnaXN0ZXJTdHJpbmdLZXkodG9rZW4sIGRpc3BsYXlIaW50KTtcbiAgICAgICAgICAgIHJldHVybiBgJHtCRUdJTl9TVFJJTkdfVE9LRU5fTUFSS0VSfSR7a2V5fSR7RU5EX1RPS0VOX01BUktFUn1gO1xuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogR2VuZXJhdGUgYSB1bmlxdWUgc3RyaW5nIGZvciB0aGlzIFRva2VuLCByZXR1cm5pbmcgYSBrZXlcbiAgICAgKi9cbiAgICBwdWJsaWMgcmVnaXN0ZXJMaXN0KHRva2VuOiBJUmVzb2x2YWJsZSwgZGlzcGxheUhpbnQ/OiBzdHJpbmcpOiBzdHJpbmdbXSB7XG4gICAgICAgIHJldHVybiBjYWNoZWRWYWx1ZSh0b2tlbiwgTElTVF9TWU1CT0wsICgpID0+IHtcbiAgICAgICAgICAgIGNvbnN0IGtleSA9IHRoaXMucmVnaXN0ZXJTdHJpbmdLZXkodG9rZW4sIGRpc3BsYXlIaW50KTtcbiAgICAgICAgICAgIHJldHVybiBbYCR7QkVHSU5fTElTVF9UT0tFTl9NQVJLRVJ9JHtrZXl9JHtFTkRfVE9LRU5fTUFSS0VSfWBdO1xuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogQ3JlYXRlIGEgdW5pcXVlIG51bWJlciByZXByZXNlbnRhdGlvbiBmb3IgdGhpcyBUb2tlbiBhbmQgcmV0dXJuIGl0XG4gICAgICovXG4gICAgcHVibGljIHJlZ2lzdGVyTnVtYmVyKHRva2VuOiBJUmVzb2x2YWJsZSk6IG51bWJlciB7XG4gICAgICAgIHJldHVybiBjYWNoZWRWYWx1ZSh0b2tlbiwgTlVNQkVSX1NZTUJPTCwgKCkgPT4ge1xuICAgICAgICAgICAgcmV0dXJuIHRoaXMucmVnaXN0ZXJOdW1iZXJLZXkodG9rZW4pO1xuICAgICAgICB9KTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogTG9va3VwIGEgdG9rZW4gZnJvbSBhbiBlbmNvZGVkIHZhbHVlXG4gICAgICovXG4gICAgcHVibGljIHRva2VuRnJvbUVuY29kaW5nKHg6IGFueSk6IElSZXNvbHZhYmxlIHwgdW5kZWZpbmVkIHtcbiAgICAgICAgaWYgKHR5cGVvZiB4ID09PSAnc3RyaW5nJykge1xuICAgICAgICAgICAgcmV0dXJuIHRoaXMubG9va3VwU3RyaW5nKHgpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChBcnJheS5pc0FycmF5KHgpKSB7XG4gICAgICAgICAgICByZXR1cm4gdGhpcy5sb29rdXBMaXN0KHgpO1xuICAgICAgICB9XG4gICAgICAgIGlmIChUb2tlbi5pc1VucmVzb2x2ZWQoeCkpIHtcbiAgICAgICAgICAgIHJldHVybiB4O1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJldmVyc2UgYSBzdHJpbmcgcmVwcmVzZW50YXRpb24gaW50byBhIFRva2VuIG9iamVjdFxuICAgICAqL1xuICAgIHB1YmxpYyBsb29rdXBTdHJpbmcoczogc3RyaW5nKTogSVJlc29sdmFibGUgfCB1bmRlZmluZWQge1xuICAgICAgICBjb25zdCBmcmFnbWVudHMgPSB0aGlzLnNwbGl0U3RyaW5nKHMpO1xuICAgICAgICBpZiAoZnJhZ21lbnRzLnRva2Vucy5sZW5ndGggPiAwICYmIGZyYWdtZW50cy5sZW5ndGggPT09IDEpIHtcbiAgICAgICAgICAgIHJldHVybiBmcmFnbWVudHMuZmlyc3RUb2tlbjtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXZlcnNlIGEgc3RyaW5nIHJlcHJlc2VudGF0aW9uIGludG8gYSBUb2tlbiBvYmplY3RcbiAgICAgKi9cbiAgICBwdWJsaWMgbG9va3VwTGlzdCh4czogc3RyaW5nW10pOiBJUmVzb2x2YWJsZSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGlmICh4cy5sZW5ndGggIT09IDEpIHtcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICAgICAgY29uc3Qgc3RyID0gVG9rZW5TdHJpbmcuZm9yTGlzdFRva2VuKHhzWzBdKTtcbiAgICAgICAgY29uc3QgZnJhZ21lbnRzID0gc3RyLnNwbGl0KHRoaXMubG9va3VwVG9rZW4uYmluZCh0aGlzKSk7XG4gICAgICAgIGlmIChmcmFnbWVudHMubGVuZ3RoID09PSAxKSB7XG4gICAgICAgICAgICByZXR1cm4gZnJhZ21lbnRzLmZpcnN0VG9rZW47XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICB9XG4gICAgLyoqXG4gICAgICogU3BsaXQgYSBzdHJpbmcgaW50byBsaXRlcmFscyBhbmQgVG9rZW5zXG4gICAgICovXG4gICAgcHVibGljIHNwbGl0U3RyaW5nKHM6IHN0cmluZyk6IFRva2VuaXplZFN0cmluZ0ZyYWdtZW50cyB7XG4gICAgICAgIGNvbnN0IHN0ciA9IFRva2VuU3RyaW5nLmZvclN0cmluZyhzKTtcbiAgICAgICAgcmV0dXJuIHN0ci5zcGxpdCh0aGlzLmxvb2t1cFRva2VuLmJpbmQodGhpcykpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBSZXZlcnNlIGEgbnVtYmVyIGVuY29kaW5nIGludG8gYSBUb2tlbiwgb3IgdW5kZWZpbmVkIGlmIHRoZSBudW1iZXIgd2Fzbid0IGEgVG9rZW5cbiAgICAgKi9cbiAgICBwdWJsaWMgbG9va3VwTnVtYmVyVG9rZW4oeDogbnVtYmVyKTogSVJlc29sdmFibGUgfCB1bmRlZmluZWQge1xuICAgICAgICBjb25zdCB0b2tlbkluZGV4ID0gZXh0cmFjdFRva2VuRG91YmxlKHgpO1xuICAgICAgICBpZiAodG9rZW5JbmRleCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIGNvbnN0IHQgPSB0aGlzLm51bWJlclRva2VuTWFwLmdldCh0b2tlbkluZGV4KTtcbiAgICAgICAgaWYgKHQgPT09IHVuZGVmaW5lZCkge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdFbmNvZGVkIHJlcHJlc2VudGF0aW9uIG9mIHVua25vd24gbnVtYmVyIFRva2VuIGZvdW5kJyk7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHQ7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIEZpbmQgYSBUb2tlbiBieSBrZXkuXG4gICAgICpcbiAgICAgKiBUaGlzIGV4Y2x1ZGVzIHRoZSB0b2tlbiBtYXJrZXJzLlxuICAgICAqL1xuICAgIHB1YmxpYyBsb29rdXBUb2tlbihrZXk6IHN0cmluZyk6IElSZXNvbHZhYmxlIHtcbiAgICAgICAgY29uc3QgdG9rZW4gPSB0aGlzLnN0cmluZ1Rva2VuTWFwLmdldChrZXkpO1xuICAgICAgICBpZiAoIXRva2VuKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFVucmVjb2duaXplZCB0b2tlbiBrZXk6ICR7a2V5fWApO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0b2tlbjtcbiAgICB9XG4gICAgcHJpdmF0ZSByZWdpc3RlclN0cmluZ0tleSh0b2tlbjogSVJlc29sdmFibGUsIGRpc3BsYXlIaW50Pzogc3RyaW5nKTogc3RyaW5nIHtcbiAgICAgICAgY29uc3QgY291bnRlciA9IHRoaXMudG9rZW5Db3VudGVyKys7XG4gICAgICAgIGNvbnN0IHJlcHJlc2VudGF0aW9uID0gKGRpc3BsYXlIaW50IHx8ICdUT0tFTicpLnJlcGxhY2UobmV3IFJlZ0V4cChgW14ke1ZBTElEX0tFWV9DSEFSU31dYCwgJ2cnKSwgJy4nKTtcbiAgICAgICAgY29uc3Qga2V5ID0gYCR7cmVwcmVzZW50YXRpb259LiR7Y291bnRlcn1gO1xuICAgICAgICB0aGlzLnN0cmluZ1Rva2VuTWFwLnNldChrZXksIHRva2VuKTtcbiAgICAgICAgcmV0dXJuIGtleTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZWdpc3Rlck51bWJlcktleSh0b2tlbjogSVJlc29sdmFibGUpOiBudW1iZXIge1xuICAgICAgICBjb25zdCBjb3VudGVyID0gdGhpcy50b2tlbkNvdW50ZXIrKztcbiAgICAgICAgdGhpcy5udW1iZXJUb2tlbk1hcC5zZXQoY291bnRlciwgdG9rZW4pO1xuICAgICAgICByZXR1cm4gY3JlYXRlVG9rZW5Eb3VibGUoY291bnRlcik7XG4gICAgfVxufVxuLyoqXG4gKiBHZXQgYSBjYWNoZWQgdmFsdWUgZm9yIGFuIG9iamVjdCwgc3RvcmluZyBpdCBvbiB0aGUgb2JqZWN0IGluIGEgc3ltYm9sXG4gKi9cbmZ1bmN0aW9uIGNhY2hlZFZhbHVlPEEgZXh0ZW5kcyBvYmplY3QsIEI+KHg6IEEsIHN5bTogc3ltYm9sLCBwcm9kOiAoKSA9PiBCKSB7XG4gICAgbGV0IGNhY2hlZCA9ICh4IGFzIGFueSlbc3ltIGFzIGFueV07XG4gICAgaWYgKGNhY2hlZCA9PT0gdW5kZWZpbmVkKSB7XG4gICAgICAgIGNhY2hlZCA9IHByb2QoKTtcbiAgICAgICAgT2JqZWN0LmRlZmluZVByb3BlcnR5KHgsIHN5bSwgeyB2YWx1ZTogY2FjaGVkIH0pO1xuICAgIH1cbiAgICByZXR1cm4gY2FjaGVkO1xufVxuIl19