"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("../../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const stage_1 = require("../stage");
const prepare_app_1 = require("./prepare-app");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder);
    return builder.buildAssembly({
        runtimeInfo: options.runtimeInfo,
    });
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        // hackery to be able to access some private members with strong types (yack!)
        const node = construct.node._actualNode;
        const allAspectsHere = [...inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : [], ...node._aspects];
        for (const aspect of allAspectsHere) {
            if (node.invokedAspects.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            node.invokedAspects.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, 'post', construct => construct.onPrepare());
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder) {
    visit(root, 'post', construct => construct.onSynthesize({
        outdir: builder.outdir,
        assembly: builder,
    }));
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, 'pre', construct => {
        for (const message of construct.onValidate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${e.source.node.path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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