"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// ----------------------------------------------------
// CROSS REFERENCES
// ----------------------------------------------------
const cfn_element_1 = require("../cfn-element");
const cfn_output_1 = require("../cfn-output");
const cfn_parameter_1 = require("../cfn-parameter");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const token_1 = require("../token");
const cfn_reference_1 = require("./cfn-reference");
const intrinsic_1 = require("./intrinsic");
const resolve_1 = require("./resolve");
const uniqueid_1 = require("./uniqueid");
/**
 * This is called from the App level to resolve all references defined. Each
 * reference is resolved based on it's consumption context.
 */
function resolveReferences(scope) {
    const edges = findAllReferences(scope);
    for (const { source, value } of edges) {
        const consumer = stack_1.Stack.of(source);
        // resolve the value in the context of the consumer
        if (!value.hasValueForStack(consumer)) {
            const resolved = resolveValue(consumer, value);
            value.assignValueForStack(consumer, resolved);
        }
    }
}
exports.resolveReferences = resolveReferences;
/**
 * Resolves the value for `reference` in the context of `consumer`.
 */
function resolveValue(consumer, reference) {
    const producer = stack_1.Stack.of(reference.target);
    // produce and consumer stacks are the same, we can just return the value itself.
    if (producer === consumer) {
        return reference;
    }
    // unsupported: stacks from different apps
    if (producer.node.root !== consumer.node.root) {
        throw new Error('Cannot reference across apps. Consuming and producing stacks must be defined within the same CDK app.');
    }
    // unsupported: stacks are not in the same environment
    if (producer.environment !== consumer.environment) {
        throw new Error(`Stack "${consumer.node.path}" cannot consume a cross reference from stack "${producer.node.path}". ` +
            'Cross stack references are only supported for stacks deployed to the same environment or between nested stacks and their parent stack');
    }
    // ----------------------------------------------------------------------
    // consumer is nested in the producer (directly or indirectly)
    // ----------------------------------------------------------------------
    // if the consumer is nested within the producer (directly or indirectly),
    // wire through a CloudFormation parameter and then resolve the reference with
    // the parent stack as the consumer.
    if (consumer.nestedStackParent && isNested(consumer, producer)) {
        const parameterValue = resolveValue(consumer.nestedStackParent, reference);
        return createNestedStackParameter(consumer, reference, parameterValue);
    }
    // ----------------------------------------------------------------------
    // producer is a nested stack
    // ----------------------------------------------------------------------
    // if the producer is nested, always publish the value through a
    // cloudformation output and resolve recursively with the Fn::GetAtt
    // of the output in the parent stack.
    // one might ask, if the consumer is not a parent of the producer,
    // why not just use export/import? the reason is that we cannot
    // generate an "export name" from a nested stack because the export
    // name must contain the stack name to ensure uniqueness, and we
    // don't know the stack name of a nested stack before we deploy it.
    // therefore, we can only export from a top-level stack.
    if (producer.nested) {
        const outputValue = createNestedStackOutput(producer, reference);
        return resolveValue(consumer, outputValue);
    }
    // ----------------------------------------------------------------------
    // export/import
    // ----------------------------------------------------------------------
    // export the value through a cloudformation "export name" and use an
    // Fn::ImportValue in the consumption site.
    // add a dependency between the producer and the consumer. dependency logic
    // will take care of applying the dependency at the right level (e.g. the
    // top-level stacks).
    consumer.addDependency(producer, `${consumer.node.path} -> ${reference.target.node.path}.${reference.displayName}`);
    return createImportValue(reference);
}
/**
 * Finds all the CloudFormation references in a construct tree.
 */
function findAllReferences(root) {
    const result = new Array();
    for (const consumer of root.node.findAll()) {
        // include only CfnElements (i.e. resources)
        if (!cfn_element_1.CfnElement.isCfnElement(consumer)) {
            continue;
        }
        try {
            const tokens = resolve_1.findTokens(consumer, () => consumer._toCloudFormation());
            // iterate over all the tokens (e.g. intrinsic functions, lazies, etc) that
            // were found in the cloudformation representation of this resource.
            for (const token of tokens) {
                // include only CfnReferences (i.e. "Ref" and "Fn::GetAtt")
                if (!cfn_reference_1.CfnReference.isCfnReference(token)) {
                    continue;
                }
                result.push({
                    source: consumer,
                    value: token,
                });
            }
        }
        catch (e) {
            // Note: it might be that the properties of the CFN object aren't valid.
            // This will usually be preventatively caught in a construct's validate()
            // and turned into a nicely descriptive error, but we're running prepare()
            // before validate(). Swallow errors that occur because the CFN layer
            // doesn't validate completely.
            //
            // This does make the assumption that the error will not be rectified,
            // but the error will be thrown later on anyway. If the error doesn't
            // get thrown down the line, we may miss references.
            if (e.type === 'CfnSynthesisError') {
                continue;
            }
            throw e;
        }
    }
    return result;
}
// ------------------------------------------------------------------------------------------------
// export/import
// ------------------------------------------------------------------------------------------------
/**
 * Imports a value from another stack by creating an "Output" with an "ExportName"
 * and returning an "Fn::ImportValue" token.
 */
function createImportValue(reference) {
    const exportingStack = stack_1.Stack.of(reference.target);
    // Ensure a singleton "Exports" scoping Construct
    // This mostly exists to trigger LogicalID munging, which would be
    // disabled if we parented constructs directly under Stack.
    // Also it nicely prevents likely construct name clashes
    const exportsScope = getCreateExportsScope(exportingStack);
    // Ensure a singleton CfnOutput for this value
    const resolved = exportingStack.resolve(reference);
    const id = 'Output' + JSON.stringify(resolved);
    const exportName = generateExportName(exportsScope, id);
    if (token_1.Token.isUnresolved(exportName)) {
        throw new Error(`unresolved token in generated export name: ${JSON.stringify(exportingStack.resolve(exportName))}`);
    }
    const output = exportsScope.node.tryFindChild(id);
    if (!output) {
        new cfn_output_1.CfnOutput(exportsScope, id, { value: token_1.Token.asString(reference), exportName });
    }
    // We want to return an actual FnImportValue Token here, but Fn.importValue() returns a 'string',
    // so construct one in-place.
    return new intrinsic_1.Intrinsic({ 'Fn::ImportValue': exportName });
}
function getCreateExportsScope(stack) {
    const exportsName = 'Exports';
    let stackExports = stack.node.tryFindChild(exportsName);
    if (stackExports === undefined) {
        stackExports = new construct_compat_1.Construct(stack, exportsName);
    }
    return stackExports;
}
function generateExportName(stackExports, id) {
    const stack = stack_1.Stack.of(stackExports);
    const components = [...stackExports.node.scopes.slice(2).map(c => c.node.id), id];
    const prefix = stack.stackName ? stack.stackName + ':' : '';
    const exportName = prefix + uniqueid_1.makeUniqueId(components);
    return exportName;
}
// ------------------------------------------------------------------------------------------------
// nested stacks
// ------------------------------------------------------------------------------------------------
/**
 * Adds a CloudFormation parameter to a nested stack and assigns it with the
 * value of the reference.
 */
function createNestedStackParameter(nested, reference, value) {
    // we call "this.resolve" to ensure that tokens do not creep in (for example, if the reference display name includes tokens)
    const paramId = nested.resolve(`reference-to-${reference.target.node.uniqueId}.${reference.displayName}`);
    let param = nested.node.tryFindChild(paramId);
    if (!param) {
        param = new cfn_parameter_1.CfnParameter(nested, paramId, { type: 'String' });
        // Ugly little hack until we move NestedStack to this module.
        if (!('setParameter' in nested)) {
            throw new Error('assertion failed: nested stack should have a "setParameter" method');
        }
        nested.setParameter(param.logicalId, token_1.Token.asString(value));
    }
    return param.value;
}
/**
 * Adds a CloudFormation output to a nested stack and returns an "Fn::GetAtt"
 * intrinsic that can be used to reference this output in the parent stack.
 */
function createNestedStackOutput(producer, reference) {
    const outputId = `${reference.target.node.uniqueId}${reference.displayName}`;
    let output = producer.node.tryFindChild(outputId);
    if (!output) {
        output = new cfn_output_1.CfnOutput(producer, outputId, { value: token_1.Token.asString(reference) });
    }
    if (!producer.nestedStackResource) {
        throw new Error('assertion failed');
    }
    return producer.nestedStackResource.getAtt(`Outputs.${output.logicalId}`);
}
/**
 * @returns true if this stack is a direct or indirect parent of the nested
 * stack `nested`.
 *
 * If `child` is not a nested stack, always returns `false` because it can't
 * have a parent, dah.
 */
function isNested(nested, parent) {
    // if the parent is a direct parent
    if (nested.nestedStackParent === parent) {
        return true;
    }
    // we reached a top-level (non-nested) stack without finding the parent
    if (!nested.nestedStackParent) {
        return false;
    }
    // recurse with the child's direct parent
    return isNested(nested.nestedStackParent, parent);
}
//# sourceMappingURL=data:application/json;base64,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