"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    if (root.node.scope && !stage_1.Stage.isStage(root)) {
        throw new Error('prepareApp can only be called on a stage or a root construct');
    }
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare its template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    const includeStack = (stack) => {
        if (!stack_1.Stack.isStack(stack)) {
            return false;
        }
        if (!stack.nested) {
            return false;
        }
        // test: if we are not within a stage, then include it.
        if (!stage_1.Stage.of(stack)) {
            return true;
        }
        return stage_1.Stage.of(stack) === root;
    };
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (includeStack(stack)) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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