"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const minimatch = require("minimatch");
const path = require("path");
const options_1 = require("./options");
/**
 * Determines whether a given file should be excluded or not based on given
 * exclusion glob patterns.
 *
 * @param exclude  exclusion patterns
 * @param filePath file path to be assessed against the pattern
 *
 * @returns `true` if the file should be excluded
 */
function shouldExclude(exclude, filePath) {
    let excludeOutput = false;
    for (const pattern of exclude) {
        const negate = pattern.startsWith('!');
        const match = minimatch(filePath, pattern, { matchBase: true, flipNegate: true });
        if (!negate && match) {
            excludeOutput = true;
        }
        if (negate && match) {
            excludeOutput = false;
        }
    }
    return excludeOutput;
}
exports.shouldExclude = shouldExclude;
/**
 * Determines whether a symlink should be followed or not, based on a FollowMode.
 *
 * @param mode       the FollowMode.
 * @param sourceRoot the root of the source tree.
 * @param realPath   the real path of the target of the symlink.
 *
 * @returns true if the link should be followed.
 */
function shouldFollow(mode, sourceRoot, realPath) {
    switch (mode) {
        case options_1.SymlinkFollowMode.ALWAYS:
            return fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.EXTERNAL:
            return !_isInternal() && fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.BLOCK_EXTERNAL:
            return _isInternal() && fs.existsSync(realPath);
        case options_1.SymlinkFollowMode.NEVER:
            return false;
        default:
            throw new Error(`Unsupported FollowMode: ${mode}`);
    }
    function _isInternal() {
        return path.resolve(realPath).startsWith(path.resolve(sourceRoot));
    }
}
exports.shouldFollow = shouldFollow;
//# sourceMappingURL=data:application/json;base64,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