"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const options_1 = require("./options");
const utils_1 = require("./utils");
const BUFFER_SIZE = 8 * 1024;
const CTRL_SOH = '\x01';
const CTRL_SOT = '\x02';
const CTRL_ETX = '\x03';
/**
 * Produces fingerprint based on the contents of a single file or an entire directory tree.
 *
 * The fingerprint will also include:
 * 1. An extra string if defined in `options.extra`.
 * 2. The set of exclude patterns, if defined in `options.exclude`
 * 3. The symlink follow mode value.
 *
 * @param fileOrDirectory The directory or file to fingerprint
 * @param options Fingerprinting options
 */
function fingerprint(fileOrDirectory, options = {}) {
    const hash = crypto.createHash('sha256');
    _hashField(hash, 'options.extra', options.extraHash || '');
    const follow = options.follow || options_1.SymlinkFollowMode.EXTERNAL;
    _hashField(hash, 'options.follow', follow);
    const rootDirectory = fs.statSync(fileOrDirectory).isDirectory()
        ? fileOrDirectory
        : path.dirname(fileOrDirectory);
    const exclude = options.exclude || [];
    if (exclude.length) {
        _hashField(hash, 'options.exclude', JSON.stringify(exclude));
    }
    const isDir = fs.statSync(fileOrDirectory).isDirectory();
    _processFileOrDirectory(fileOrDirectory, isDir);
    return hash.digest('hex');
    function _processFileOrDirectory(symbolicPath, isRootDir = false, realPath = symbolicPath) {
        if (!isRootDir && utils_1.shouldExclude(exclude, symbolicPath)) {
            return;
        }
        const stat = fs.lstatSync(realPath);
        const relativePath = path.relative(fileOrDirectory, symbolicPath);
        if (stat.isSymbolicLink()) {
            const linkTarget = fs.readlinkSync(realPath);
            const resolvedLinkTarget = path.resolve(path.dirname(realPath), linkTarget);
            if (utils_1.shouldFollow(follow, rootDirectory, resolvedLinkTarget)) {
                _processFileOrDirectory(symbolicPath, false, resolvedLinkTarget);
            }
            else {
                _hashField(hash, `link:${relativePath}`, linkTarget);
            }
        }
        else if (stat.isFile()) {
            _hashField(hash, `file:${relativePath}`, _contentFingerprint(realPath, stat));
        }
        else if (stat.isDirectory()) {
            for (const item of fs.readdirSync(realPath).sort()) {
                _processFileOrDirectory(path.join(symbolicPath, item), false, path.join(realPath, item));
            }
        }
        else {
            throw new Error(`Unable to hash ${symbolicPath}: it is neither a file nor a directory`);
        }
    }
}
exports.fingerprint = fingerprint;
function _contentFingerprint(file, stat) {
    const hash = crypto.createHash('sha256');
    const buffer = Buffer.alloc(BUFFER_SIZE);
    // tslint:disable-next-line: no-bitwise
    const fd = fs.openSync(file, fs.constants.O_DSYNC | fs.constants.O_RDONLY | fs.constants.O_SYNC);
    try {
        let read = 0;
        // tslint:disable-next-line: no-conditional-assignment
        while ((read = fs.readSync(fd, buffer, 0, BUFFER_SIZE, null)) !== 0) {
            hash.update(buffer.slice(0, read));
        }
    }
    finally {
        fs.closeSync(fd);
    }
    return `${stat.size}:${hash.digest('hex')}`;
}
function _hashField(hash, header, value) {
    hash.update(CTRL_SOH).update(header).update(CTRL_SOT).update(value).update(CTRL_ETX);
}
//# sourceMappingURL=data:application/json;base64,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