"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A set of constructs to be used as a dependable
 *
 * This class can be used when a set of constructs which are disjoint in the
 * construct tree needs to be combined to be used as a single dependable.
 *
 * @experimental
 */
class ConcreteDependable {
    constructor() {
        this._dependencyRoots = new Array();
        const self = this;
        DependableTrait.implement(this, {
            get dependencyRoots() { return self._dependencyRoots; },
        });
    }
    /**
     * Add a construct to the dependency roots
     */
    add(construct) {
        this._dependencyRoots.push(construct);
    }
}
exports.ConcreteDependable = ConcreteDependable;
const DEPENDABLE_SYMBOL = Symbol.for('@aws-cdk/core.DependableTrait');
/**
 * Trait for IDependable
 *
 * Traits are interfaces that are privately implemented by objects. Instead of
 * showing up in the public interface of a class, they need to be queried
 * explicitly. This is used to implement certain framework features that are
 * not intended to be used by Construct consumers, and so should be hidden
 * from accidental use.
 *
 * @example
 *
 * // Usage
 * const roots = DependableTrait.get(construct).dependencyRoots;
 *
 * // Definition
 * DependableTrait.implement(construct, {
 *   get dependencyRoots() { return []; }
 * });
 *
 * @experimental
 */
class DependableTrait {
    /**
     * Register `instance` to have the given DependableTrait
     *
     * Should be called in the class constructor.
     */
    static implement(instance, trait) {
        // I would also like to reference classes (to cut down on the list of objects
        // we need to manage), but we can't do that either since jsii doesn't have the
        // concept of a class reference.
        instance[DEPENDABLE_SYMBOL] = trait;
    }
    /**
     * Return the matching DependableTrait for the given class instance.
     */
    static get(instance) {
        const ret = instance[DEPENDABLE_SYMBOL];
        if (!ret) {
            throw new Error(`${instance} does not implement DependableTrait`);
        }
        return ret;
    }
}
exports.DependableTrait = DependableTrait;
//# sourceMappingURL=data:application/json;base64,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