"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const fs = require("fs");
const os = require("os");
const path = require("path");
const assets_1 = require("./assets");
const bundling_1 = require("./bundling");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        if (props.bundling) {
            this.bundleDir = this.bundle(props.bundling);
        }
        this.assetHash = this.calculateHash(props);
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.stagedPath = (_a = this.bundleDir) !== null && _a !== void 0 ? _a : this.sourcePath;
        }
        else {
            this.relativePath = `asset.${this.assetHash}${path.extname((_b = this.bundleDir) !== null && _b !== void 0 ? _b : this.sourcePath)}`;
            this.stagedPath = this.relativePath;
        }
        this.sourceHash = this.assetHash;
    }
    synthesize(session) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(session.assembly.outdir, this.relativePath);
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Asset has been bundled
        if (this.bundleDir) {
            // Try to rename bundling directory to staging directory
            try {
                fs.renameSync(this.bundleDir, targetPath);
                return;
            }
            catch (err) {
                // /tmp and cdk.out could be mounted across different mount points
                // in this case we will fallback to copying. This can happen in Windows
                // Subsystem for Linux (WSL).
                if (err.code === 'EXDEV') {
                    fs.mkdirSync(targetPath);
                    fs_1.FileSystem.copyDirectory(this.bundleDir, targetPath, this.fingerprintOptions);
                    return;
                }
                throw err;
            }
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    bundle(options) {
        var _a, _b;
        // Create temporary directory for bundling
        const bundleDir = fs.mkdtempSync(path.resolve(path.join(os.tmpdir(), 'cdk-asset-bundle-')));
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: bundling_1.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: bundling_1.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        try {
            options.image._run({
                command: options.command,
                volumes,
                environment: options.environment,
                workingDirectory: (_b = options.workingDirectory) !== null && _b !== void 0 ? _b : bundling_1.BUNDLING_INPUT_DIR,
            });
        }
        catch (err) {
            throw new Error(`Failed to run bundling Docker image for asset ${this.node.path}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            throw new Error(`Bundling did not produce any output. Check that your container writes content to ${bundling_1.BUNDLING_OUTPUT_DIR}.`);
        }
        return bundleDir;
    }
    calculateHash(props) {
        let hashType;
        if (props.assetHash) {
            if (props.assetHashType && props.assetHashType !== assets_1.AssetHashType.CUSTOM) {
                throw new Error(`Cannot specify \`${props.assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
            }
            hashType = assets_1.AssetHashType.CUSTOM;
        }
        else if (props.assetHashType) {
            hashType = props.assetHashType;
        }
        else {
            hashType = assets_1.AssetHashType.SOURCE;
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            case assets_1.AssetHashType.CUSTOM:
                if (!props.assetHash) {
                    throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
                }
                return props.assetHash;
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
//# sourceMappingURL=data:application/json;base64,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