"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const core = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cfn_parse = require("../../core/lib/cfn-parse"); // Automatically re-written from '@aws-cdk/core/lib/cfn-parse'
const cfn_type_to_l1_mapping = require("./cfn-type-to-l1-mapping");
const futils = require("./file-utils");
/**
 * Construct to import an existing CloudFormation template file into a CDK application.
 * All resources defined in the template file can be retrieved by calling the {@link getResource} method.
 * Any modifications made on the returned resource objects will be reflected in the resulting CDK template.
 */
class CfnInclude extends core.CfnElement {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        this.resources = {};
        // read the template into a JS object
        this.template = futils.readJsonSync(props.templateFile);
        // ToDo implement preserveLogicalIds=false
        this.preserveLogicalIds = true;
        // first, instantiate the conditions
        for (const conditionName of Object.keys(this.template.Conditions || {})) {
            this.createCondition(conditionName);
        }
        // then, instantiate all resources as CDK L1 objects
        for (const logicalId of Object.keys(this.template.Resources || {})) {
            this.getOrCreateResource(logicalId);
        }
    }
    /**
     * Returns the low-level CfnResource from the template with the given logical ID.
     * Any modifications performed on that resource will be reflected in the resulting CDK template.
     *
     * The returned object will be of the proper underlying class;
     * you can always cast it to the correct type in your code:
     *
     *     // assume the template contains an AWS::S3::Bucket with logical ID 'Bucket'
     *     const cfnBucket = cfnTemplate.getResource('Bucket') as s3.CfnBucket;
     *     // cfnBucket is of type s3.CfnBucket
     *
     * If the template does not contain a resource with the given logical ID,
     * an exception will be thrown.
     *
     * @param logicalId the logical ID of the resource in the CloudFormation template file
     */
    getResource(logicalId) {
        const ret = this.resources[logicalId];
        if (!ret) {
            throw new Error(`Resource with logical ID '${logicalId}' was not found in the template`);
        }
        return ret;
    }
    /**
     * Returns the CfnCondition object from the 'Conditions'
     * section of the CloudFormation template with the give name.
     * Any modifications performed on that object will be reflected in the resulting CDK template.
     *
     * If a Condition with the given name is not present in the template,
     * throws an exception.
     *
     * @param conditionName the name of the Condition in the CloudFormation template file
     */
    getCondition(conditionName) {
        const ret = this.conditions[conditionName];
        if (!ret) {
            throw new Error(`Condition with name '${conditionName}' was not found in the template`);
        }
        return ret;
    }
    /** @internal */
    _toCloudFormation() {
        const ret = {};
        for (const section of Object.keys(this.template)) {
            // render all sections of the template unchanged,
            // except Conditions and Resources, which will be taken care of by the created L1s
            if (section !== 'Conditions' && section !== 'Resources') {
                ret[section] = this.template[section];
            }
        }
        return ret;
    }
    createCondition(conditionName) {
        // ToDo condition expressions can refer to other conditions -
        // will be important when implementing preserveLogicalIds=false
        const expression = cfn_parse.FromCloudFormation.parseValue(this.template.Conditions[conditionName]);
        const cfnCondition = new core.CfnCondition(this, conditionName, {
            expression,
        });
        // ToDo handle renaming of the logical IDs of the conditions
        cfnCondition.overrideLogicalId(conditionName);
        this.conditions[conditionName] = cfnCondition;
    }
    getOrCreateResource(logicalId) {
        const ret = this.resources[logicalId];
        if (ret) {
            return ret;
        }
        const resourceAttributes = this.template.Resources[logicalId];
        const l1ClassFqn = cfn_type_to_l1_mapping.lookup(resourceAttributes.Type);
        if (!l1ClassFqn) {
            // currently, we only handle types we know the L1 for -
            // in the future, we might construct an instance of CfnResource instead
            throw new Error(`Unrecognized CloudFormation resource type: '${resourceAttributes.Type}'`);
        }
        // fail early for resource attributes we don't support yet
        const knownAttributes = ['Type', 'Properties', 'Condition', 'DependsOn', 'DeletionPolicy', 'UpdateReplacePolicy', 'Metadata'];
        for (const attribute of Object.keys(resourceAttributes)) {
            if (!knownAttributes.includes(attribute)) {
                throw new Error(`The ${attribute} resource attribute is not supported by cloudformation-include yet. ` +
                    'Either remove it from the template, or use the CdkInclude class from the core package instead.');
            }
        }
        const [moduleName, ...className] = l1ClassFqn.split('.');
        const module = require(moduleName); // eslint-disable-line @typescript-eslint/no-require-imports
        const jsClassFromModule = module[className.join('.')];
        const self = this;
        const finder = {
            findCondition(conditionName) {
                return self.conditions[conditionName];
            },
            findResource(lId) {
                if (!(lId in (self.template.Resources || {}))) {
                    return undefined;
                }
                return self.getOrCreateResource(lId);
            },
        };
        const options = {
            finder,
        };
        const l1Instance = jsClassFromModule.fromCloudFormation(this, logicalId, resourceAttributes, options);
        if (this.preserveLogicalIds) {
            // override the logical ID to match the original template
            l1Instance.overrideLogicalId(logicalId);
        }
        this.resources[logicalId] = l1Instance;
        return l1Instance;
    }
}
exports.CfnInclude = CfnInclude;
//# sourceMappingURL=data:application/json;base64,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