"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const semver = require("semver");
// eslint-disable-next-line import/no-extraneous-dependencies
const tjs = require("typescript-json-schema");
function log(message) {
    // tslint:disable-next-line:no-console
    console.log(message);
}
function bump() {
    const metadataPath = '../schema/cloud-assembly.version.json';
    // eslint-disable-next-line @typescript-eslint/no-require-imports
    const metadata = require(metadataPath);
    const oldVersion = metadata.version;
    const newVersion = semver.inc(oldVersion, 'major');
    log(`Updating schema version: ${oldVersion} -> ${newVersion}`);
    const out = path.join(__dirname, metadataPath);
    fs.writeFileSync(out, JSON.stringify({ version: newVersion }));
}
function generate(out, shouldBump) {
    const settings = {
        required: true,
        ref: true,
        topRef: true,
        noExtraProps: false,
        out,
    };
    const compilerOptions = {
        strictNullChecks: true,
    };
    const program = tjs.getProgramFromFiles([path.join(__dirname, '../lib/schema.d.ts')], compilerOptions);
    const schema = tjs.generateSchema(program, 'AssemblyManifest', settings);
    augmentDescription(schema);
    addAnyMetadataEntry(schema);
    if (shouldBump) {
        bump();
    }
    if (out) {
        log(`Generating schema to ${out}`);
        fs.writeFileSync(out, JSON.stringify(schema, null, 4));
    }
    return schema;
}
exports.generate = generate;
/**
 * Remove 'default' from the schema since its generated
 * from the tsdocs, which are not necessarily actual values,
 * but rather descriptive behavior.
 *
 * To keep this inforamtion in the schema, we append it to the
 * 'description' of the property.
 */
function augmentDescription(schema) {
    function _recurse(o) {
        for (const prop in o) {
            if (prop === 'description' && typeof o[prop] === 'string') {
                const description = o[prop];
                const defaultValue = o.default;
                if (!defaultValue) {
                    // property doesn't have a default value
                    // skip
                    continue;
                }
                const descriptionWithDefault = `${description} (Default ${defaultValue})`;
                delete o.default;
                o[prop] = descriptionWithDefault;
            }
            else if (typeof o[prop] === 'object') {
                _recurse(o[prop]);
            }
        }
    }
    _recurse(schema);
}
/**
 * Patch the properties of MetadataEntry to allow
 * specifying any free form data. This is needed since source
 * code doesn't allow this in order to enforce stricter jsii
 * compatibility checks.
 */
function addAnyMetadataEntry(schema) {
    schema.definitions.MetadataEntry.properties.data.anyOf.push({ description: 'Free form data.' });
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidXBkYXRlLXNjaGVtYS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInVwZGF0ZS1zY2hlbWEudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSx5QkFBeUI7QUFDekIsNkJBQTZCO0FBQzdCLGlDQUFpQztBQUNqQyw2REFBNkQ7QUFDN0QsOENBQThDO0FBQzlDLFNBQVMsR0FBRyxDQUFDLE9BQWU7SUFDeEIsc0NBQXNDO0lBQ3RDLE9BQU8sQ0FBQyxHQUFHLENBQUMsT0FBTyxDQUFDLENBQUM7QUFDekIsQ0FBQztBQUNELFNBQVMsSUFBSTtJQUNULE1BQU0sWUFBWSxHQUFHLHVDQUF1QyxDQUFDO0lBQzdELGlFQUFpRTtJQUNqRSxNQUFNLFFBQVEsR0FBRyxPQUFPLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDdkMsTUFBTSxVQUFVLEdBQUcsUUFBUSxDQUFDLE9BQU8sQ0FBQztJQUNwQyxNQUFNLFVBQVUsR0FBRyxNQUFNLENBQUMsR0FBRyxDQUFDLFVBQVUsRUFBRSxPQUFPLENBQUMsQ0FBQztJQUNuRCxHQUFHLENBQUMsNEJBQTRCLFVBQVUsT0FBTyxVQUFVLEVBQUUsQ0FBQyxDQUFDO0lBQy9ELE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLFlBQVksQ0FBQyxDQUFDO0lBQy9DLEVBQUUsQ0FBQyxhQUFhLENBQUMsR0FBRyxFQUFFLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxPQUFPLEVBQUUsVUFBVSxFQUFFLENBQUMsQ0FBQyxDQUFDO0FBQ25FLENBQUM7QUFDRCxTQUFnQixRQUFRLENBQUMsR0FBVyxFQUFFLFVBQW1CO0lBQ3JELE1BQU0sUUFBUSxHQUFHO1FBQ2IsUUFBUSxFQUFFLElBQUk7UUFDZCxHQUFHLEVBQUUsSUFBSTtRQUNULE1BQU0sRUFBRSxJQUFJO1FBQ1osWUFBWSxFQUFFLEtBQUs7UUFDbkIsR0FBRztLQUNOLENBQUM7SUFDRixNQUFNLGVBQWUsR0FBRztRQUNwQixnQkFBZ0IsRUFBRSxJQUFJO0tBQ3pCLENBQUM7SUFDRixNQUFNLE9BQU8sR0FBRyxHQUFHLENBQUMsbUJBQW1CLENBQUMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxvQkFBb0IsQ0FBQyxDQUFDLEVBQUUsZUFBZSxDQUFDLENBQUM7SUFDdkcsTUFBTSxNQUFNLEdBQUcsR0FBRyxDQUFDLGNBQWMsQ0FBQyxPQUFPLEVBQUUsa0JBQWtCLEVBQUUsUUFBUSxDQUFDLENBQUM7SUFDekUsa0JBQWtCLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDM0IsbUJBQW1CLENBQUMsTUFBTSxDQUFDLENBQUM7SUFDNUIsSUFBSSxVQUFVLEVBQUU7UUFDWixJQUFJLEVBQUUsQ0FBQztLQUNWO0lBQ0QsSUFBSSxHQUFHLEVBQUU7UUFDTCxHQUFHLENBQUMsd0JBQXdCLEdBQUcsRUFBRSxDQUFDLENBQUM7UUFDbkMsRUFBRSxDQUFDLGFBQWEsQ0FBQyxHQUFHLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxNQUFNLEVBQUUsSUFBSSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7S0FDMUQ7SUFDRCxPQUFPLE1BQU0sQ0FBQztBQUNsQixDQUFDO0FBdkJELDRCQXVCQztBQUNEOzs7Ozs7O0dBT0c7QUFDSCxTQUFTLGtCQUFrQixDQUFDLE1BQVc7SUFDbkMsU0FBUyxRQUFRLENBQUMsQ0FBTTtRQUNwQixLQUFLLE1BQU0sSUFBSSxJQUFJLENBQUMsRUFBRTtZQUNsQixJQUFJLElBQUksS0FBSyxhQUFhLElBQUksT0FBTyxDQUFDLENBQUMsSUFBSSxDQUFDLEtBQUssUUFBUSxFQUFFO2dCQUN2RCxNQUFNLFdBQVcsR0FBRyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUM7Z0JBQzVCLE1BQU0sWUFBWSxHQUFHLENBQUMsQ0FBQyxPQUFPLENBQUM7Z0JBQy9CLElBQUksQ0FBQyxZQUFZLEVBQUU7b0JBQ2Ysd0NBQXdDO29CQUN4QyxPQUFPO29CQUNQLFNBQVM7aUJBQ1o7Z0JBQ0QsTUFBTSxzQkFBc0IsR0FBRyxHQUFHLFdBQVcsYUFBYSxZQUFZLEdBQUcsQ0FBQztnQkFDMUUsT0FBTyxDQUFDLENBQUMsT0FBTyxDQUFDO2dCQUNqQixDQUFDLENBQUMsSUFBSSxDQUFDLEdBQUcsc0JBQXNCLENBQUM7YUFDcEM7aUJBQ0ksSUFBSSxPQUFPLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxRQUFRLEVBQUU7Z0JBQ2xDLFFBQVEsQ0FBQyxDQUFDLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQzthQUNyQjtTQUNKO0lBQ0wsQ0FBQztJQUNELFFBQVEsQ0FBQyxNQUFNLENBQUMsQ0FBQztBQUNyQixDQUFDO0FBQ0Q7Ozs7O0dBS0c7QUFDSCxTQUFTLG1CQUFtQixDQUFDLE1BQVc7SUFDcEMsTUFBTSxDQUFDLFdBQVcsQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLEVBQUUsV0FBVyxFQUFFLGlCQUFpQixFQUFFLENBQUMsQ0FBQztBQUNwRyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgZnMgZnJvbSAnZnMnO1xuaW1wb3J0ICogYXMgcGF0aCBmcm9tICdwYXRoJztcbmltcG9ydCAqIGFzIHNlbXZlciBmcm9tICdzZW12ZXInO1xuLy8gZXNsaW50LWRpc2FibGUtbmV4dC1saW5lIGltcG9ydC9uby1leHRyYW5lb3VzLWRlcGVuZGVuY2llc1xuaW1wb3J0ICogYXMgdGpzIGZyb20gJ3R5cGVzY3JpcHQtanNvbi1zY2hlbWEnO1xuZnVuY3Rpb24gbG9nKG1lc3NhZ2U6IHN0cmluZykge1xuICAgIC8vIHRzbGludDpkaXNhYmxlLW5leHQtbGluZTpuby1jb25zb2xlXG4gICAgY29uc29sZS5sb2cobWVzc2FnZSk7XG59XG5mdW5jdGlvbiBidW1wKCkge1xuICAgIGNvbnN0IG1ldGFkYXRhUGF0aCA9ICcuLi9zY2hlbWEvY2xvdWQtYXNzZW1ibHkudmVyc2lvbi5qc29uJztcbiAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25vLXJlcXVpcmUtaW1wb3J0c1xuICAgIGNvbnN0IG1ldGFkYXRhID0gcmVxdWlyZShtZXRhZGF0YVBhdGgpO1xuICAgIGNvbnN0IG9sZFZlcnNpb24gPSBtZXRhZGF0YS52ZXJzaW9uO1xuICAgIGNvbnN0IG5ld1ZlcnNpb24gPSBzZW12ZXIuaW5jKG9sZFZlcnNpb24sICdtYWpvcicpO1xuICAgIGxvZyhgVXBkYXRpbmcgc2NoZW1hIHZlcnNpb246ICR7b2xkVmVyc2lvbn0gLT4gJHtuZXdWZXJzaW9ufWApO1xuICAgIGNvbnN0IG91dCA9IHBhdGguam9pbihfX2Rpcm5hbWUsIG1ldGFkYXRhUGF0aCk7XG4gICAgZnMud3JpdGVGaWxlU3luYyhvdXQsIEpTT04uc3RyaW5naWZ5KHsgdmVyc2lvbjogbmV3VmVyc2lvbiB9KSk7XG59XG5leHBvcnQgZnVuY3Rpb24gZ2VuZXJhdGUob3V0OiBzdHJpbmcsIHNob3VsZEJ1bXA6IGJvb2xlYW4pIHtcbiAgICBjb25zdCBzZXR0aW5ncyA9IHtcbiAgICAgICAgcmVxdWlyZWQ6IHRydWUsXG4gICAgICAgIHJlZjogdHJ1ZSxcbiAgICAgICAgdG9wUmVmOiB0cnVlLFxuICAgICAgICBub0V4dHJhUHJvcHM6IGZhbHNlLFxuICAgICAgICBvdXQsXG4gICAgfTtcbiAgICBjb25zdCBjb21waWxlck9wdGlvbnMgPSB7XG4gICAgICAgIHN0cmljdE51bGxDaGVja3M6IHRydWUsXG4gICAgfTtcbiAgICBjb25zdCBwcm9ncmFtID0gdGpzLmdldFByb2dyYW1Gcm9tRmlsZXMoW3BhdGguam9pbihfX2Rpcm5hbWUsICcuLi9saWIvc2NoZW1hLmQudHMnKV0sIGNvbXBpbGVyT3B0aW9ucyk7XG4gICAgY29uc3Qgc2NoZW1hID0gdGpzLmdlbmVyYXRlU2NoZW1hKHByb2dyYW0sICdBc3NlbWJseU1hbmlmZXN0Jywgc2V0dGluZ3MpO1xuICAgIGF1Z21lbnREZXNjcmlwdGlvbihzY2hlbWEpO1xuICAgIGFkZEFueU1ldGFkYXRhRW50cnkoc2NoZW1hKTtcbiAgICBpZiAoc2hvdWxkQnVtcCkge1xuICAgICAgICBidW1wKCk7XG4gICAgfVxuICAgIGlmIChvdXQpIHtcbiAgICAgICAgbG9nKGBHZW5lcmF0aW5nIHNjaGVtYSB0byAke291dH1gKTtcbiAgICAgICAgZnMud3JpdGVGaWxlU3luYyhvdXQsIEpTT04uc3RyaW5naWZ5KHNjaGVtYSwgbnVsbCwgNCkpO1xuICAgIH1cbiAgICByZXR1cm4gc2NoZW1hO1xufVxuLyoqXG4gKiBSZW1vdmUgJ2RlZmF1bHQnIGZyb20gdGhlIHNjaGVtYSBzaW5jZSBpdHMgZ2VuZXJhdGVkXG4gKiBmcm9tIHRoZSB0c2RvY3MsIHdoaWNoIGFyZSBub3QgbmVjZXNzYXJpbHkgYWN0dWFsIHZhbHVlcyxcbiAqIGJ1dCByYXRoZXIgZGVzY3JpcHRpdmUgYmVoYXZpb3IuXG4gKlxuICogVG8ga2VlcCB0aGlzIGluZm9yYW10aW9uIGluIHRoZSBzY2hlbWEsIHdlIGFwcGVuZCBpdCB0byB0aGVcbiAqICdkZXNjcmlwdGlvbicgb2YgdGhlIHByb3BlcnR5LlxuICovXG5mdW5jdGlvbiBhdWdtZW50RGVzY3JpcHRpb24oc2NoZW1hOiBhbnkpIHtcbiAgICBmdW5jdGlvbiBfcmVjdXJzZShvOiBhbnkpIHtcbiAgICAgICAgZm9yIChjb25zdCBwcm9wIGluIG8pIHtcbiAgICAgICAgICAgIGlmIChwcm9wID09PSAnZGVzY3JpcHRpb24nICYmIHR5cGVvZiBvW3Byb3BdID09PSAnc3RyaW5nJykge1xuICAgICAgICAgICAgICAgIGNvbnN0IGRlc2NyaXB0aW9uID0gb1twcm9wXTtcbiAgICAgICAgICAgICAgICBjb25zdCBkZWZhdWx0VmFsdWUgPSBvLmRlZmF1bHQ7XG4gICAgICAgICAgICAgICAgaWYgKCFkZWZhdWx0VmFsdWUpIHtcbiAgICAgICAgICAgICAgICAgICAgLy8gcHJvcGVydHkgZG9lc24ndCBoYXZlIGEgZGVmYXVsdCB2YWx1ZVxuICAgICAgICAgICAgICAgICAgICAvLyBza2lwXG4gICAgICAgICAgICAgICAgICAgIGNvbnRpbnVlO1xuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBjb25zdCBkZXNjcmlwdGlvbldpdGhEZWZhdWx0ID0gYCR7ZGVzY3JpcHRpb259IChEZWZhdWx0ICR7ZGVmYXVsdFZhbHVlfSlgO1xuICAgICAgICAgICAgICAgIGRlbGV0ZSBvLmRlZmF1bHQ7XG4gICAgICAgICAgICAgICAgb1twcm9wXSA9IGRlc2NyaXB0aW9uV2l0aERlZmF1bHQ7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICBlbHNlIGlmICh0eXBlb2Ygb1twcm9wXSA9PT0gJ29iamVjdCcpIHtcbiAgICAgICAgICAgICAgICBfcmVjdXJzZShvW3Byb3BdKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgfVxuICAgIH1cbiAgICBfcmVjdXJzZShzY2hlbWEpO1xufVxuLyoqXG4gKiBQYXRjaCB0aGUgcHJvcGVydGllcyBvZiBNZXRhZGF0YUVudHJ5IHRvIGFsbG93XG4gKiBzcGVjaWZ5aW5nIGFueSBmcmVlIGZvcm0gZGF0YS4gVGhpcyBpcyBuZWVkZWQgc2luY2Ugc291cmNlXG4gKiBjb2RlIGRvZXNuJ3QgYWxsb3cgdGhpcyBpbiBvcmRlciB0byBlbmZvcmNlIHN0cmljdGVyIGpzaWlcbiAqIGNvbXBhdGliaWxpdHkgY2hlY2tzLlxuICovXG5mdW5jdGlvbiBhZGRBbnlNZXRhZGF0YUVudHJ5KHNjaGVtYTogYW55KSB7XG4gICAgc2NoZW1hLmRlZmluaXRpb25zLk1ldGFkYXRhRW50cnkucHJvcGVydGllcy5kYXRhLmFueU9mLnB1c2goeyBkZXNjcmlwdGlvbjogJ0ZyZWUgZm9ybSBkYXRhLicgfSk7XG59XG4iXX0=