"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const jsonschema = require("jsonschema");
const semver = require("semver");
const metadata_schema_1 = require("./metadata-schema");
const assembly = require("./schema");
// this prefix is used by the CLI to identify this specific error.
// in which case we want to instruct the user to upgrade his CLI.
// see exec.ts#createAssembly
exports.VERSION_MISMATCH = 'Cloud assembly schema version mismatch';
/**
 * Protocol utility class.
 */
class Manifest {
    constructor() { }
    /**
     * Save manifest to file.
     *
     * @param manifest - manifest.
     */
    static save(manifest, filePath) {
        fs.writeFileSync(filePath, JSON.stringify(manifest, undefined, 2));
    }
    /**
     * Load manifest from file.
     *
     * @param filePath - path to the manifest file.
     */
    static load(filePath) {
        const raw = JSON.parse(fs.readFileSync(filePath, 'UTF-8'));
        Manifest.patchStackTags(raw);
        Manifest.validate(raw);
        return raw;
    }
    /**
     * Fetch the current schema version number.
     */
    static version() {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        return require('../schema/cloud-assembly.version.json').version;
    }
    static validate(manifest) {
        function parseVersion(version) {
            const ver = semver.valid(version);
            if (!ver) {
                throw new Error(`Invalid semver string: "${version}"`);
            }
            return ver;
        }
        const maxSupported = parseVersion(Manifest.version());
        const actual = parseVersion(manifest.version);
        // first validate the version should be accepted.
        if (semver.gt(actual, maxSupported)) {
            // we use a well known error prefix so that the CLI can identify this specific error
            // and print some more context to the user.
            throw new Error(`${exports.VERSION_MISMATCH}: Maximum schema version supported is ${maxSupported}, but found ${actual}`);
        }
        // now validate the format is good.
        const validator = new jsonschema.Validator();
        const result = validator.validate(manifest, Manifest.schema, {
            // does exist but is not in the TypeScript definitions
            nestedErrors: true,
            allowUnknownAttributes: false,
        });
        if (!result.valid) {
            throw new Error(`Invalid assembly manifest:\n${result}`);
        }
    }
    /**
     * This requires some explaining...
     *
     * We previously used `{ Key, Value }` for the object that represents a stack tag. (Notice the casing)
     * @link https://github.com/aws/aws-cdk/blob/v1.27.0/packages/aws-cdk/lib/api/cxapp/stacks.ts#L427.
     *
     * When that object moved to this package, it had to be JSII compliant, which meant the property
     * names must be `camelCased`, and not `PascalCased`. This meant it no longer matches the structure in the `manifest.json` file.
     * In order to support current manifest files, we have to translate the `PascalCased` representation to the new `camelCased` one.
     *
     * Note that the serialization itself still writes `PascalCased` because it relates to how CloudFormation expects it.
     *
     * Ideally, we would start writing the `camelCased` and translate to how CloudFormation expects it when needed. But this requires nasty
     * backwards-compatibility code and it just doesn't seem to be worth the effort.
     */
    static patchStackTags(manifest) {
        for (const artifact of Object.values(manifest.artifacts || [])) {
            if (artifact.type === assembly.ArtifactType.AWS_CLOUDFORMATION_STACK) {
                for (const metadataEntries of Object.values(artifact.metadata || [])) {
                    for (const metadataEntry of metadataEntries) {
                        if (metadataEntry.type === metadata_schema_1.ArtifactMetadataEntryType.STACK_TAGS && metadataEntry.data) {
                            const metadataAny = metadataEntry;
                            metadataAny.data = metadataAny.data.map((t) => {
                                return { key: t.Key, value: t.Value };
                            });
                        }
                    }
                }
            }
        }
    }
}
exports.Manifest = Manifest;
// eslint-disable-next-line @typescript-eslint/no-require-imports
Manifest.schema = require('../schema/cloud-assembly.schema.json');
//# sourceMappingURL=data:application/json;base64,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