"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const file_asset_1 = require("../file-asset");
/**
 * Validate that a given key is of a given type in an object
 *
 * If not optional, the key is considered required.
 *
 * Uses predicate validators that return a 'witness', so we can use the return type of
 * the validator function to infer the actual type of the value and enrich the type information
 * of the given input field.
 *
 * In effect, validators should be written like this:
 *
 *   if (!valid(input)) { throw; }
 *   return input;
 */
function expectKey(obj, key, validate, optional) {
    if (typeof obj !== 'object' || obj === null || (!(key in obj) && !optional)) {
        throw new Error(`Expected key '${key}' missing: ${JSON.stringify(obj)}`);
    }
    if (key in obj) {
        try {
            validate(obj[key]);
        }
        catch (e) {
            throw new Error(`${key}: ${e.message}`);
        }
    }
}
exports.expectKey = expectKey;
function isString(x) {
    if (typeof x !== 'string') {
        throw new Error(`Expected a string, got '${x}'`);
    }
    return x;
}
exports.isString = isString;
function isMapOf(pred) {
    return x => {
        assertIsObject(x);
        Object.values(x).forEach(pred);
        return x;
    };
}
exports.isMapOf = isMapOf;
function isObjectAnd(p) {
    return x => {
        assertIsObject(x);
        return p(x);
    };
}
exports.isObjectAnd = isObjectAnd;
function assertIsObject(x) {
    if (typeof x !== 'object' || x === null) {
        throw new Error(`Expected a map, got '${x}'`);
    }
}
exports.assertIsObject = assertIsObject;
function isFileAssetPackaging(x) {
    const str = isString(x);
    const validValues = Object.values(file_asset_1.FileAssetPackaging); // Explicit cast needed because this is a string-valued enum
    if (!validValues.includes(str)) {
        throw new Error(`Expected a FileAssetPackaging (one of ${validValues.map(v => `'${v}'`).join(', ')}), got '${str}'`);
    }
    return x;
}
exports.isFileAssetPackaging = isFileAssetPackaging;
//# sourceMappingURL=data:application/json;base64,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