"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chain_1 = require("../chain");
const state_type_1 = require("./private/state-type");
const state_1 = require("./state");
/**
 * Define a Choice in the state machine
 *
 * A choice state can be used to make decisions based on the execution
 * state.
 */
class Choice extends state_1.State {
    constructor(scope, id, props = {}) {
        super(scope, id, props);
        this.endStates = [];
    }
    /**
     * If the given condition matches, continue execution with the given state
     */
    when(condition, next) {
        super.addChoice(condition, next.startState);
        return this;
    }
    /**
     * If none of the given conditions match, continue execution with the given state
     *
     * If no conditions match and no otherwise() has been given, an execution
     * error will be raised.
     */
    otherwise(def) {
        super.makeDefault(def.startState);
        return this;
    }
    /**
     * Return a Chain that contains all reachable end states from this Choice
     *
     * Use this to combine all possible choice paths back.
     */
    afterwards(options = {}) {
        const endStates = state_1.State.filterNextables(state_1.State.findReachableEndStates(this, { includeErrorHandlers: options.includeErrorHandlers }));
        if (options.includeOtherwise && this.defaultChoice) {
            throw new Error(`'includeOtherwise' set but Choice state ${this.stateId} already has an 'otherwise' transition`);
        }
        if (options.includeOtherwise) {
            endStates.push(new DefaultAsNext(this));
        }
        return chain_1.Chain.custom(this, endStates, this);
    }
    /**
     * Return the Amazon States Language object for this state
     */
    toStateJson() {
        return {
            Type: state_type_1.StateType.CHOICE,
            Comment: this.comment,
            ...this.renderInputOutput(),
            ...this.renderChoices(),
        };
    }
}
exports.Choice = Choice;
/**
 * Adapter to make the .otherwise() transition settable through .next()
 */
class DefaultAsNext {
    constructor(choice) {
        this.choice = choice;
    }
    next(state) {
        this.choice.otherwise(state);
        return chain_1.Chain.sequence(this.choice, state);
    }
}
//# sourceMappingURL=data:application/json;base64,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