"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const chain_1 = require("./chain");
const parallel_1 = require("./states/parallel");
const state_1 = require("./states/state");
/**
 * Base class for reusable state machine fragments
 */
class StateMachineFragment extends cdk.Construct {
    get id() {
        return this.node.id;
    }
    /**
     * Prefix the IDs of all states in this state machine fragment
     *
     * Use this to avoid multiple copies of the state machine all having the
     * same state IDs.
     *
     * @param prefix The prefix to add. Will use construct ID by default.
     */
    prefixStates(prefix) {
        state_1.State.prefixStates(this, prefix || `${this.id}: `);
        return this;
    }
    /**
     * Wrap all states in this state machine fragment up into a single state.
     *
     * This can be used to add retry or error handling onto this state
     * machine fragment.
     *
     * Be aware that this changes the result of the inner state machine
     * to be an array with the result of the state machine in it. Adjust
     * your paths accordingly. For example, change 'outputPath' to
     * '$[0]'.
     */
    toSingleState(options = {}) {
        const stateId = options.stateId || this.id;
        this.prefixStates(options.prefixStates || `${stateId}: `);
        return new parallel_1.Parallel(this, stateId, options).branch(this);
    }
    /**
     * Continue normal execution with the given state
     */
    next(next) {
        return chain_1.Chain.start(this).next(next);
    }
}
exports.StateMachineFragment = StateMachineFragment;
//# sourceMappingURL=data:application/json;base64,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