"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fields_1 = require("./fields");
/**
 * Type union for task classes that accept multiple types of payload
 */
class TaskInput {
    /**
     *
     * @param type type of task input
     * @param value payload for the corresponding input type.
     * It can be a JSON-encoded object, context, data, etc.
     */
    constructor(type, value) {
        this.type = type;
        this.value = value;
    }
    /**
     * Use a literal string as task input
     *
     * This might be a JSON-encoded object, or just a text.
     */
    static fromText(text) {
        return new TaskInput(InputType.TEXT, text);
    }
    /**
     * Use an object as task input
     *
     * This object may contain Data and Context fields
     * as object values, if desired.
     */
    static fromObject(obj) {
        return new TaskInput(InputType.OBJECT, obj);
    }
    /**
     * Use a part of the execution data as task input
     *
     * Use this when you want to use a subobject or string from
     * the current state machine execution as complete payload
     * to a task.
     */
    static fromDataAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Data.stringAt(path));
    }
    /**
     * Use a part of the task context as task input
     *
     * Use this when you want to use a subobject or string from
     * the current task context as complete payload
     * to a task.
     */
    static fromContextAt(path) {
        return new TaskInput(InputType.TEXT, fields_1.Context.stringAt(path));
    }
}
exports.TaskInput = TaskInput;
/**
 * The type of task input
 */
var InputType;
(function (InputType) {
    /**
     * Use a literal string
     * This might be a JSON-encoded object, or just text.
     * valid JSON text: standalone, quote-delimited strings; objects; arrays; numbers; Boolean values; and null.
     *
     * example: `literal string`
     * example: {"json": "encoded"}
     */
    InputType[InputType["TEXT"] = 0] = "TEXT";
    /**
     * Use an object which may contain Data and Context fields
     * as object values, if desired.
     *
     * example:
     * {
     *  literal: 'literal',
     *  SomeInput: sfn.Data.stringAt('$.someField')
     * }
     *
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/concepts-state-machine-data.html
     * @see https://docs.aws.amazon.com/step-functions/latest/dg/input-output-contextobject.html
     */
    InputType[InputType["OBJECT"] = 1] = "OBJECT";
})(InputType = exports.InputType || (exports.InputType = {}));
//# sourceMappingURL=data:application/json;base64,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