"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * A Condition for use in a Choice state branch
 */
class Condition {
    /**
     * Matches if a boolean field has the given value
     */
    static booleanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.BooleanEquals, value);
    }
    /**
     * Matches if a string field has the given value
     */
    static stringEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringEquals, value);
    }
    /**
     * Matches if a string field sorts before a given value
     */
    static stringLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThan, value);
    }
    /**
     * Matches if a string field sorts equal to or before a given value
     */
    static stringLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringLessThanEquals, value);
    }
    /**
     * Matches if a string field sorts after a given value
     */
    static stringGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThan, value);
    }
    /**
     * Matches if a string field sorts after or equal to a given value
     */
    static stringGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.StringGreaterThanEquals, value);
    }
    /**
     * Matches if a numeric field has the given value
     */
    static numberEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericEquals, value);
    }
    /**
     * Matches if a numeric field is less than the given value
     */
    static numberLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThan, value);
    }
    /**
     * Matches if a numeric field is less than or equal to the given value
     */
    static numberLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericLessThanEquals, value);
    }
    /**
     * Matches if a numeric field is greater than the given value
     */
    static numberGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThan, value);
    }
    /**
     * Matches if a numeric field is greater than or equal to the given value
     */
    static numberGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.NumericGreaterThanEquals, value);
    }
    /**
     * Matches if a timestamp field is the same time as the given timestamp
     */
    static timestampEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampEquals, value);
    }
    /**
     * Matches if a timestamp field is before the given timestamp
     */
    static timestampLessThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThan, value);
    }
    /**
     * Matches if a timestamp field is before or equal to the given timestamp
     */
    static timestampLessThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampLessThanEquals, value);
    }
    /**
     * Matches if a timestamp field is after the given timestamp
     */
    static timestampGreaterThan(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThan, value);
    }
    /**
     * Matches if a timestamp field is after or equal to the given timestamp
     */
    static timestampGreaterThanEquals(variable, value) {
        return new VariableComparison(variable, ComparisonOperator.TimestampGreaterThanEquals, value);
    }
    /**
     * Combine two or more conditions with a logical AND
     */
    static and(...conditions) {
        return new CompoundCondition(CompoundOperator.And, ...conditions);
    }
    /**
     * Combine two or more conditions with a logical OR
     */
    static or(...conditions) {
        return new CompoundCondition(CompoundOperator.Or, ...conditions);
    }
    /**
     * Negate a condition
     */
    static not(condition) {
        return new NotCondition(condition);
    }
}
exports.Condition = Condition;
/**
 * Comparison Operator types
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    ComparisonOperator[ComparisonOperator["StringEquals"] = 0] = "StringEquals";
    ComparisonOperator[ComparisonOperator["StringLessThan"] = 1] = "StringLessThan";
    ComparisonOperator[ComparisonOperator["StringGreaterThan"] = 2] = "StringGreaterThan";
    ComparisonOperator[ComparisonOperator["StringLessThanEquals"] = 3] = "StringLessThanEquals";
    ComparisonOperator[ComparisonOperator["StringGreaterThanEquals"] = 4] = "StringGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["NumericEquals"] = 5] = "NumericEquals";
    ComparisonOperator[ComparisonOperator["NumericLessThan"] = 6] = "NumericLessThan";
    ComparisonOperator[ComparisonOperator["NumericGreaterThan"] = 7] = "NumericGreaterThan";
    ComparisonOperator[ComparisonOperator["NumericLessThanEquals"] = 8] = "NumericLessThanEquals";
    ComparisonOperator[ComparisonOperator["NumericGreaterThanEquals"] = 9] = "NumericGreaterThanEquals";
    ComparisonOperator[ComparisonOperator["BooleanEquals"] = 10] = "BooleanEquals";
    ComparisonOperator[ComparisonOperator["TimestampEquals"] = 11] = "TimestampEquals";
    ComparisonOperator[ComparisonOperator["TimestampLessThan"] = 12] = "TimestampLessThan";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThan"] = 13] = "TimestampGreaterThan";
    ComparisonOperator[ComparisonOperator["TimestampLessThanEquals"] = 14] = "TimestampLessThanEquals";
    ComparisonOperator[ComparisonOperator["TimestampGreaterThanEquals"] = 15] = "TimestampGreaterThanEquals";
})(ComparisonOperator || (ComparisonOperator = {}));
/**
 * Compound Operator types
 */
var CompoundOperator;
(function (CompoundOperator) {
    CompoundOperator[CompoundOperator["And"] = 0] = "And";
    CompoundOperator[CompoundOperator["Or"] = 1] = "Or";
})(CompoundOperator || (CompoundOperator = {}));
/**
 * Scalar comparison
 */
class VariableComparison extends Condition {
    constructor(variable, comparisonOperator, value) {
        super();
        this.variable = variable;
        this.comparisonOperator = comparisonOperator;
        this.value = value;
        if (!/^\$[.[]/.test(variable)) {
            throw new Error(`Variable reference must start with '$.' or '$[', got '${variable}'`);
        }
    }
    renderCondition() {
        return {
            Variable: this.variable,
            [ComparisonOperator[this.comparisonOperator]]: this.value,
        };
    }
}
/**
 * Logical compound condition
 */
class CompoundCondition extends Condition {
    constructor(operator, ...conditions) {
        super();
        this.operator = operator;
        this.conditions = conditions;
        if (conditions.length === 0) {
            throw new Error('Must supply at least one inner condition for a logical combination');
        }
    }
    renderCondition() {
        return {
            [CompoundOperator[this.operator]]: this.conditions.map(c => c.renderCondition()),
        };
    }
}
/**
 * Logical unary condition
 */
class NotCondition extends Condition {
    constructor(comparisonOperation) {
        super();
        this.comparisonOperation = comparisonOperation;
    }
    renderCondition() {
        return {
            Not: this.comparisonOperation.renderCondition(),
        };
    }
}
//# sourceMappingURL=data:application/json;base64,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