"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const task_utils_1 = require("../private/task-utils");
const base_types_1 = require("./base-types");
/**
 * Class representing the SageMaker Create Training Job task.
 *
 *  @experimental
 */
class SageMakerCreateTransformJob extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS);
        // set the sagemaker role or create new one
        if (props.role) {
            this._role = props.role;
        }
        // set the S3 Data type of the input data config objects to be 'S3Prefix' if not defined
        this.transformInput = props.transformInput.transformDataSource.s3DataSource.s3DataType
            ? props.transformInput
            : Object.assign({}, props.transformInput, {
                transformDataSource: { s3DataSource: { ...props.transformInput.transformDataSource.s3DataSource, s3DataType: base_types_1.S3DataType.S3_PREFIX } },
            });
        // set the default value for the transform resources
        this.transformResources = props.transformResources || {
            instanceCount: 1,
            instanceType: ec2.InstanceType.of(ec2.InstanceClass.M4, ec2.InstanceSize.XLARGE),
        };
        this.taskPolicies = this.makePolicyStatements();
    }
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('sagemaker', 'createTransformJob', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject(this.renderParameters()),
        };
    }
    /**
     * The execution role for the Sagemaker training job.
     *
     * Only available after task has been added to a state machine.
     */
    get role() {
        if (this._role === undefined) {
            throw new Error('role not available yet--use the object in a Task first');
        }
        return this._role;
    }
    renderParameters() {
        return {
            ...(this.props.batchStrategy ? { BatchStrategy: this.props.batchStrategy } : {}),
            ...this.renderEnvironment(this.props.environment),
            ...(this.props.maxConcurrentTransforms ? { MaxConcurrentTransforms: this.props.maxConcurrentTransforms } : {}),
            ...(this.props.maxPayload ? { MaxPayloadInMB: this.props.maxPayload.toMebibytes() } : {}),
            ModelName: this.props.modelName,
            ...this.renderTags(this.props.tags),
            ...this.renderTransformInput(this.transformInput),
            TransformJobName: this.props.transformJobName,
            ...this.renderTransformOutput(this.props.transformOutput),
            ...this.renderTransformResources(this.transformResources),
        };
    }
    renderTransformInput(input) {
        return {
            TransformInput: {
                ...(input.compressionType ? { CompressionType: input.compressionType } : {}),
                ...(input.contentType ? { ContentType: input.contentType } : {}),
                DataSource: {
                    S3DataSource: {
                        S3Uri: input.transformDataSource.s3DataSource.s3Uri,
                        S3DataType: input.transformDataSource.s3DataSource.s3DataType,
                    },
                },
                ...(input.splitType ? { SplitType: input.splitType } : {}),
            },
        };
    }
    renderTransformOutput(output) {
        return {
            TransformOutput: {
                S3OutputPath: output.s3OutputPath,
                ...(output.encryptionKey ? { KmsKeyId: output.encryptionKey.keyArn } : {}),
                ...(output.accept ? { Accept: output.accept } : {}),
                ...(output.assembleWith ? { AssembleWith: output.assembleWith } : {}),
            },
        };
    }
    renderTransformResources(resources) {
        return {
            TransformResources: {
                InstanceCount: resources.instanceCount,
                InstanceType: 'ml.' + resources.instanceType,
                ...(resources.volumeEncryptionKey ? { VolumeKmsKeyId: resources.volumeEncryptionKey.keyArn } : {}),
            },
        };
    }
    renderEnvironment(environment) {
        return environment ? { Environment: environment } : {};
    }
    renderTags(tags) {
        return tags ? { Tags: Object.keys(tags).map((key) => ({ Key: key, Value: tags[key] })) } : {};
    }
    makePolicyStatements() {
        const stack = core_1.Stack.of(this);
        // create new role if doesn't exist
        if (this._role === undefined) {
            this._role = new iam.Role(this, 'SagemakerTransformRole', {
                assumedBy: new iam.ServicePrincipal('sagemaker.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('AmazonSageMakerFullAccess')],
            });
        }
        // https://docs.aws.amazon.com/step-functions/latest/dg/sagemaker-iam.html
        const policyStatements = [
            new iam.PolicyStatement({
                actions: ['sagemaker:CreateTransformJob', 'sagemaker:DescribeTransformJob', 'sagemaker:StopTransformJob'],
                resources: [
                    stack.formatArn({
                        service: 'sagemaker',
                        resource: 'transform-job',
                        resourceName: '*',
                    }),
                ],
            }),
            new iam.PolicyStatement({
                actions: ['sagemaker:ListTags'],
                resources: ['*'],
            }),
            new iam.PolicyStatement({
                actions: ['iam:PassRole'],
                resources: [this.role.roleArn],
                conditions: {
                    StringEquals: { 'iam:PassedToService': 'sagemaker.amazonaws.com' },
                },
            }),
        ];
        if (this.integrationPattern === sfn.IntegrationPattern.RUN_JOB) {
            policyStatements.push(new iam.PolicyStatement({
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
                resources: [
                    stack.formatArn({
                        service: 'events',
                        resource: 'rule',
                        resourceName: 'StepFunctionsGetEventsForSageMakerTransformJobsRule',
                    }),
                ],
            }));
        }
        return policyStatements;
    }
}
exports.SageMakerCreateTransformJob = SageMakerCreateTransformJob;
SageMakerCreateTransformJob.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.RUN_JOB,
];
//# sourceMappingURL=data:application/json;base64,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