"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const aws_ecr_assets_1 = require("../../../aws-ecr-assets"); // Automatically re-written from '@aws-cdk/aws-ecr-assets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
/**
 * Constructs `IS3Location` objects.
 *
 * @experimental
 */
class S3Location {
    /**
     * An `IS3Location` built with a determined bucket and key prefix.
     *
     * @param bucket    is the bucket where the objects are to be stored.
     * @param keyPrefix is the key prefix used by the location.
     */
    static fromBucket(bucket, keyPrefix) {
        return new StandardS3Location({ bucket, keyPrefix, uri: bucket.urlForObject(keyPrefix) });
    }
    /**
     * An `IS3Location` determined fully by a JSON Path from the task input.
     *
     * Due to the dynamic nature of those locations, the IAM grants that will be set by `grantRead` and `grantWrite`
     * apply to the `*` resource.
     *
     * @param expression the JSON expression resolving to an S3 location URI.
     */
    static fromJsonExpression(expression) {
        return new StandardS3Location({ uri: sfn.Data.stringAt(expression) });
    }
}
exports.S3Location = S3Location;
/**
 * Creates `IDockerImage` instances.
 *
 * @experimental
 */
class DockerImage {
    /**
     * Reference a Docker image stored in an ECR repository.
     *
     * @param repository the ECR repository where the image is hosted.
     * @param tag an optional `tag`
     */
    static fromEcrRepository(repository, tag = 'latest') {
        return new StandardDockerImage({ repository, imageUri: repository.repositoryUriForTag(tag) });
    }
    /**
     * Reference a Docker image which URI is obtained from the task's input.
     *
     * @param expression           the JSON path expression with the task input.
     * @param allowAnyEcrImagePull whether ECR access should be permitted (set to `false` if the image will never be in ECR).
     */
    static fromJsonExpression(expression, allowAnyEcrImagePull = true) {
        return new StandardDockerImage({ imageUri: expression, allowAnyEcrImagePull });
    }
    /**
     * Reference a Docker image by it's URI.
     *
     * When referencing ECR images, prefer using `inEcr`.
     *
     * @param imageUri the URI to the docker image.
     */
    static fromRegistry(imageUri) {
        return new StandardDockerImage({ imageUri });
    }
    /**
     * Reference a Docker image that is provided as an Asset in the current app.
     *
     * @param scope the scope in which to create the Asset.
     * @param id    the ID for the asset in the construct tree.
     * @param props the configuration props of the asset.
     */
    static fromAsset(scope, id, props) {
        const asset = new aws_ecr_assets_1.DockerImageAsset(scope, id, props);
        return new StandardDockerImage({ repository: asset.repository, imageUri: asset.imageUri });
    }
}
exports.DockerImage = DockerImage;
/**
 * S3 Data Type.
 *
 * @experimental
 */
var S3DataType;
(function (S3DataType) {
    /**
     * Manifest File Data Type
     */
    S3DataType["MANIFEST_FILE"] = "ManifestFile";
    /**
     * S3 Prefix Data Type
     */
    S3DataType["S3_PREFIX"] = "S3Prefix";
    /**
     * Augmented Manifest File Data Type
     */
    S3DataType["AUGMENTED_MANIFEST_FILE"] = "AugmentedManifestFile";
})(S3DataType = exports.S3DataType || (exports.S3DataType = {}));
/**
 * S3 Data Distribution Type.
 *
 * @experimental
 */
var S3DataDistributionType;
(function (S3DataDistributionType) {
    /**
     * Fully replicated S3 Data Distribution Type
     */
    S3DataDistributionType["FULLY_REPLICATED"] = "FullyReplicated";
    /**
     * Sharded By S3 Key Data Distribution Type
     */
    S3DataDistributionType["SHARDED_BY_S3_KEY"] = "ShardedByS3Key";
})(S3DataDistributionType = exports.S3DataDistributionType || (exports.S3DataDistributionType = {}));
/**
 * Define the format of the input data.
 *
 * @experimental
 */
var RecordWrapperType;
(function (RecordWrapperType) {
    /**
     * None record wrapper type
     */
    RecordWrapperType["NONE"] = "None";
    /**
     * RecordIO record wrapper type
     */
    RecordWrapperType["RECORD_IO"] = "RecordIO";
})(RecordWrapperType = exports.RecordWrapperType || (exports.RecordWrapperType = {}));
/**
 *  Input mode that the algorithm supports.
 *
 * @experimental
 */
var InputMode;
(function (InputMode) {
    /**
     * Pipe mode
     */
    InputMode["PIPE"] = "Pipe";
    /**
     * File mode.
     */
    InputMode["FILE"] = "File";
})(InputMode = exports.InputMode || (exports.InputMode = {}));
/**
 * Compression type of the data.
 *
 * @experimental
 */
var CompressionType;
(function (CompressionType) {
    /**
     * None compression type
     */
    CompressionType["NONE"] = "None";
    /**
     * Gzip compression type
     */
    CompressionType["GZIP"] = "Gzip";
})(CompressionType = exports.CompressionType || (exports.CompressionType = {}));
/**
 * Specifies the number of records to include in a mini-batch for an HTTP inference request.
 *
 * @experimental
 */
var BatchStrategy;
(function (BatchStrategy) {
    /**
     * Fits multiple records in a mini-batch.
     */
    BatchStrategy["MULTI_RECORD"] = "MultiRecord";
    /**
     * Use a single record when making an invocation request.
     */
    BatchStrategy["SINGLE_RECORD"] = "SingleRecord";
})(BatchStrategy = exports.BatchStrategy || (exports.BatchStrategy = {}));
/**
 * Method to use to split the transform job's data files into smaller batches.
 *
 * @experimental
 */
var SplitType;
(function (SplitType) {
    /**
     * Input data files are not split,
     */
    SplitType["NONE"] = "None";
    /**
     * Split records on a newline character boundary.
     */
    SplitType["LINE"] = "Line";
    /**
     * Split using MXNet RecordIO format.
     */
    SplitType["RECORD_IO"] = "RecordIO";
    /**
     * Split using TensorFlow TFRecord format.
     */
    SplitType["TF_RECORD"] = "TFRecord";
})(SplitType = exports.SplitType || (exports.SplitType = {}));
/**
 * How to assemble the results of the transform job as a single S3 object.
 *
 * @experimental
 */
var AssembleWith;
(function (AssembleWith) {
    /**
     * Concatenate the results in binary format.
     */
    AssembleWith["NONE"] = "None";
    /**
     * Add a newline character at the end of every transformed record.
     */
    AssembleWith["LINE"] = "Line";
})(AssembleWith = exports.AssembleWith || (exports.AssembleWith = {}));
class StandardDockerImage extends DockerImage {
    constructor(opts) {
        super();
        this.allowAnyEcrImagePull = !!opts.allowAnyEcrImagePull;
        this.imageUri = opts.imageUri;
        this.repository = opts.repository;
    }
    bind(task) {
        if (this.repository) {
            this.repository.grantPull(task);
        }
        if (this.allowAnyEcrImagePull) {
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({
                actions: [
                    'ecr:BatchCheckLayerAvailability',
                    'ecr:GetDownloadUrlForLayer',
                    'ecr:BatchGetImage',
                ],
                resources: ['*'],
            }));
        }
        return {
            imageUri: this.imageUri,
        };
    }
}
class StandardS3Location extends S3Location {
    constructor(opts) {
        super();
        this.bucket = opts.bucket;
        this.keyGlob = `${opts.keyPrefix || ''}*`;
        this.uri = opts.uri;
    }
    bind(task, opts) {
        if (this.bucket) {
            if (opts.forReading) {
                this.bucket.grantRead(task, this.keyGlob);
            }
            if (opts.forWriting) {
                this.bucket.grantWrite(task, this.keyGlob);
            }
        }
        else {
            const actions = new Array();
            if (opts.forReading) {
                actions.push('s3:GetObject', 's3:ListBucket');
            }
            if (opts.forWriting) {
                actions.push('s3:PutObject');
            }
            task.grantPrincipal.addToPolicy(new iam.PolicyStatement({ actions, resources: ['*'] }));
        }
        return { uri: this.uri };
    }
}
//# sourceMappingURL=data:application/json;base64,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