"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Invoke a Lambda function as a Task
 *
 * OUTPUT: the output of this task is either the return value of Lambda's
 * Invoke call, or whatever the Lambda Function posted back using
 * `SendTaskSuccess/SendTaskFailure` in `waitForTaskToken` mode.
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 * @deprecated Use `LambdaInvoke`
 */
class RunLambdaTask {
    constructor(lambdaFunction, props = {}) {
        this.lambdaFunction = lambdaFunction;
        this.props = props;
        this.integrationPattern = props.integrationPattern || sfn.ServiceIntegrationPattern.FIRE_AND_FORGET;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call Lambda.`);
        }
        if (this.integrationPattern === sfn.ServiceIntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is missing in payload (pass Context.taskToken somewhere in payload)');
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('lambda', 'invoke', this.integrationPattern),
            policyStatements: [new iam.PolicyStatement({
                    resources: [this.lambdaFunction.functionArn],
                    actions: ['lambda:InvokeFunction'],
                })],
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: { LambdaFunctionArn: this.lambdaFunction.functionArn },
            parameters: {
                FunctionName: this.lambdaFunction.functionName,
                Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier,
            },
        };
    }
}
exports.RunLambdaTask = RunLambdaTask;
/**
 * Invocation type of a Lambda
 */
var InvocationType;
(function (InvocationType) {
    /**
     * Invoke synchronously
     *
     * The API response includes the function response and additional data.
     */
    InvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke asynchronously
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    InvocationType["EVENT"] = "Event";
    /**
     * TValidate parameter values and verify that the user or role has permission to invoke the function.
     */
    InvocationType["DRY_RUN"] = "DryRun";
})(InvocationType = exports.InvocationType || (exports.InvocationType = {}));
//# sourceMappingURL=data:application/json;base64,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