"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const task_utils_1 = require("../private/task-utils");
/**
 * Invoke a Lambda function as a Task
 *
 * @see https://docs.aws.amazon.com/step-functions/latest/dg/connect-lambda.html
 */
class LambdaInvoke extends sfn.TaskStateBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, props);
        this.props = props;
        this.integrationPattern = (_a = props.integrationPattern) !== null && _a !== void 0 ? _a : sfn.IntegrationPattern.REQUEST_RESPONSE;
        task_utils_1.validatePatternSupported(this.integrationPattern, LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS);
        if (this.integrationPattern === sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN
            && !sfn.FieldUtils.containsTaskToken(props.payload)) {
            throw new Error('Task Token is required in `payload` for callback. Use Context.taskToken to set the token.');
        }
        this.taskMetrics = {
            metricPrefixSingular: 'LambdaFunction',
            metricPrefixPlural: 'LambdaFunctions',
            metricDimensions: {
                LambdaFunctionArn: this.props.lambdaFunction.functionArn,
                ...(this.props.qualifier && { Qualifier: this.props.qualifier }),
            },
        };
        this.taskPolicies = [
            new iam.PolicyStatement({
                resources: [this.props.lambdaFunction.functionArn],
                actions: ['lambda:InvokeFunction'],
            }),
        ];
    }
    /**
     * Provides the Lambda Invoke service integration task configuration
     */
    renderTask() {
        return {
            Resource: task_utils_1.integrationResourceArn('lambda', 'invoke', this.integrationPattern),
            Parameters: sfn.FieldUtils.renderObject({
                FunctionName: this.props.lambdaFunction.functionArn,
                Payload: this.props.payload ? this.props.payload.value : sfn.TaskInput.fromDataAt('$').value,
                InvocationType: this.props.invocationType,
                ClientContext: this.props.clientContext,
                Qualifier: this.props.qualifier,
            }),
        };
    }
}
exports.LambdaInvoke = LambdaInvoke;
LambdaInvoke.SUPPORTED_INTEGRATION_PATTERNS = [
    sfn.IntegrationPattern.REQUEST_RESPONSE,
    sfn.IntegrationPattern.WAIT_FOR_TASK_TOKEN,
];
/**
 * Invocation type of a Lambda
 */
var LambdaInvocationType;
(function (LambdaInvocationType) {
    /**
     * Invoke the function synchronously.
     *
     * Keep the connection open until the function returns a response or times out.
     * The API response includes the function response and additional data.
     */
    LambdaInvocationType["REQUEST_RESPONSE"] = "RequestResponse";
    /**
     * Invoke the function asynchronously.
     *
     * Send events that fail multiple times to the function's dead-letter queue (if it's configured).
     * The API response only includes a status code.
     */
    LambdaInvocationType["EVENT"] = "Event";
    /**
     * Validate parameter values and verify that the user or role has permission to invoke the function.
     */
    LambdaInvocationType["DRY_RUN"] = "DryRun";
})(LambdaInvocationType = exports.LambdaInvocationType || (exports.LambdaInvocationType = {}));
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW52b2tlLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiaW52b2tlLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7O0FBQUEsd0NBQXdDLENBQUMsbURBQW1EO0FBRTVGLGtEQUFrRCxDQUFDLDZEQUE2RDtBQUVoSCxzREFBeUY7QUF1Q3pGOzs7O0dBSUc7QUFDSCxNQUFhLFlBQWEsU0FBUSxHQUFHLENBQUMsYUFBYTtJQVEvQyxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFtQixLQUF3Qjs7UUFDbkYsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7UUFEbUMsVUFBSyxHQUFMLEtBQUssQ0FBbUI7UUFFbkYsSUFBSSxDQUFDLGtCQUFrQixTQUFHLEtBQUssQ0FBQyxrQkFBa0IsbUNBQUksR0FBRyxDQUFDLGtCQUFrQixDQUFDLGdCQUFnQixDQUFDO1FBQzlGLHFDQUF3QixDQUFDLElBQUksQ0FBQyxrQkFBa0IsRUFBRSxZQUFZLENBQUMsOEJBQThCLENBQUMsQ0FBQztRQUMvRixJQUFJLElBQUksQ0FBQyxrQkFBa0IsS0FBSyxHQUFHLENBQUMsa0JBQWtCLENBQUMsbUJBQW1CO2VBQ25FLENBQUMsR0FBRyxDQUFDLFVBQVUsQ0FBQyxpQkFBaUIsQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLEVBQUU7WUFDckQsTUFBTSxJQUFJLEtBQUssQ0FBQywyRkFBMkYsQ0FBQyxDQUFDO1NBQ2hIO1FBQ0QsSUFBSSxDQUFDLFdBQVcsR0FBRztZQUNmLG9CQUFvQixFQUFFLGdCQUFnQjtZQUN0QyxrQkFBa0IsRUFBRSxpQkFBaUI7WUFDckMsZ0JBQWdCLEVBQUU7Z0JBQ2QsaUJBQWlCLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsV0FBVztnQkFDeEQsR0FBRyxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxJQUFJLEVBQUUsU0FBUyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUM7YUFDbkU7U0FDSixDQUFDO1FBQ0YsSUFBSSxDQUFDLFlBQVksR0FBRztZQUNoQixJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUM7Z0JBQ3BCLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLFdBQVcsQ0FBQztnQkFDbEQsT0FBTyxFQUFFLENBQUMsdUJBQXVCLENBQUM7YUFDckMsQ0FBQztTQUNMLENBQUM7SUFDTixDQUFDO0lBQ0Q7O09BRUc7SUFDTyxVQUFVO1FBQ2hCLE9BQU87WUFDSCxRQUFRLEVBQUUsbUNBQXNCLENBQUMsUUFBUSxFQUFFLFFBQVEsRUFBRSxJQUFJLENBQUMsa0JBQWtCLENBQUM7WUFDN0UsVUFBVSxFQUFFLEdBQUcsQ0FBQyxVQUFVLENBQUMsWUFBWSxDQUFDO2dCQUNwQyxZQUFZLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUMsV0FBVztnQkFDbkQsT0FBTyxFQUFFLElBQUksQ0FBQyxLQUFLLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxTQUFTLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEtBQUs7Z0JBQzVGLGNBQWMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGNBQWM7Z0JBQ3pDLGFBQWEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLGFBQWE7Z0JBQ3ZDLFNBQVMsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVM7YUFDbEMsQ0FBQztTQUNMLENBQUM7SUFDTixDQUFDOztBQTdDTCxvQ0E4Q0M7QUE3QzJCLDJDQUE4QixHQUE2QjtJQUMvRSxHQUFHLENBQUMsa0JBQWtCLENBQUMsZ0JBQWdCO0lBQ3ZDLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxtQkFBbUI7Q0FDN0MsQ0FBQztBQTJDTjs7R0FFRztBQUNILElBQVksb0JBbUJYO0FBbkJELFdBQVksb0JBQW9CO0lBQzVCOzs7OztPQUtHO0lBQ0gsNERBQW9DLENBQUE7SUFDcEM7Ozs7O09BS0c7SUFDSCx1Q0FBZSxDQUFBO0lBQ2Y7O09BRUc7SUFDSCwwQ0FBa0IsQ0FBQTtBQUN0QixDQUFDLEVBbkJXLG9CQUFvQixHQUFwQiw0QkFBb0IsS0FBcEIsNEJBQW9CLFFBbUIvQiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFwiLi4vLi4vLi4vYXdzLWxhbWJkYVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSdcbmltcG9ydCAqIGFzIHNmbiBmcm9tIFwiLi4vLi4vLi4vYXdzLXN0ZXBmdW5jdGlvbnNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zdGVwZnVuY3Rpb25zJ1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0IHsgaW50ZWdyYXRpb25SZXNvdXJjZUFybiwgdmFsaWRhdGVQYXR0ZXJuU3VwcG9ydGVkIH0gZnJvbSAnLi4vcHJpdmF0ZS90YXNrLXV0aWxzJztcbi8qKlxuICogUHJvcGVydGllcyBmb3IgaW52b2tpbmcgYSBMYW1iZGEgZnVuY3Rpb24gd2l0aCBMYW1iZGFJbnZva2VcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMYW1iZGFJbnZva2VQcm9wcyBleHRlbmRzIHNmbi5UYXNrU3RhdGVCYXNlUHJvcHMge1xuICAgIC8qKlxuICAgICAqIExhbWJkYSBmdW5jdGlvbiB0byBpbnZva2VcbiAgICAgKi9cbiAgICByZWFkb25seSBsYW1iZGFGdW5jdGlvbjogbGFtYmRhLklGdW5jdGlvbjtcbiAgICAvKipcbiAgICAgKiBUaGUgSlNPTiB0aGF0IHdpbGwgYmUgc3VwcGxpZWQgYXMgaW5wdXQgdG8gdGhlIExhbWJkYSBmdW5jdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBUaGUgc3RhdGUgaW5wdXQgKEpTT04gcGF0aCAnJCcpXG4gICAgICovXG4gICAgcmVhZG9ubHkgcGF5bG9hZD86IHNmbi5UYXNrSW5wdXQ7XG4gICAgLyoqXG4gICAgICogSW52b2NhdGlvbiB0eXBlIG9mIHRoZSBMYW1iZGEgZnVuY3Rpb25cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IEludm9jYXRpb25UeXBlLlJFUVVFU1RfUkVTUE9OU0VcbiAgICAgKi9cbiAgICByZWFkb25seSBpbnZvY2F0aW9uVHlwZT86IExhbWJkYUludm9jYXRpb25UeXBlO1xuICAgIC8qKlxuICAgICAqIFVwIHRvIDM1ODMgYnl0ZXMgb2YgYmFzZTY0LWVuY29kZWQgZGF0YSBhYm91dCB0aGUgaW52b2tpbmcgY2xpZW50XG4gICAgICogdG8gcGFzcyB0byB0aGUgZnVuY3Rpb24uXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIE5vIGNvbnRleHRcbiAgICAgKi9cbiAgICByZWFkb25seSBjbGllbnRDb250ZXh0Pzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFZlcnNpb24gb3IgYWxpYXMgdG8gaW52b2tlIGEgcHVibGlzaGVkIHZlcnNpb24gb2YgdGhlIGZ1bmN0aW9uXG4gICAgICpcbiAgICAgKiBZb3Ugb25seSBuZWVkIHRvIHN1cHBseSB0aGlzIGlmIHlvdSB3YW50IHRoZSB2ZXJzaW9uIG9mIHRoZSBMYW1iZGEgRnVuY3Rpb24gdG8gZGVwZW5kXG4gICAgICogb24gZGF0YSBpbiB0aGUgc3RhdGUgbWFjaGluZSBzdGF0ZS4gSWYgbm90LCB5b3UgY2FuIHBhc3MgdGhlIGFwcHJvcHJpYXRlIEFsaWFzIG9yIFZlcnNpb24gb2JqZWN0XG4gICAgICogZGlyZWN0bHkgYXMgdGhlIGBsYW1iZGFGdW5jdGlvbmAgYXJndW1lbnQuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIFZlcnNpb24gb3IgYWxpYXMgaW5oZXJlbnQgdG8gdGhlIGBsYW1iZGFGdW5jdGlvbmAgb2JqZWN0LlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHF1YWxpZmllcj86IHN0cmluZztcbn1cbi8qKlxuICogSW52b2tlIGEgTGFtYmRhIGZ1bmN0aW9uIGFzIGEgVGFza1xuICpcbiAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL3N0ZXAtZnVuY3Rpb25zL2xhdGVzdC9kZy9jb25uZWN0LWxhbWJkYS5odG1sXG4gKi9cbmV4cG9ydCBjbGFzcyBMYW1iZGFJbnZva2UgZXh0ZW5kcyBzZm4uVGFza1N0YXRlQmFzZSB7XG4gICAgcHJpdmF0ZSBzdGF0aWMgcmVhZG9ubHkgU1VQUE9SVEVEX0lOVEVHUkFUSU9OX1BBVFRFUk5TOiBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuW10gPSBbXG4gICAgICAgIHNmbi5JbnRlZ3JhdGlvblBhdHRlcm4uUkVRVUVTVF9SRVNQT05TRSxcbiAgICAgICAgc2ZuLkludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOLFxuICAgIF07XG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IHRhc2tNZXRyaWNzPzogc2ZuLlRhc2tNZXRyaWNzQ29uZmlnO1xuICAgIHByb3RlY3RlZCByZWFkb25seSB0YXNrUG9saWNpZXM/OiBpYW0uUG9saWN5U3RhdGVtZW50W107XG4gICAgcHJpdmF0ZSByZWFkb25seSBpbnRlZ3JhdGlvblBhdHRlcm46IHNmbi5JbnRlZ3JhdGlvblBhdHRlcm47XG4gICAgY29uc3RydWN0b3Ioc2NvcGU6IGNkay5Db25zdHJ1Y3QsIGlkOiBzdHJpbmcsIHByaXZhdGUgcmVhZG9ubHkgcHJvcHM6IExhbWJkYUludm9rZVByb3BzKSB7XG4gICAgICAgIHN1cGVyKHNjb3BlLCBpZCwgcHJvcHMpO1xuICAgICAgICB0aGlzLmludGVncmF0aW9uUGF0dGVybiA9IHByb3BzLmludGVncmF0aW9uUGF0dGVybiA/PyBzZm4uSW50ZWdyYXRpb25QYXR0ZXJuLlJFUVVFU1RfUkVTUE9OU0U7XG4gICAgICAgIHZhbGlkYXRlUGF0dGVyblN1cHBvcnRlZCh0aGlzLmludGVncmF0aW9uUGF0dGVybiwgTGFtYmRhSW52b2tlLlNVUFBPUlRFRF9JTlRFR1JBVElPTl9QQVRURVJOUyk7XG4gICAgICAgIGlmICh0aGlzLmludGVncmF0aW9uUGF0dGVybiA9PT0gc2ZuLkludGVncmF0aW9uUGF0dGVybi5XQUlUX0ZPUl9UQVNLX1RPS0VOXG4gICAgICAgICAgICAmJiAhc2ZuLkZpZWxkVXRpbHMuY29udGFpbnNUYXNrVG9rZW4ocHJvcHMucGF5bG9hZCkpIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignVGFzayBUb2tlbiBpcyByZXF1aXJlZCBpbiBgcGF5bG9hZGAgZm9yIGNhbGxiYWNrLiBVc2UgQ29udGV4dC50YXNrVG9rZW4gdG8gc2V0IHRoZSB0b2tlbi4nKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLnRhc2tNZXRyaWNzID0ge1xuICAgICAgICAgICAgbWV0cmljUHJlZml4U2luZ3VsYXI6ICdMYW1iZGFGdW5jdGlvbicsXG4gICAgICAgICAgICBtZXRyaWNQcmVmaXhQbHVyYWw6ICdMYW1iZGFGdW5jdGlvbnMnLFxuICAgICAgICAgICAgbWV0cmljRGltZW5zaW9uczoge1xuICAgICAgICAgICAgICAgIExhbWJkYUZ1bmN0aW9uQXJuOiB0aGlzLnByb3BzLmxhbWJkYUZ1bmN0aW9uLmZ1bmN0aW9uQXJuLFxuICAgICAgICAgICAgICAgIC4uLih0aGlzLnByb3BzLnF1YWxpZmllciAmJiB7IFF1YWxpZmllcjogdGhpcy5wcm9wcy5xdWFsaWZpZXIgfSksXG4gICAgICAgICAgICB9LFxuICAgICAgICB9O1xuICAgICAgICB0aGlzLnRhc2tQb2xpY2llcyA9IFtcbiAgICAgICAgICAgIG5ldyBpYW0uUG9saWN5U3RhdGVtZW50KHtcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFt0aGlzLnByb3BzLmxhbWJkYUZ1bmN0aW9uLmZ1bmN0aW9uQXJuXSxcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbJ2xhbWJkYTpJbnZva2VGdW5jdGlvbiddLFxuICAgICAgICAgICAgfSksXG4gICAgICAgIF07XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFByb3ZpZGVzIHRoZSBMYW1iZGEgSW52b2tlIHNlcnZpY2UgaW50ZWdyYXRpb24gdGFzayBjb25maWd1cmF0aW9uXG4gICAgICovXG4gICAgcHJvdGVjdGVkIHJlbmRlclRhc2soKTogYW55IHtcbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIFJlc291cmNlOiBpbnRlZ3JhdGlvblJlc291cmNlQXJuKCdsYW1iZGEnLCAnaW52b2tlJywgdGhpcy5pbnRlZ3JhdGlvblBhdHRlcm4pLFxuICAgICAgICAgICAgUGFyYW1ldGVyczogc2ZuLkZpZWxkVXRpbHMucmVuZGVyT2JqZWN0KHtcbiAgICAgICAgICAgICAgICBGdW5jdGlvbk5hbWU6IHRoaXMucHJvcHMubGFtYmRhRnVuY3Rpb24uZnVuY3Rpb25Bcm4sXG4gICAgICAgICAgICAgICAgUGF5bG9hZDogdGhpcy5wcm9wcy5wYXlsb2FkID8gdGhpcy5wcm9wcy5wYXlsb2FkLnZhbHVlIDogc2ZuLlRhc2tJbnB1dC5mcm9tRGF0YUF0KCckJykudmFsdWUsXG4gICAgICAgICAgICAgICAgSW52b2NhdGlvblR5cGU6IHRoaXMucHJvcHMuaW52b2NhdGlvblR5cGUsXG4gICAgICAgICAgICAgICAgQ2xpZW50Q29udGV4dDogdGhpcy5wcm9wcy5jbGllbnRDb250ZXh0LFxuICAgICAgICAgICAgICAgIFF1YWxpZmllcjogdGhpcy5wcm9wcy5xdWFsaWZpZXIsXG4gICAgICAgICAgICB9KSxcbiAgICAgICAgfTtcbiAgICB9XG59XG4vKipcbiAqIEludm9jYXRpb24gdHlwZSBvZiBhIExhbWJkYVxuICovXG5leHBvcnQgZW51bSBMYW1iZGFJbnZvY2F0aW9uVHlwZSB7XG4gICAgLyoqXG4gICAgICogSW52b2tlIHRoZSBmdW5jdGlvbiBzeW5jaHJvbm91c2x5LlxuICAgICAqXG4gICAgICogS2VlcCB0aGUgY29ubmVjdGlvbiBvcGVuIHVudGlsIHRoZSBmdW5jdGlvbiByZXR1cm5zIGEgcmVzcG9uc2Ugb3IgdGltZXMgb3V0LlxuICAgICAqIFRoZSBBUEkgcmVzcG9uc2UgaW5jbHVkZXMgdGhlIGZ1bmN0aW9uIHJlc3BvbnNlIGFuZCBhZGRpdGlvbmFsIGRhdGEuXG4gICAgICovXG4gICAgUkVRVUVTVF9SRVNQT05TRSA9ICdSZXF1ZXN0UmVzcG9uc2UnLFxuICAgIC8qKlxuICAgICAqIEludm9rZSB0aGUgZnVuY3Rpb24gYXN5bmNocm9ub3VzbHkuXG4gICAgICpcbiAgICAgKiBTZW5kIGV2ZW50cyB0aGF0IGZhaWwgbXVsdGlwbGUgdGltZXMgdG8gdGhlIGZ1bmN0aW9uJ3MgZGVhZC1sZXR0ZXIgcXVldWUgKGlmIGl0J3MgY29uZmlndXJlZCkuXG4gICAgICogVGhlIEFQSSByZXNwb25zZSBvbmx5IGluY2x1ZGVzIGEgc3RhdHVzIGNvZGUuXG4gICAgICovXG4gICAgRVZFTlQgPSAnRXZlbnQnLFxuICAgIC8qKlxuICAgICAqIFZhbGlkYXRlIHBhcmFtZXRlciB2YWx1ZXMgYW5kIHZlcmlmeSB0aGF0IHRoZSB1c2VyIG9yIHJvbGUgaGFzIHBlcm1pc3Npb24gdG8gaW52b2tlIHRoZSBmdW5jdGlvbi5cbiAgICAgKi9cbiAgICBEUllfUlVOID0gJ0RyeVJ1bidcbn1cbiJdfQ==