"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * Determines the level of detail about provisioned throughput consumption that is returned.
 */
var DynamoConsumedCapacity;
(function (DynamoConsumedCapacity) {
    /**
     * The response includes the aggregate ConsumedCapacity for the operation,
     * together with ConsumedCapacity for each table and secondary index that was accessed
     */
    DynamoConsumedCapacity["INDEXES"] = "INDEXES";
    /**
     * The response includes only the aggregate ConsumedCapacity for the operation.
     */
    DynamoConsumedCapacity["TOTAL"] = "TOTAL";
    /**
     * No ConsumedCapacity details are included in the response.
     */
    DynamoConsumedCapacity["NONE"] = "NONE";
})(DynamoConsumedCapacity = exports.DynamoConsumedCapacity || (exports.DynamoConsumedCapacity = {}));
/**
 * Determines whether item collection metrics are returned.
 */
var DynamoItemCollectionMetrics;
(function (DynamoItemCollectionMetrics) {
    /**
     * If set to SIZE, the response includes statistics about item collections,
     * if any, that were modified during the operation.
     */
    DynamoItemCollectionMetrics["SIZE"] = "SIZE";
    /**
     * If set to NONE, no statistics are returned.
     */
    DynamoItemCollectionMetrics["NONE"] = "NONE";
})(DynamoItemCollectionMetrics = exports.DynamoItemCollectionMetrics || (exports.DynamoItemCollectionMetrics = {}));
/**
 * Use ReturnValues if you want to get the item attributes as they appear before or after they are changed
 */
var DynamoReturnValues;
(function (DynamoReturnValues) {
    /**
     * Nothing is returned
     */
    DynamoReturnValues["NONE"] = "NONE";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_OLD"] = "ALL_OLD";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_OLD"] = "UPDATED_OLD";
    /**
     * Returns all of the attributes of the item
     */
    DynamoReturnValues["ALL_NEW"] = "ALL_NEW";
    /**
     * Returns only the updated attributes
     */
    DynamoReturnValues["UPDATED_NEW"] = "UPDATED_NEW";
})(DynamoReturnValues = exports.DynamoReturnValues || (exports.DynamoReturnValues = {}));
/**
 * Class to generate AttributeValue
 */
class DynamoAttributeValue {
    constructor() {
        this.attributeValue = {};
    }
    /**
     * Sets an attribute of type String. For example:  "S": "Hello"
     */
    withS(value) {
        this.attributeValue.S = value;
        return this;
    }
    /**
     * Sets an attribute of type Number. For example:  "N": "123.45"
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withN(value) {
        this.attributeValue.N = value;
        return this;
    }
    /**
     * Sets an attribute of type Binary. For example:  "B": "dGhpcyB0ZXh0IGlzIGJhc2U2NC1lbmNvZGVk"
     */
    withB(value) {
        this.attributeValue.B = value;
        return this;
    }
    /**
     * Sets an attribute of type String Set. For example:  "SS": ["Giraffe", "Hippo" ,"Zebra"]
     */
    withSS(value) {
        this.attributeValue.SS = value;
        return this;
    }
    /**
     * Sets an attribute of type Number Set. For example:  "NS": ["42.2", "-19", "7.5", "3.14"]
     * Numbers are sent across the network to DynamoDB as strings,
     * to maximize compatibility across languages and libraries.
     * However, DynamoDB treats them as number type attributes for mathematical operations.
     */
    withNS(value) {
        this.attributeValue.NS = value;
        return this;
    }
    /**
     * Sets an attribute of type Binary Set. For example:  "BS": ["U3Vubnk=", "UmFpbnk=", "U25vd3k="]
     */
    withBS(value) {
        this.attributeValue.BS = value;
        return this;
    }
    /**
     * Sets an attribute of type Map. For example:  "M": {"Name": {"S": "Joe"}, "Age": {"N": "35"}}
     */
    withM(value) {
        this.attributeValue.M = transformAttributeValueMap(value);
        return this;
    }
    /**
     * Sets an attribute of type List. For example:  "L": [ {"S": "Cookies"} , {"S": "Coffee"}, {"N", "3.14159"}]
     */
    withL(value) {
        this.attributeValue.L = value.map(val => val.toObject());
        return this;
    }
    /**
     * Sets an attribute of type Null. For example:  "NULL": true
     */
    withNULL(value) {
        this.attributeValue.NULL = value;
        return this;
    }
    /**
     * Sets an attribute of type Boolean. For example:  "BOOL": true
     */
    withBOOL(value) {
        this.attributeValue.BOOL = value;
        return this;
    }
    /**
     * Return the attributeValue object
     */
    toObject() {
        return this.attributeValue;
    }
}
exports.DynamoAttributeValue = DynamoAttributeValue;
/**
 * Class to generate projection expression
 */
class DynamoProjectionExpression {
    constructor() {
        this.expression = [];
    }
    /**
     * Adds the passed attribute to the chain
     *
     * @param attr Attribute name
     */
    withAttribute(attr) {
        if (this.expression.length) {
            this.expression.push(`.${attr}`);
        }
        else {
            this.expression.push(attr);
        }
        return this;
    }
    /**
     * Adds the array literal access for passed index
     *
     * @param index array index
     */
    atIndex(index) {
        if (!this.expression.length) {
            throw new Error('Expression must start with an attribute');
        }
        this.expression.push(`[${index}]`);
        return this;
    }
    /**
     * converts and return the string expression
     */
    toString() {
        return this.expression.join('');
    }
}
exports.DynamoProjectionExpression = DynamoProjectionExpression;
/**
 * A StepFunctions task to call DynamoGetItem
 */
class DynamoGetItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        var _a;
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.GET),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.GET),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConsistentRead: (_a = this.props.consistentRead) !== null && _a !== void 0 ? _a : false,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ProjectionExpression: this.configureProjectionExpression(this.props.projectionExpression),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
            },
        };
    }
    configureProjectionExpression(expressions) {
        return expressions
            ? expressions.map(expression => expression.toString()).join(',')
            : undefined;
    }
}
exports.DynamoGetItem = DynamoGetItem;
/**
 * A StepFunctions task to call DynamoPutItem
 */
class DynamoPutItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.PUT),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.PUT),
            parameters: {
                Item: transformAttributeValueMap(this.props.item),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
            },
        };
    }
}
exports.DynamoPutItem = DynamoPutItem;
/**
 * A StepFunctions task to call DynamoDeleteItem
 */
class DynamoDeleteItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.DELETE),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.DELETE),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
            },
        };
    }
}
exports.DynamoDeleteItem = DynamoDeleteItem;
/**
 * A StepFunctions task to call DynamoUpdateItem
 */
class DynamoUpdateItem {
    constructor(props) {
        this.props = props;
        core_1.withResolved(props.tableName, validateTableName);
    }
    bind(_task) {
        return {
            resourceArn: getDynamoResourceArn(DynamoMethod.UPDATE),
            policyStatements: getDynamoPolicyStatements(_task, this.props.tableName, DynamoMethod.UPDATE),
            parameters: {
                Key: configurePrimaryKey(this.props.partitionKey, this.props.sortKey),
                TableName: this.props.tableName,
                ConditionExpression: this.props.conditionExpression,
                ExpressionAttributeNames: this.props.expressionAttributeNames,
                ExpressionAttributeValues: transformAttributeValueMap(this.props.expressionAttributeValues),
                ReturnConsumedCapacity: this.props.returnConsumedCapacity,
                ReturnItemCollectionMetrics: this.props.returnItemCollectionMetrics,
                ReturnValues: this.props.returnValues,
                UpdateExpression: this.props.updateExpression,
            },
        };
    }
}
exports.DynamoUpdateItem = DynamoUpdateItem;
/**
 * A helper wrapper class to call all DynamoDB APIs
 */
class CallDynamoDB {
    /**
     * Method to get DynamoGetItem task
     *
     * @param props DynamoGetItemProps
     */
    static getItem(props) {
        return new DynamoGetItem(props);
    }
    /**
     * Method to get DynamoPutItem task
     *
     * @param props DynamoPutItemProps
     */
    static putItem(props) {
        return new DynamoPutItem(props);
    }
    /**
     * Method to get DynamoDeleteItem task
     *
     * @param props DynamoDeleteItemProps
     */
    static deleteItem(props) {
        return new DynamoDeleteItem(props);
    }
    /**
     * Method to get DynamoUpdateItem task
     *
     * @param props DynamoUpdateItemProps
     */
    static updateItem(props) {
        return new DynamoUpdateItem(props);
    }
}
exports.CallDynamoDB = CallDynamoDB;
var DynamoMethod;
(function (DynamoMethod) {
    DynamoMethod["GET"] = "Get";
    DynamoMethod["PUT"] = "Put";
    DynamoMethod["DELETE"] = "Delete";
    DynamoMethod["UPDATE"] = "Update";
})(DynamoMethod || (DynamoMethod = {}));
function validateTableName(tableName) {
    if (tableName.length < 3 ||
        tableName.length > 255 ||
        !new RegExp(/[a-zA-Z0-9_.-]+$/).test(tableName)) {
        throw new Error(`TableName should not contain alphanumeric characters and should be between 3-255 characters long. Received: ${tableName}`);
    }
}
function getDynamoResourceArn(method) {
    return resource_arn_suffix_1.getResourceArn('dynamodb', `${method.toLowerCase()}Item`, sfn.ServiceIntegrationPattern.FIRE_AND_FORGET);
}
function getDynamoPolicyStatements(task, tableName, method) {
    return [
        new iam.PolicyStatement({
            resources: [
                core_1.Stack.of(task).formatArn({
                    service: 'dynamodb',
                    resource: 'table',
                    resourceName: tableName,
                }),
            ],
            actions: [`dynamodb:${method}Item`],
        }),
    ];
}
function configurePrimaryKey(partitionKey, sortKey) {
    const key = {
        [partitionKey.name]: partitionKey.value.toObject(),
    };
    if (sortKey) {
        key[sortKey.name] = sortKey.value.toObject();
    }
    return key;
}
function transformAttributeValueMap(attrMap) {
    const transformedValue = {};
    for (const key in attrMap) {
        if (key) {
            transformedValue[key] = attrMap[key].toObject();
        }
    }
    return attrMap ? transformedValue : undefined;
}
//# sourceMappingURL=data:application/json;base64,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