"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const sfn = require("../../../aws-stepfunctions"); // Automatically re-written from '@aws-cdk/aws-stepfunctions'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const resource_arn_suffix_1 = require("../resource-arn-suffix");
/**
 * A Step Functions Task to run AWS Batch
 *
 * @deprecated use `BatchSubmitJob`
 */
class RunBatchJob {
    constructor(props) {
        var _a;
        this.props = props;
        // validate integrationPattern
        this.integrationPattern =
            props.integrationPattern || sfn.ServiceIntegrationPattern.SYNC;
        const supportedPatterns = [
            sfn.ServiceIntegrationPattern.FIRE_AND_FORGET,
            sfn.ServiceIntegrationPattern.SYNC,
        ];
        if (!supportedPatterns.includes(this.integrationPattern)) {
            throw new Error(`Invalid Service Integration Pattern: ${this.integrationPattern} is not supported to call RunBatchJob.`);
        }
        // validate arraySize limits
        core_1.withResolved(props.arraySize, (arraySize) => {
            if (arraySize !== undefined && (arraySize < 2 || arraySize > 10000)) {
                throw new Error(`arraySize must be between 2 and 10,000. Received ${arraySize}.`);
            }
        });
        // validate dependency size
        if (props.dependsOn && props.dependsOn.length > 20) {
            throw new Error(`dependencies must be 20 or less. Received ${props.dependsOn.length}.`);
        }
        // validate attempts
        core_1.withResolved(props.attempts, (attempts) => {
            if (attempts !== undefined && (attempts < 1 || attempts > 10)) {
                throw new Error(`attempts must be between 1 and 10. Received ${attempts}.`);
            }
        });
        // validate timeout
        // tslint:disable-next-line:no-unused-expression
        props.timeout !== undefined && core_1.withResolved(props.timeout.toSeconds(), (timeout) => {
            if (timeout < 60) {
                throw new Error(`timeout must be greater than 60 seconds. Received ${timeout} seconds.`);
            }
        });
        // This is reuqired since environment variables must not start with AWS_BATCH;
        // this naming convention is reserved for variables that are set by the AWS Batch service.
        if ((_a = props.containerOverrides) === null || _a === void 0 ? void 0 : _a.environment) {
            Object.keys(props.containerOverrides.environment).forEach(key => {
                if (key.match(/^AWS_BATCH/)) {
                    throw new Error(`Invalid environment variable name: ${key}. Environment variable names starting with 'AWS_BATCH' are reserved.`);
                }
            });
        }
    }
    bind(_task) {
        return {
            resourceArn: resource_arn_suffix_1.getResourceArn('batch', 'submitJob', this.integrationPattern),
            policyStatements: this.configurePolicyStatements(_task),
            parameters: {
                JobDefinition: this.props.jobDefinition.jobDefinitionArn,
                JobName: this.props.jobName,
                JobQueue: this.props.jobQueue.jobQueueArn,
                Parameters: this.props.payload,
                ArrayProperties: this.props.arraySize !== undefined
                    ? { Size: this.props.arraySize }
                    : undefined,
                ContainerOverrides: this.props.containerOverrides
                    ? this.configureContainerOverrides(this.props.containerOverrides)
                    : undefined,
                DependsOn: this.props.dependsOn
                    ? this.props.dependsOn.map(jobDependency => ({
                        JobId: jobDependency.jobId,
                        Type: jobDependency.type,
                    }))
                    : undefined,
                RetryStrategy: this.props.attempts !== undefined
                    ? { Attempts: this.props.attempts }
                    : undefined,
                Timeout: this.props.timeout
                    ? { AttemptDurationSeconds: this.props.timeout.toSeconds() }
                    : undefined,
            },
        };
    }
    configurePolicyStatements(task) {
        return [
            // Resource level access control for job-definition requires revision which batch does not support yet
            // Using the alternative permissions as mentioned here:
            // https://docs.aws.amazon.com/batch/latest/userguide/batch-supported-iam-actions-resources.html
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'batch',
                        resource: 'job-definition',
                        resourceName: '*',
                    }),
                    this.props.jobQueue.jobQueueArn,
                ],
                actions: ['batch:SubmitJob'],
            }),
            new iam.PolicyStatement({
                resources: [
                    core_1.Stack.of(task).formatArn({
                        service: 'events',
                        resource: 'rule/StepFunctionsGetEventsForBatchJobsRule',
                    }),
                ],
                actions: ['events:PutTargets', 'events:PutRule', 'events:DescribeRule'],
            }),
        ];
    }
    configureContainerOverrides(containerOverrides) {
        var _a;
        let environment;
        if (containerOverrides.environment) {
            environment = Object.entries(containerOverrides.environment).map(([key, value]) => ({
                Name: key,
                Value: value,
            }));
        }
        let resources;
        if (containerOverrides.gpuCount) {
            resources = [
                {
                    Type: 'GPU',
                    Value: `${containerOverrides.gpuCount}`,
                },
            ];
        }
        return {
            Command: containerOverrides.command,
            Environment: environment,
            InstanceType: (_a = containerOverrides.instanceType) === null || _a === void 0 ? void 0 : _a.toString(),
            Memory: containerOverrides.memory,
            ResourceRequirements: resources,
            Vcpus: containerOverrides.vcpus,
        };
    }
}
exports.RunBatchJob = RunBatchJob;
//# sourceMappingURL=data:application/json;base64,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