"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_1 = require("./policy");
/**
 * Reference to a new or existing Amazon SQS queue
 */
class QueueBase extends core_1.Resource {
    /**
     * Adds a statement to the IAM resource policy associated with this queue.
     *
     * If this queue was created in this stack (`new Queue`), a queue policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the queue is improted (`Queue.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.QueuePolicy(this, 'Policy', { queues: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Grant permissions to consume messages from a queue
     *
     * This will grant the following permissions:
     *
     *   - sqs:ChangeMessageVisibility
     *   - sqs:DeleteMessage
     *   - sqs:ReceiveMessage
     *   - sqs:GetQueueAttributes
     *   - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant consume rights to
     */
    grantConsumeMessages(grantee) {
        const ret = this.grant(grantee, 'sqs:ReceiveMessage', 'sqs:ChangeMessageVisibility', 'sqs:GetQueueUrl', 'sqs:DeleteMessage', 'sqs:GetQueueAttributes');
        if (this.encryptionMasterKey) {
            this.encryptionMasterKey.grantDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant access to send messages to a queue to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - sqs:SendMessage
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantSendMessages(grantee) {
        const ret = this.grant(grantee, 'sqs:SendMessage', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
        if (this.encryptionMasterKey) {
            // kms:Decrypt necessary to execute grantsendMessages to an SSE enabled SQS queue
            this.encryptionMasterKey.grantEncryptDecrypt(grantee);
        }
        return ret;
    }
    /**
     * Grant an IAM principal permissions to purge all messages from the queue.
     *
     * This will grant the following permissions:
     *
     *  - sqs:PurgeQueue
     *  - sqs:GetQueueAttributes
     *  - sqs:GetQueueUrl
     *
     * @param grantee Principal to grant send rights to
     */
    grantPurge(grantee) {
        return this.grant(grantee, 'sqs:PurgeQueue', 'sqs:GetQueueAttributes', 'sqs:GetQueueUrl');
    }
    /**
     * Grant the actions defined in queueActions to the identity Principal given
     * on this SQS queue resource.
     *
     * @param grantee Principal to grant right to
     * @param actions The actions to grant
     */
    grant(grantee, ...actions) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions,
            resourceArns: [this.queueArn],
            resource: this,
        });
    }
}
exports.QueueBase = QueueBase;
//# sourceMappingURL=data:application/json;base64,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