"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const policy_1 = require("./policy");
const subscription_1 = require("./subscription");
/**
 * Either a new or imported Topic
 */
class TopicBase extends core_1.Resource {
    /**
     * Subscribe some endpoint to this topic
     */
    addSubscription(subscription) {
        const subscriptionConfig = subscription.bind(this);
        const scope = subscriptionConfig.subscriberScope || this;
        let id = subscriptionConfig.subscriberId;
        if (core_1.Token.isUnresolved(subscriptionConfig.subscriberId)) {
            id = this.nextTokenId(scope);
        }
        // We use the subscriber's id as the construct id. There's no meaning
        // to subscribing the same subscriber twice on the same topic.
        if (scope.node.tryFindChild(id)) {
            throw new Error(`A subscription with id "${id}" already exists under the scope ${scope.node.path}`);
        }
        new subscription_1.Subscription(scope, id, {
            topic: this,
            ...subscriptionConfig,
        });
    }
    /**
     * Adds a statement to the IAM resource policy associated with this topic.
     *
     * If this topic was created in this stack (`new Topic`), a topic policy
     * will be automatically created upon the first call to `addToPolicy`. If
     * the topic is improted (`Topic.import`), then this is a no-op.
     */
    addToResourcePolicy(statement) {
        if (!this.policy && this.autoCreatePolicy) {
            this.policy = new policy_1.TopicPolicy(this, 'Policy', { topics: [this] });
        }
        if (this.policy) {
            this.policy.document.addStatements(statement);
            return { statementAdded: true, policyDependable: this.policy };
        }
        return { statementAdded: false };
    }
    /**
     * Grant topic publishing permissions to the given identity
     */
    grantPublish(grantee) {
        return iam.Grant.addToPrincipalOrResource({
            grantee,
            actions: ['sns:Publish'],
            resourceArns: [this.topicArn],
            resource: this,
        });
    }
    nextTokenId(scope) {
        let nextSuffix = 1;
        const re = /TokenSubscription:([\d]*)/gm;
        // Search through the construct and all of its children
        // for previous subscriptions that match our regex pattern
        for (const source of scope.node.findAll()) {
            const m = re.exec(source.node.id); // Use regex to find a match
            if (m !== null) { // if we found a match
                const matchSuffix = parseInt(m[1], 10); // get the suffix for that match (as integer)
                if (matchSuffix >= nextSuffix) { // check if the match suffix is larger or equal to currently proposed suffix
                    nextSuffix = matchSuffix + 1; // increment the suffix
                }
            }
        }
        return `TokenSubscription:${nextSuffix}`;
    }
}
exports.TopicBase = TopicBase;
//# sourceMappingURL=data:application/json;base64,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