"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const instance_1 = require("./instance");
const namespace_1 = require("./namespace");
const service_1 = require("./service");
const servicediscovery_generated_1 = require("./servicediscovery.generated");
/**
 * Instance that uses Route 53 Alias record type. Currently, the only resource types supported are Elastic Load
 * Balancers.
 *
 * @resource AWS::ServiceDiscovery::Instance
 */
class AliasTargetInstance extends instance_1.InstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.service.namespace.type === namespace_1.NamespaceType.HTTP) {
            throw new Error('Namespace associated with Service must be a DNS Namespace.');
        }
        // Should already be enforced when creating service, but validates if service is not instantiated with #createService
        const dnsRecordType = props.service.dnsRecordType;
        if (dnsRecordType !== service_1.DnsRecordType.A
            && dnsRecordType !== service_1.DnsRecordType.AAAA
            && dnsRecordType !== service_1.DnsRecordType.A_AAAA) {
            throw new Error('Service must use `A` or `AAAA` records to register an AliasRecordTarget.');
        }
        if (props.service.routingPolicy !== service_1.RoutingPolicy.WEIGHTED) {
            throw new Error('Service must use `WEIGHTED` routing policy.');
        }
        const resource = new servicediscovery_generated_1.CfnInstance(this, 'Resource', {
            instanceAttributes: {
                AWS_ALIAS_DNS_NAME: props.dnsName,
                ...props.customAttributes,
            },
            instanceId: props.instanceId || this.node.uniqueId,
            serviceId: props.service.serviceId,
        });
        this.service = props.service;
        this.instanceId = resource.ref;
        this.dnsName = props.dnsName;
    }
}
exports.AliasTargetInstance = AliasTargetInstance;
//# sourceMappingURL=data:application/json;base64,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