"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const serverless = require("../../aws-sam"); // Automatically re-written from '@aws-cdk/aws-sam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A secret rotation serverless application.
 */
class SecretRotationApplication {
    constructor(applicationId, semanticVersion, options) {
        this.applicationId = `arn:aws:serverlessrepo:us-east-1:297356227824:applications/${applicationId}`;
        this.semanticVersion = semanticVersion;
        this.isMultiUser = options && options.isMultiUser;
    }
}
exports.SecretRotationApplication = SecretRotationApplication;
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the single user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MariaDB using the multi user rotation scheme
 */
SecretRotationApplication.MARIADB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMariaDBRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the single user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS MySQL using the multi user rotation scheme
 */
SecretRotationApplication.MYSQL_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSMySQLRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the single user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS Oracle using the multi user rotation scheme
 */
SecretRotationApplication.ORACLE_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSOracleRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the single user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS PostgreSQL using the multi user rotation scheme
 */
SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSPostgreSQLRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the single user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for RDS SQL Server using the multi user rotation scheme
 */
SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRDSSQLServerRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the single user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for Amazon Redshift using the multi user rotation scheme
 */
SecretRotationApplication.REDSHIFT_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerRedshiftRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the single user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_SINGLE_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationSingleUser', '1.1.3');
/**
 * Conducts an AWS SecretsManager secret rotation for MongoDB using the multi user rotation scheme
 */
SecretRotationApplication.MONGODB_ROTATION_MULTI_USER = new SecretRotationApplication('SecretsManagerMongoDBRotationMultiUser', '1.1.3', {
    isMultiUser: true,
});
/**
 * Secret rotation for a service or database
 */
class SecretRotation extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (!props.target.connections.defaultPort) {
            throw new Error('The `target` connections must have a default port range.');
        }
        if (props.application.isMultiUser && !props.masterSecret) {
            throw new Error('The `masterSecret` must be specified for application using the multi user scheme.');
        }
        const rotationFunctionName = this.node.uniqueId;
        const securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
        });
        props.target.connections.allowDefaultPortFrom(securityGroup);
        const parameters = {
            endpoint: `https://secretsmanager.${core_1.Stack.of(this).region}.${core_1.Stack.of(this).urlSuffix}`,
            functionName: rotationFunctionName,
            vpcSubnetIds: props.vpc.selectSubnets(props.vpcSubnets).subnetIds.join(','),
            vpcSecurityGroupIds: securityGroup.securityGroupId,
        };
        if (props.secret.encryptionKey) {
            parameters.kmsKeyArn = props.secret.encryptionKey.keyArn;
        }
        if (props.masterSecret) {
            parameters.masterSecretArn = props.masterSecret.secretArn;
            if (props.masterSecret.encryptionKey) {
                parameters.masterSecretKmsKeyArn = props.masterSecret.encryptionKey.keyArn;
            }
        }
        const application = new serverless.CfnApplication(this, 'Resource', {
            location: props.application,
            parameters,
        });
        // This creates a CF a dependency between the rotation schedule and the
        // serverless application. This is needed because it's the application
        // that creates the Lambda permission to invoke the function.
        // See https://docs.aws.amazon.com/secretsmanager/latest/userguide/integrating_cloudformation.html
        const rotationLambda = lambda.Function.fromFunctionArn(this, 'RotationLambda', core_1.Token.asString(application.getAtt('Outputs.RotationLambdaARN')));
        props.secret.addRotationSchedule('RotationSchedule', {
            rotationLambda,
            automaticallyAfter: props.automaticallyAfter,
        });
        // Prevent secrets deletions when rotation is in place
        props.secret.denyAccountRootDelete();
        if (props.masterSecret) {
            props.masterSecret.denyAccountRootDelete();
        }
    }
}
exports.SecretRotation = SecretRotation;
//# sourceMappingURL=data:application/json;base64,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