"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3_assets = require("../../aws-s3-assets"); // Automatically re-written from '@aws-cdk/aws-s3-assets'
/**
 * Specifies bucket deployment source.
 *
 * Usage:
 *
 *     Source.bucket(bucket, key)
 *     Source.asset('/local/path/to/directory')
 *     Source.asset('/local/path/to/a/file.zip')
 *
 */
class Source {
    /**
     * Uses a .zip file stored in an S3 bucket as the source for the destination bucket contents.
     * @param bucket The S3 Bucket
     * @param zipObjectKey The S3 object key of the zip file with contents
     */
    static bucket(bucket, zipObjectKey) {
        return {
            bind: () => ({ bucket, zipObjectKey }),
        };
    }
    /**
     * Uses a local asset as the deployment source.
     * @param path The path to a local .zip file or a directory
     */
    static asset(path, options) {
        return {
            bind(context) {
                let id = 1;
                while (context.node.tryFindChild(`Asset${id}`)) {
                    id++;
                }
                const asset = new s3_assets.Asset(context, `Asset${id}`, {
                    path,
                    ...options,
                });
                if (!asset.isZipArchive) {
                    throw new Error('Asset path must be either a .zip file or a directory');
                }
                return {
                    bucket: asset.bucket,
                    zipObjectKey: asset.s3ObjectKey,
                };
            },
        };
    }
    constructor() { }
}
exports.Source = Source;
//# sourceMappingURL=data:application/json;base64,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