"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const crypto = require("crypto");
const fs = require("fs");
const path = require("path");
const now = Date.now();
const handlerCodeBundle = path.join(__dirname, '..', 'lambda', 'bundle.zip');
const handlerSourceDirectory = path.join(__dirname, '..', 'lambda', 'src');
class BucketDeployment extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.distributionPaths && !props.distribution) {
            throw new Error('Distribution must be specified if distribution paths are specified');
        }
        const sourceHash = calcSourceHash(handlerSourceDirectory);
        const handler = new lambda.SingletonFunction(this, 'CustomResourceHandler', {
            uuid: this.renderSingletonUuid(props.memoryLimit),
            code: lambda.Code.fromAsset(handlerCodeBundle, { sourceHash }),
            runtime: lambda.Runtime.PYTHON_3_6,
            handler: 'index.handler',
            lambdaPurpose: 'Custom::CDKBucketDeployment',
            timeout: cdk.Duration.minutes(15),
            role: props.role,
            memorySize: props.memoryLimit,
        });
        const sources = props.sources.map((source) => source.bind(this));
        sources.forEach(source => source.bucket.grantRead(handler));
        props.destinationBucket.grantReadWrite(handler);
        if (props.distribution) {
            handler.addToRolePolicy(new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['cloudfront:GetInvalidation', 'cloudfront:CreateInvalidation'],
                resources: ['*'],
            }));
        }
        new cdk.CustomResource(this, 'CustomResource', {
            serviceToken: handler.functionArn,
            resourceType: 'Custom::CDKBucketDeployment',
            properties: {
                SourceBucketNames: sources.map(source => source.bucket.bucketName),
                SourceObjectKeys: sources.map(source => source.zipObjectKey),
                DestinationBucketName: props.destinationBucket.bucketName,
                DestinationBucketKeyPrefix: props.destinationKeyPrefix,
                RetainOnDelete: props.retainOnDelete,
                UserMetadata: props.metadata ? mapUserMetadata(props.metadata) : undefined,
                SystemMetadata: mapSystemMetadata(props),
                DistributionId: props.distribution ? props.distribution.distributionId : undefined,
                DistributionPaths: props.distributionPaths,
            },
        });
    }
    renderSingletonUuid(memoryLimit) {
        let uuid = '8693BB64-9689-44B6-9AAF-B0CC9EB8756C';
        // if user specify a custom memory limit, define another singleton handler
        // with this configuration. otherwise, it won't be possible to use multiple
        // configurations since we have a singleton.
        if (memoryLimit) {
            if (cdk.Token.isUnresolved(memoryLimit)) {
                throw new Error('Can\'t use tokens when specifying "memoryLimit" since we use it to identify the singleton custom resource handler');
            }
            uuid += `-${memoryLimit.toString()}MiB`;
        }
        return uuid;
    }
}
exports.BucketDeployment = BucketDeployment;
/**
 * We need a custom source hash calculation since the bundle.zip file
 * contains python dependencies installed during build and results in a
 * non-deterministic behavior.
 *
 * So we just take the `src/` directory of our custom resoruce code.
 */
function calcSourceHash(srcDir) {
    const sha = crypto.createHash('sha256');
    for (const file of fs.readdirSync(srcDir)) {
        const data = fs.readFileSync(path.join(srcDir, file));
        sha.update(`<file name=${file}>`);
        sha.update(data);
        sha.update('</file>');
    }
    return sha.digest('hex');
}
/**
 * Metadata
 */
function mapUserMetadata(metadata) {
    const mapKey = (key) => key.toLowerCase().startsWith('x-amzn-meta-')
        ? key.toLowerCase()
        : `x-amzn-meta-${key.toLowerCase()}`;
    return Object.keys(metadata).reduce((o, key) => ({ ...o, [mapKey(key)]: metadata[key] }), {});
}
function mapSystemMetadata(metadata) {
    const res = {};
    if (metadata.cacheControl) {
        res['cache-control'] = metadata.cacheControl.map(c => c.value).join(', ');
    }
    if (metadata.expires) {
        res.expires = metadata.expires.value;
    }
    if (metadata.contentDisposition) {
        res['content-disposition'] = metadata.contentDisposition;
    }
    if (metadata.contentEncoding) {
        res['content-encoding'] = metadata.contentEncoding;
    }
    if (metadata.contentLanguage) {
        res['content-language'] = metadata.contentLanguage;
    }
    if (metadata.contentType) {
        res['content-type'] = metadata.contentType;
    }
    if (metadata.serverSideEncryption) {
        res.sse = metadata.serverSideEncryption;
    }
    if (metadata.storageClass) {
        res['storage-class'] = metadata.storageClass;
    }
    if (metadata.websiteRedirectLocation) {
        res['website-redirect'] = metadata.websiteRedirectLocation;
    }
    if (metadata.serverSideEncryptionAwsKmsKeyId) {
        res['sse-kms-key-id'] = metadata.serverSideEncryptionAwsKmsKeyId;
    }
    if (metadata.serverSideEncryptionCustomerAlgorithm) {
        res['sse-c-copy-source'] = metadata.serverSideEncryptionCustomerAlgorithm;
    }
    return Object.keys(res).length === 0 ? undefined : res;
}
/**
 * Used for HTTP cache-control header, which influences downstream caches.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class CacheControl {
    constructor(value) {
        this.value = value;
    }
    static mustRevalidate() { return new CacheControl('must-revalidate'); }
    static noCache() { return new CacheControl('no-cache'); }
    static noTransform() { return new CacheControl('no-transform'); }
    static setPublic() { return new CacheControl('public'); }
    static setPrivate() { return new CacheControl('private'); }
    static proxyRevalidate() { return new CacheControl('proxy-revalidate'); }
    static maxAge(t) { return new CacheControl(`max-age=${t.toSeconds()}`); }
    static sMaxAge(t) { return new CacheControl(`s-maxage=${t.toSeconds()}`); }
    static fromString(s) { return new CacheControl(s); }
}
exports.CacheControl = CacheControl;
/**
 * Indicates whether server-side encryption is enabled for the object, and whether that encryption is
 * from the AWS Key Management Service (AWS KMS) or from Amazon S3 managed encryption (SSE-S3).
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var ServerSideEncryption;
(function (ServerSideEncryption) {
    ServerSideEncryption["AES_256"] = "AES256";
    ServerSideEncryption["AWS_KMS"] = "aws:kms";
})(ServerSideEncryption = exports.ServerSideEncryption || (exports.ServerSideEncryption = {}));
/**
 * Storage class used for storing the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
var StorageClass;
(function (StorageClass) {
    StorageClass["STANDARD"] = "STANDARD";
    StorageClass["REDUCED_REDUNDANCY"] = "REDUCED_REDUNDANCY";
    StorageClass["STANDARD_IA"] = "STANDARD_IA";
    StorageClass["ONEZONE_IA"] = "ONEZONE_IA";
    StorageClass["INTELLIGENT_TIERING"] = "INTELLIGENT_TIERING";
    StorageClass["GLACIER"] = "GLACIER";
    StorageClass["DEEP_ARCHIVE"] = "DEEP_ARCHIVE";
})(StorageClass = exports.StorageClass || (exports.StorageClass = {}));
/**
 * Used for HTTP expires header, which influences downstream caches. Does NOT influence deletion of the object.
 * @see https://docs.aws.amazon.com/AmazonS3/latest/dev/UsingMetadata.html#SysMetadata
 */
class Expires {
    constructor(value) {
        this.value = value;
    }
    /**
     * Expire at the specified date
     * @param d date to expire at
     */
    static atDate(d) { return new Expires(d.toUTCString()); }
    /**
     * Expire at the specified timestamp
     * @param t timestamp in unix milliseconds
     */
    static atTimestamp(t) { return Expires.atDate(new Date(t)); }
    /**
     * Expire once the specified duration has passed since deployment time
     * @param t the duration to wait before expiring
     */
    static after(t) { return Expires.atDate(new Date(now + t.toMilliseconds())); }
    static fromString(s) { return new Expires(s); }
}
exports.Expires = Expires;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYnVja2V0LWRlcGxveW1lbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJidWNrZXQtZGVwbG95bWVudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUNBLHFDQUFxQyxDQUFDLG1EQUFtRDtBQUN6RiwyQ0FBMkMsQ0FBQyxzREFBc0Q7QUFFbEcsa0NBQWtDLENBQUMsZ0RBQWdEO0FBQ25GLGlDQUFpQztBQUNqQyx5QkFBeUI7QUFDekIsNkJBQTZCO0FBRTdCLE1BQU0sR0FBRyxHQUFHLElBQUksQ0FBQyxHQUFHLEVBQUUsQ0FBQztBQUN2QixNQUFNLGlCQUFpQixHQUFHLElBQUksQ0FBQyxJQUFJLENBQUMsU0FBUyxFQUFFLElBQUksRUFBRSxRQUFRLEVBQUUsWUFBWSxDQUFDLENBQUM7QUFDN0UsTUFBTSxzQkFBc0IsR0FBRyxJQUFJLENBQUMsSUFBSSxDQUFDLFNBQVMsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLEtBQUssQ0FBQyxDQUFDO0FBb0kzRSxNQUFhLGdCQUFpQixTQUFRLEdBQUcsQ0FBQyxTQUFTO0lBQy9DLFlBQVksS0FBb0IsRUFBRSxFQUFVLEVBQUUsS0FBNEI7UUFDdEUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztRQUNqQixJQUFJLEtBQUssQ0FBQyxpQkFBaUIsSUFBSSxDQUFDLEtBQUssQ0FBQyxZQUFZLEVBQUU7WUFDaEQsTUFBTSxJQUFJLEtBQUssQ0FBQyxvRUFBb0UsQ0FBQyxDQUFDO1NBQ3pGO1FBQ0QsTUFBTSxVQUFVLEdBQUcsY0FBYyxDQUFDLHNCQUFzQixDQUFDLENBQUM7UUFDMUQsTUFBTSxPQUFPLEdBQUcsSUFBSSxNQUFNLENBQUMsaUJBQWlCLENBQUMsSUFBSSxFQUFFLHVCQUF1QixFQUFFO1lBQ3hFLElBQUksRUFBRSxJQUFJLENBQUMsbUJBQW1CLENBQUMsS0FBSyxDQUFDLFdBQVcsQ0FBQztZQUNqRCxJQUFJLEVBQUUsTUFBTSxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsaUJBQWlCLEVBQUUsRUFBRSxVQUFVLEVBQUUsQ0FBQztZQUM5RCxPQUFPLEVBQUUsTUFBTSxDQUFDLE9BQU8sQ0FBQyxVQUFVO1lBQ2xDLE9BQU8sRUFBRSxlQUFlO1lBQ3hCLGFBQWEsRUFBRSw2QkFBNkI7WUFDNUMsT0FBTyxFQUFFLEdBQUcsQ0FBQyxRQUFRLENBQUMsT0FBTyxDQUFDLEVBQUUsQ0FBQztZQUNqQyxJQUFJLEVBQUUsS0FBSyxDQUFDLElBQUk7WUFDaEIsVUFBVSxFQUFFLEtBQUssQ0FBQyxXQUFXO1NBQ2hDLENBQUMsQ0FBQztRQUNILE1BQU0sT0FBTyxHQUFtQixLQUFLLENBQUMsT0FBTyxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQWUsRUFBRSxFQUFFLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQzFGLE9BQU8sQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUFDO1FBQzVELEtBQUssQ0FBQyxpQkFBaUIsQ0FBQyxjQUFjLENBQUMsT0FBTyxDQUFDLENBQUM7UUFDaEQsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO1lBQ3BCLE9BQU8sQ0FBQyxlQUFlLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO2dCQUM1QyxNQUFNLEVBQUUsR0FBRyxDQUFDLE1BQU0sQ0FBQyxLQUFLO2dCQUN4QixPQUFPLEVBQUUsQ0FBQyw0QkFBNEIsRUFBRSwrQkFBK0IsQ0FBQztnQkFDeEUsU0FBUyxFQUFFLENBQUMsR0FBRyxDQUFDO2FBQ25CLENBQUMsQ0FBQyxDQUFDO1NBQ1A7UUFDRCxJQUFJLEdBQUcsQ0FBQyxjQUFjLENBQUMsSUFBSSxFQUFFLGdCQUFnQixFQUFFO1lBQzNDLFlBQVksRUFBRSxPQUFPLENBQUMsV0FBVztZQUNqQyxZQUFZLEVBQUUsNkJBQTZCO1lBQzNDLFVBQVUsRUFBRTtnQkFDUixpQkFBaUIsRUFBRSxPQUFPLENBQUMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUM7Z0JBQ2xFLGdCQUFnQixFQUFFLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxDQUFDLEVBQUUsQ0FBQyxNQUFNLENBQUMsWUFBWSxDQUFDO2dCQUM1RCxxQkFBcUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCLENBQUMsVUFBVTtnQkFDekQsMEJBQTBCLEVBQUUsS0FBSyxDQUFDLG9CQUFvQjtnQkFDdEQsY0FBYyxFQUFFLEtBQUssQ0FBQyxjQUFjO2dCQUNwQyxZQUFZLEVBQUUsS0FBSyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsQ0FBQyxDQUFDLENBQUMsU0FBUztnQkFDMUUsY0FBYyxFQUFFLGlCQUFpQixDQUFDLEtBQUssQ0FBQztnQkFDeEMsY0FBYyxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxTQUFTO2dCQUNsRixpQkFBaUIsRUFBRSxLQUFLLENBQUMsaUJBQWlCO2FBQzdDO1NBQ0osQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNPLG1CQUFtQixDQUFDLFdBQW9CO1FBQzVDLElBQUksSUFBSSxHQUFHLHNDQUFzQyxDQUFDO1FBQ2xELDBFQUEwRTtRQUMxRSwyRUFBMkU7UUFDM0UsNENBQTRDO1FBQzVDLElBQUksV0FBVyxFQUFFO1lBQ2IsSUFBSSxHQUFHLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQyxXQUFXLENBQUMsRUFBRTtnQkFDckMsTUFBTSxJQUFJLEtBQUssQ0FBQyxtSEFBbUgsQ0FBQyxDQUFDO2FBQ3hJO1lBQ0QsSUFBSSxJQUFJLElBQUksV0FBVyxDQUFDLFFBQVEsRUFBRSxLQUFLLENBQUM7U0FDM0M7UUFDRCxPQUFPLElBQUksQ0FBQztJQUNoQixDQUFDO0NBQ0o7QUF4REQsNENBd0RDO0FBQ0Q7Ozs7OztHQU1HO0FBQ0gsU0FBUyxjQUFjLENBQUMsTUFBYztJQUNsQyxNQUFNLEdBQUcsR0FBRyxNQUFNLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxDQUFDO0lBQ3hDLEtBQUssTUFBTSxJQUFJLElBQUksRUFBRSxDQUFDLFdBQVcsQ0FBQyxNQUFNLENBQUMsRUFBRTtRQUN2QyxNQUFNLElBQUksR0FBRyxFQUFFLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLElBQUksQ0FBQyxDQUFDLENBQUM7UUFDdEQsR0FBRyxDQUFDLE1BQU0sQ0FBQyxjQUFjLElBQUksR0FBRyxDQUFDLENBQUM7UUFDbEMsR0FBRyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUNqQixHQUFHLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxDQUFDO0tBQ3pCO0lBQ0QsT0FBTyxHQUFHLENBQUMsTUFBTSxDQUFDLEtBQUssQ0FBQyxDQUFDO0FBQzdCLENBQUM7QUFDRDs7R0FFRztBQUNILFNBQVMsZUFBZSxDQUFDLFFBQW1DO0lBQ3hELE1BQU0sTUFBTSxHQUFHLENBQUMsR0FBVyxFQUFFLEVBQUUsQ0FBQyxHQUFHLENBQUMsV0FBVyxFQUFFLENBQUMsVUFBVSxDQUFDLGNBQWMsQ0FBQztRQUN4RSxDQUFDLENBQUMsR0FBRyxDQUFDLFdBQVcsRUFBRTtRQUNuQixDQUFDLENBQUMsZUFBZSxHQUFHLENBQUMsV0FBVyxFQUFFLEVBQUUsQ0FBQztJQUN6QyxPQUFPLE1BQU0sQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxFQUFFLEdBQUcsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFFLEdBQUcsQ0FBQyxFQUFFLENBQUMsTUFBTSxDQUFDLEdBQUcsQ0FBQyxDQUFDLEVBQUUsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQztBQUNsRyxDQUFDO0FBQ0QsU0FBUyxpQkFBaUIsQ0FBQyxRQUErQjtJQUN0RCxNQUFNLEdBQUcsR0FFTCxFQUFFLENBQUM7SUFDUCxJQUFJLFFBQVEsQ0FBQyxZQUFZLEVBQUU7UUFDdkIsR0FBRyxDQUFDLGVBQWUsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxDQUFDLElBQUksQ0FBQyxJQUFJLENBQUMsQ0FBQztLQUM3RTtJQUNELElBQUksUUFBUSxDQUFDLE9BQU8sRUFBRTtRQUNsQixHQUFHLENBQUMsT0FBTyxHQUFHLFFBQVEsQ0FBQyxPQUFPLENBQUMsS0FBSyxDQUFDO0tBQ3hDO0lBQ0QsSUFBSSxRQUFRLENBQUMsa0JBQWtCLEVBQUU7UUFDN0IsR0FBRyxDQUFDLHFCQUFxQixDQUFDLEdBQUcsUUFBUSxDQUFDLGtCQUFrQixDQUFDO0tBQzVEO0lBQ0QsSUFBSSxRQUFRLENBQUMsZUFBZSxFQUFFO1FBQzFCLEdBQUcsQ0FBQyxrQkFBa0IsQ0FBQyxHQUFHLFFBQVEsQ0FBQyxlQUFlLENBQUM7S0FDdEQ7SUFDRCxJQUFJLFFBQVEsQ0FBQyxlQUFlLEVBQUU7UUFDMUIsR0FBRyxDQUFDLGtCQUFrQixDQUFDLEdBQUcsUUFBUSxDQUFDLGVBQWUsQ0FBQztLQUN0RDtJQUNELElBQUksUUFBUSxDQUFDLFdBQVcsRUFBRTtRQUN0QixHQUFHLENBQUMsY0FBYyxDQUFDLEdBQUcsUUFBUSxDQUFDLFdBQVcsQ0FBQztLQUM5QztJQUNELElBQUksUUFBUSxDQUFDLG9CQUFvQixFQUFFO1FBQy9CLEdBQUcsQ0FBQyxHQUFHLEdBQUcsUUFBUSxDQUFDLG9CQUFvQixDQUFDO0tBQzNDO0lBQ0QsSUFBSSxRQUFRLENBQUMsWUFBWSxFQUFFO1FBQ3ZCLEdBQUcsQ0FBQyxlQUFlLENBQUMsR0FBRyxRQUFRLENBQUMsWUFBWSxDQUFDO0tBQ2hEO0lBQ0QsSUFBSSxRQUFRLENBQUMsdUJBQXVCLEVBQUU7UUFDbEMsR0FBRyxDQUFDLGtCQUFrQixDQUFDLEdBQUcsUUFBUSxDQUFDLHVCQUF1QixDQUFDO0tBQzlEO0lBQ0QsSUFBSSxRQUFRLENBQUMsK0JBQStCLEVBQUU7UUFDMUMsR0FBRyxDQUFDLGdCQUFnQixDQUFDLEdBQUcsUUFBUSxDQUFDLCtCQUErQixDQUFDO0tBQ3BFO0lBQ0QsSUFBSSxRQUFRLENBQUMscUNBQXFDLEVBQUU7UUFDaEQsR0FBRyxDQUFDLG1CQUFtQixDQUFDLEdBQUcsUUFBUSxDQUFDLHFDQUFxQyxDQUFDO0tBQzdFO0lBQ0QsT0FBTyxNQUFNLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDLE1BQU0sS0FBSyxDQUFDLENBQUMsQ0FBQyxDQUFDLFNBQVMsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDO0FBQzNELENBQUM7QUFDRDs7O0dBR0c7QUFDSCxNQUFhLFlBQVk7SUFVckIsWUFBb0MsS0FBVTtRQUFWLFVBQUssR0FBTCxLQUFLLENBQUs7SUFBSSxDQUFDO0lBVDVDLE1BQU0sQ0FBQyxjQUFjLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxpQkFBaUIsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN2RSxNQUFNLENBQUMsT0FBTyxLQUFLLE9BQU8sSUFBSSxZQUFZLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3pELE1BQU0sQ0FBQyxXQUFXLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxjQUFjLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDakUsTUFBTSxDQUFDLFNBQVMsS0FBSyxPQUFPLElBQUksWUFBWSxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN6RCxNQUFNLENBQUMsVUFBVSxLQUFLLE9BQU8sSUFBSSxZQUFZLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzNELE1BQU0sQ0FBQyxlQUFlLEtBQUssT0FBTyxJQUFJLFlBQVksQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQztJQUN6RSxNQUFNLENBQUMsTUFBTSxDQUFDLENBQWUsSUFBSSxPQUFPLElBQUksWUFBWSxDQUFDLFdBQVcsQ0FBQyxDQUFDLFNBQVMsRUFBRSxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdkYsTUFBTSxDQUFDLE9BQU8sQ0FBQyxDQUFlLElBQUksT0FBTyxJQUFJLFlBQVksQ0FBQyxZQUFZLENBQUMsQ0FBQyxTQUFTLEVBQUUsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQ3pGLE1BQU0sQ0FBQyxVQUFVLENBQUMsQ0FBUyxJQUFJLE9BQU8sSUFBSSxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0NBRXRFO0FBWEQsb0NBV0M7QUFDRDs7OztHQUlHO0FBQ0gsSUFBWSxvQkFHWDtBQUhELFdBQVksb0JBQW9CO0lBQzVCLDBDQUFrQixDQUFBO0lBQ2xCLDJDQUFtQixDQUFBO0FBQ3ZCLENBQUMsRUFIVyxvQkFBb0IsR0FBcEIsNEJBQW9CLEtBQXBCLDRCQUFvQixRQUcvQjtBQUNEOzs7R0FHRztBQUNILElBQVksWUFRWDtBQVJELFdBQVksWUFBWTtJQUNwQixxQ0FBcUIsQ0FBQTtJQUNyQix5REFBeUMsQ0FBQTtJQUN6QywyQ0FBMkIsQ0FBQTtJQUMzQix5Q0FBeUIsQ0FBQTtJQUN6QiwyREFBMkMsQ0FBQTtJQUMzQyxtQ0FBbUIsQ0FBQTtJQUNuQiw2Q0FBNkIsQ0FBQTtBQUNqQyxDQUFDLEVBUlcsWUFBWSxHQUFaLG9CQUFZLEtBQVosb0JBQVksUUFRdkI7QUFDRDs7O0dBR0c7QUFDSCxNQUFhLE9BQU87SUFpQmhCLFlBQW9DLEtBQVU7UUFBVixVQUFLLEdBQUwsS0FBSyxDQUFLO0lBQUksQ0FBQztJQWhCbkQ7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLE1BQU0sQ0FBQyxDQUFPLElBQUksT0FBTyxJQUFJLE9BQU8sQ0FBQyxDQUFDLENBQUMsV0FBVyxFQUFFLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDdEU7OztPQUdHO0lBQ0ksTUFBTSxDQUFDLFdBQVcsQ0FBQyxDQUFTLElBQUksT0FBTyxPQUFPLENBQUMsTUFBTSxDQUFDLElBQUksSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzVFOzs7T0FHRztJQUNJLE1BQU0sQ0FBQyxLQUFLLENBQUMsQ0FBZSxJQUFJLE9BQU8sT0FBTyxDQUFDLE1BQU0sQ0FBQyxJQUFJLElBQUksQ0FBQyxHQUFHLEdBQUcsQ0FBQyxDQUFDLGNBQWMsRUFBRSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7SUFDNUYsTUFBTSxDQUFDLFVBQVUsQ0FBQyxDQUFTLElBQUksT0FBTyxJQUFJLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUM7Q0FFakU7QUFsQkQsMEJBa0JDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2xvdWRmcm9udCBmcm9tIFwiLi4vLi4vYXdzLWNsb3VkZnJvbnRcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZGZyb250J1xuaW1wb3J0ICogYXMgaWFtIGZyb20gXCIuLi8uLi9hd3MtaWFtXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtaWFtJ1xuaW1wb3J0ICogYXMgbGFtYmRhIGZyb20gXCIuLi8uLi9hd3MtbGFtYmRhXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtbGFtYmRhJ1xuaW1wb3J0ICogYXMgczMgZnJvbSBcIi4uLy4uL2F3cy1zM1wiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLXMzJ1xuaW1wb3J0ICogYXMgY2RrIGZyb20gXCIuLi8uLi9jb3JlXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9jb3JlJ1xuaW1wb3J0ICogYXMgY3J5cHRvIGZyb20gJ2NyeXB0byc7XG5pbXBvcnQgKiBhcyBmcyBmcm9tICdmcyc7XG5pbXBvcnQgKiBhcyBwYXRoIGZyb20gJ3BhdGgnO1xuaW1wb3J0IHsgSVNvdXJjZSwgU291cmNlQ29uZmlnIH0gZnJvbSAnLi9zb3VyY2UnO1xuY29uc3Qgbm93ID0gRGF0ZS5ub3coKTtcbmNvbnN0IGhhbmRsZXJDb2RlQnVuZGxlID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJ2xhbWJkYScsICdidW5kbGUuemlwJyk7XG5jb25zdCBoYW5kbGVyU291cmNlRGlyZWN0b3J5ID0gcGF0aC5qb2luKF9fZGlybmFtZSwgJy4uJywgJ2xhbWJkYScsICdzcmMnKTtcbmV4cG9ydCBpbnRlcmZhY2UgQnVja2V0RGVwbG95bWVudFByb3BzIHtcbiAgICAvKipcbiAgICAgKiBUaGUgc291cmNlcyBmcm9tIHdoaWNoIHRvIGRlcGxveSB0aGUgY29udGVudHMgb2YgdGhpcyBidWNrZXQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgc291cmNlczogSVNvdXJjZVtdO1xuICAgIC8qKlxuICAgICAqIFRoZSBTMyBidWNrZXQgdG8gc3luYyB0aGUgY29udGVudHMgb2YgdGhlIHppcCBmaWxlIHRvLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlc3RpbmF0aW9uQnVja2V0OiBzMy5JQnVja2V0O1xuICAgIC8qKlxuICAgICAqIEtleSBwcmVmaXggaW4gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFwiL1wiICh1bnppcCB0byByb290IG9mIHRoZSBkZXN0aW5hdGlvbiBidWNrZXQpXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVzdGluYXRpb25LZXlQcmVmaXg/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogSWYgdGhpcyBpcyBzZXQgdG8gXCJmYWxzZVwiLCB0aGUgZGVzdGluYXRpb24gZmlsZXMgd2lsbCBiZSBkZWxldGVkIHdoZW4gdGhlXG4gICAgICogcmVzb3VyY2UgaXMgZGVsZXRlZCBvciB0aGUgZGVzdGluYXRpb24gaXMgdXBkYXRlZC5cbiAgICAgKlxuICAgICAqIE5PVElDRTogaWYgdGhpcyBpcyBzZXQgdG8gXCJmYWxzZVwiIGFuZCBkZXN0aW5hdGlvbiBidWNrZXQvcHJlZml4IGlzIHVwZGF0ZWQsXG4gICAgICogYWxsIGZpbGVzIGluIHRoZSBwcmV2aW91cyBkZXN0aW5hdGlvbiB3aWxsIGZpcnN0IGJlIGRlbGV0ZWQgYW5kIHRoZW5cbiAgICAgKiB1cGxvYWRlZCB0byB0aGUgbmV3IGRlc3RpbmF0aW9uIGxvY2F0aW9uLiBUaGlzIGNvdWxkIGhhdmUgYXZhaWxhYmxpdHlcbiAgICAgKiBpbXBsaWNhdGlvbnMgb24geW91ciB1c2Vycy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHRydWUgLSB3aGVuIHJlc291cmNlIGlzIGRlbGV0ZWQvdXBkYXRlZCwgZmlsZXMgYXJlIHJldGFpbmVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcmV0YWluT25EZWxldGU/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRoZSBDbG91ZEZyb250IGRpc3RyaWJ1dGlvbiB1c2luZyB0aGUgZGVzdGluYXRpb24gYnVja2V0IGFzIGFuIG9yaWdpbi5cbiAgICAgKiBGaWxlcyBpbiB0aGUgZGlzdHJpYnV0aW9uJ3MgZWRnZSBjYWNoZXMgd2lsbCBiZSBpbnZhbGlkYXRlZCBhZnRlclxuICAgICAqIGZpbGVzIGFyZSB1cGxvYWRlZCB0byB0aGUgZGVzdGluYXRpb24gYnVja2V0LlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBpbnZhbGlkYXRpb24gb2NjdXJzXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGlzdHJpYnV0aW9uPzogY2xvdWRmcm9udC5JRGlzdHJpYnV0aW9uO1xuICAgIC8qKlxuICAgICAqIFRoZSBmaWxlIHBhdGhzIHRvIGludmFsaWRhdGUgaW4gdGhlIENsb3VkRnJvbnQgZGlzdHJpYnV0aW9uLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBbGwgZmlsZXMgdW5kZXIgdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldCBrZXkgcHJlZml4IHdpbGwgYmUgaW52YWxpZGF0ZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGlzdHJpYnV0aW9uUGF0aHM/OiBzdHJpbmdbXTtcbiAgICAvKipcbiAgICAgKiBUaGUgYW1vdW50IG9mIG1lbW9yeSAoaW4gTWlCKSB0byBhbGxvY2F0ZSB0byB0aGUgQVdTIExhbWJkYSBmdW5jdGlvbiB3aGljaFxuICAgICAqIHJlcGxpY2F0ZXMgdGhlIGZpbGVzIGZyb20gdGhlIENESyBidWNrZXQgdG8gdGhlIGRlc3RpbmF0aW9uIGJ1Y2tldC5cbiAgICAgKlxuICAgICAqIElmIHlvdSBhcmUgZGVwbG95aW5nIGxhcmdlIGZpbGVzLCB5b3Ugd2lsbCBuZWVkIHRvIGluY3JlYXNlIHRoaXMgbnVtYmVyXG4gICAgICogYWNjb3JkaW5nbHkuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAxMjhcbiAgICAgKi9cbiAgICByZWFkb25seSBtZW1vcnlMaW1pdD86IG51bWJlcjtcbiAgICAvKipcbiAgICAgKiBFeGVjdXRpb24gcm9sZSBhc3NvY2lhdGVkIHdpdGggdGhpcyBmdW5jdGlvblxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBIHJvbGUgaXMgYXV0b21hdGljYWxseSBjcmVhdGVkXG4gICAgICovXG4gICAgcmVhZG9ubHkgcm9sZT86IGlhbS5JUm9sZTtcbiAgICAvKipcbiAgICAgKiBVc2VyLWRlZmluZWQgb2JqZWN0IG1ldGFkYXRhIHRvIGJlIHNldCBvbiBhbGwgb2JqZWN0cyBpbiB0aGUgZGVwbG95bWVudFxuICAgICAqIEBkZWZhdWx0IC0gTm8gdXNlciBtZXRhZGF0YSBpcyBzZXRcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nTWV0YWRhdGEuaHRtbCNVc2VyTWV0YWRhdGFcbiAgICAgKi9cbiAgICByZWFkb25seSBtZXRhZGF0YT86IFVzZXJEZWZpbmVkT2JqZWN0TWV0YWRhdGE7XG4gICAgLyoqXG4gICAgICogU3lzdGVtLWRlZmluZWQgY2FjaGUtY29udHJvbCBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnQuXG4gICAgICogQGRlZmF1bHQgLSBOb3Qgc2V0LlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1N5c01ldGFkYXRhXG4gICAgICovXG4gICAgcmVhZG9ubHkgY2FjaGVDb250cm9sPzogQ2FjaGVDb250cm9sW107XG4gICAgLyoqXG4gICAgICogU3lzdGVtLWRlZmluZWQgY2FjaGUtZGlzcG9zaXRpb24gbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgICAqIEBkZWZhdWx0IC0gTm90IHNldC5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nTWV0YWRhdGEuaHRtbCNTeXNNZXRhZGF0YVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGNvbnRlbnREaXNwb3NpdGlvbj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBTeXN0ZW0tZGVmaW5lZCBjb250ZW50LWVuY29kaW5nIG1ldGFkYXRhIHRvIGJlIHNldCBvbiBhbGwgb2JqZWN0cyBpbiB0aGUgZGVwbG95bWVudC5cbiAgICAgKiBAZGVmYXVsdCAtIE5vdCBzZXQuXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAgICAgKi9cbiAgICByZWFkb25seSBjb250ZW50RW5jb2Rpbmc/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogU3lzdGVtLWRlZmluZWQgY29udGVudC1sYW5ndWFnZSBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnQuXG4gICAgICogQGRlZmF1bHQgLSBOb3Qgc2V0LlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1N5c01ldGFkYXRhXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29udGVudExhbmd1YWdlPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFN5c3RlbS1kZWZpbmVkIGNvbnRlbnQtdHlwZSBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnQuXG4gICAgICogQGRlZmF1bHQgLSBOb3Qgc2V0LlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL0FtYXpvblMzL2xhdGVzdC9kZXYvVXNpbmdNZXRhZGF0YS5odG1sI1N5c01ldGFkYXRhXG4gICAgICovXG4gICAgcmVhZG9ubHkgY29udGVudFR5cGU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogU3lzdGVtLWRlZmluZWQgZXhwaXJlcyBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnQuXG4gICAgICogQGRlZmF1bHQgLSBUaGUgb2JqZWN0cyBpbiB0aGUgZGlzdHJpYnV0aW9uIHdpbGwgbm90IGV4cGlyZS5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nTWV0YWRhdGEuaHRtbCNTeXNNZXRhZGF0YVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGV4cGlyZXM/OiBFeHBpcmVzO1xuICAgIC8qKlxuICAgICAqIFN5c3RlbS1kZWZpbmVkIHgtYW16LXNlcnZlci1zaWRlLWVuY3J5cHRpb24gbWV0YWRhdGEgdG8gYmUgc2V0IG9uIGFsbCBvYmplY3RzIGluIHRoZSBkZXBsb3ltZW50LlxuICAgICAqIEBkZWZhdWx0IC0gU2VydmVyIHNpZGUgZW5jcnlwdGlvbiBpcyBub3QgdXNlZC5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nTWV0YWRhdGEuaHRtbCNTeXNNZXRhZGF0YVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHNlcnZlclNpZGVFbmNyeXB0aW9uPzogU2VydmVyU2lkZUVuY3J5cHRpb247XG4gICAgLyoqXG4gICAgICogU3lzdGVtLWRlZmluZWQgeC1hbXotc3RvcmFnZS1jbGFzcyBtZXRhZGF0YSB0byBiZSBzZXQgb24gYWxsIG9iamVjdHMgaW4gdGhlIGRlcGxveW1lbnQuXG4gICAgICogQGRlZmF1bHQgLSBEZWZhdWx0IHN0b3JhZ2UtY2xhc3MgZm9yIHRoZSBidWNrZXQgaXMgdXNlZC5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1VzaW5nTWV0YWRhdGEuaHRtbCNTeXNNZXRhZGF0YVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHN0b3JhZ2VDbGFzcz86IFN0b3JhZ2VDbGFzcztcbiAgICAvKipcbiAgICAgKiBTeXN0ZW0tZGVmaW5lZCB4LWFtei13ZWJzaXRlLXJlZGlyZWN0LWxvY2F0aW9uIG1ldGFkYXRhIHRvIGJlIHNldCBvbiBhbGwgb2JqZWN0cyBpbiB0aGUgZGVwbG95bWVudC5cbiAgICAgKiBAZGVmYXVsdCAtIE5vIHdlYnNpdGUgcmVkaXJlY3Rpb24uXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAgICAgKi9cbiAgICByZWFkb25seSB3ZWJzaXRlUmVkaXJlY3RMb2NhdGlvbj86IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBTeXN0ZW0tZGVmaW5lZCB4LWFtei1zZXJ2ZXItc2lkZS1lbmNyeXB0aW9uLWF3cy1rbXMta2V5LWlkIG1ldGFkYXRhIHRvIGJlIHNldCBvbiBhbGwgb2JqZWN0cyBpbiB0aGUgZGVwbG95bWVudC5cbiAgICAgKiBAZGVmYXVsdCAtIE5vdCBzZXQuXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAgICAgKi9cbiAgICByZWFkb25seSBzZXJ2ZXJTaWRlRW5jcnlwdGlvbkF3c0ttc0tleUlkPzogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFN5c3RlbS1kZWZpbmVkIHgtYW16LXNlcnZlci1zaWRlLWVuY3J5cHRpb24tY3VzdG9tZXItYWxnb3JpdGhtIG1ldGFkYXRhIHRvIGJlIHNldCBvbiBhbGwgb2JqZWN0cyBpbiB0aGUgZGVwbG95bWVudC5cbiAgICAgKiBXYXJuaW5nOiBUaGlzIGlzIG5vdCBhIHVzZWZ1bCBwYXJhbWV0ZXIgdW50aWwgdGhpcyBidWcgaXMgZml4ZWQ6IGh0dHBzOi8vZ2l0aHViLmNvbS9hd3MvYXdzLWNkay9pc3N1ZXMvNjA4MFxuICAgICAqIEBkZWZhdWx0IC0gTm90IHNldC5cbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25TMy9sYXRlc3QvZGV2L1NlcnZlclNpZGVFbmNyeXB0aW9uQ3VzdG9tZXJLZXlzLmh0bWwjc3NlLWMtaG93LXRvLXByb2dyYW1tYXRpY2FsbHktaW50cm9cbiAgICAgKi9cbiAgICByZWFkb25seSBzZXJ2ZXJTaWRlRW5jcnlwdGlvbkN1c3RvbWVyQWxnb3JpdGhtPzogc3RyaW5nO1xufVxuZXhwb3J0IGNsYXNzIEJ1Y2tldERlcGxveW1lbnQgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogY2RrLkNvbnN0cnVjdCwgaWQ6IHN0cmluZywgcHJvcHM6IEJ1Y2tldERlcGxveW1lbnRQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQpO1xuICAgICAgICBpZiAocHJvcHMuZGlzdHJpYnV0aW9uUGF0aHMgJiYgIXByb3BzLmRpc3RyaWJ1dGlvbikge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKCdEaXN0cmlidXRpb24gbXVzdCBiZSBzcGVjaWZpZWQgaWYgZGlzdHJpYnV0aW9uIHBhdGhzIGFyZSBzcGVjaWZpZWQnKTtcbiAgICAgICAgfVxuICAgICAgICBjb25zdCBzb3VyY2VIYXNoID0gY2FsY1NvdXJjZUhhc2goaGFuZGxlclNvdXJjZURpcmVjdG9yeSk7XG4gICAgICAgIGNvbnN0IGhhbmRsZXIgPSBuZXcgbGFtYmRhLlNpbmdsZXRvbkZ1bmN0aW9uKHRoaXMsICdDdXN0b21SZXNvdXJjZUhhbmRsZXInLCB7XG4gICAgICAgICAgICB1dWlkOiB0aGlzLnJlbmRlclNpbmdsZXRvblV1aWQocHJvcHMubWVtb3J5TGltaXQpLFxuICAgICAgICAgICAgY29kZTogbGFtYmRhLkNvZGUuZnJvbUFzc2V0KGhhbmRsZXJDb2RlQnVuZGxlLCB7IHNvdXJjZUhhc2ggfSksXG4gICAgICAgICAgICBydW50aW1lOiBsYW1iZGEuUnVudGltZS5QWVRIT05fM182LFxuICAgICAgICAgICAgaGFuZGxlcjogJ2luZGV4LmhhbmRsZXInLFxuICAgICAgICAgICAgbGFtYmRhUHVycG9zZTogJ0N1c3RvbTo6Q0RLQnVja2V0RGVwbG95bWVudCcsXG4gICAgICAgICAgICB0aW1lb3V0OiBjZGsuRHVyYXRpb24ubWludXRlcygxNSksXG4gICAgICAgICAgICByb2xlOiBwcm9wcy5yb2xlLFxuICAgICAgICAgICAgbWVtb3J5U2l6ZTogcHJvcHMubWVtb3J5TGltaXQsXG4gICAgICAgIH0pO1xuICAgICAgICBjb25zdCBzb3VyY2VzOiBTb3VyY2VDb25maWdbXSA9IHByb3BzLnNvdXJjZXMubWFwKChzb3VyY2U6IElTb3VyY2UpID0+IHNvdXJjZS5iaW5kKHRoaXMpKTtcbiAgICAgICAgc291cmNlcy5mb3JFYWNoKHNvdXJjZSA9PiBzb3VyY2UuYnVja2V0LmdyYW50UmVhZChoYW5kbGVyKSk7XG4gICAgICAgIHByb3BzLmRlc3RpbmF0aW9uQnVja2V0LmdyYW50UmVhZFdyaXRlKGhhbmRsZXIpO1xuICAgICAgICBpZiAocHJvcHMuZGlzdHJpYnV0aW9uKSB7XG4gICAgICAgICAgICBoYW5kbGVyLmFkZFRvUm9sZVBvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICAgICAgZWZmZWN0OiBpYW0uRWZmZWN0LkFMTE9XLFxuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFsnY2xvdWRmcm9udDpHZXRJbnZhbGlkYXRpb24nLCAnY2xvdWRmcm9udDpDcmVhdGVJbnZhbGlkYXRpb24nXSxcbiAgICAgICAgICAgICAgICByZXNvdXJjZXM6IFsnKiddLFxuICAgICAgICAgICAgfSkpO1xuICAgICAgICB9XG4gICAgICAgIG5ldyBjZGsuQ3VzdG9tUmVzb3VyY2UodGhpcywgJ0N1c3RvbVJlc291cmNlJywge1xuICAgICAgICAgICAgc2VydmljZVRva2VuOiBoYW5kbGVyLmZ1bmN0aW9uQXJuLFxuICAgICAgICAgICAgcmVzb3VyY2VUeXBlOiAnQ3VzdG9tOjpDREtCdWNrZXREZXBsb3ltZW50JyxcbiAgICAgICAgICAgIHByb3BlcnRpZXM6IHtcbiAgICAgICAgICAgICAgICBTb3VyY2VCdWNrZXROYW1lczogc291cmNlcy5tYXAoc291cmNlID0+IHNvdXJjZS5idWNrZXQuYnVja2V0TmFtZSksXG4gICAgICAgICAgICAgICAgU291cmNlT2JqZWN0S2V5czogc291cmNlcy5tYXAoc291cmNlID0+IHNvdXJjZS56aXBPYmplY3RLZXkpLFxuICAgICAgICAgICAgICAgIERlc3RpbmF0aW9uQnVja2V0TmFtZTogcHJvcHMuZGVzdGluYXRpb25CdWNrZXQuYnVja2V0TmFtZSxcbiAgICAgICAgICAgICAgICBEZXN0aW5hdGlvbkJ1Y2tldEtleVByZWZpeDogcHJvcHMuZGVzdGluYXRpb25LZXlQcmVmaXgsXG4gICAgICAgICAgICAgICAgUmV0YWluT25EZWxldGU6IHByb3BzLnJldGFpbk9uRGVsZXRlLFxuICAgICAgICAgICAgICAgIFVzZXJNZXRhZGF0YTogcHJvcHMubWV0YWRhdGEgPyBtYXBVc2VyTWV0YWRhdGEocHJvcHMubWV0YWRhdGEpIDogdW5kZWZpbmVkLFxuICAgICAgICAgICAgICAgIFN5c3RlbU1ldGFkYXRhOiBtYXBTeXN0ZW1NZXRhZGF0YShwcm9wcyksXG4gICAgICAgICAgICAgICAgRGlzdHJpYnV0aW9uSWQ6IHByb3BzLmRpc3RyaWJ1dGlvbiA/IHByb3BzLmRpc3RyaWJ1dGlvbi5kaXN0cmlidXRpb25JZCA6IHVuZGVmaW5lZCxcbiAgICAgICAgICAgICAgICBEaXN0cmlidXRpb25QYXRoczogcHJvcHMuZGlzdHJpYnV0aW9uUGF0aHMsXG4gICAgICAgICAgICB9LFxuICAgICAgICB9KTtcbiAgICB9XG4gICAgcHJpdmF0ZSByZW5kZXJTaW5nbGV0b25VdWlkKG1lbW9yeUxpbWl0PzogbnVtYmVyKSB7XG4gICAgICAgIGxldCB1dWlkID0gJzg2OTNCQjY0LTk2ODktNDRCNi05QUFGLUIwQ0M5RUI4NzU2Qyc7XG4gICAgICAgIC8vIGlmIHVzZXIgc3BlY2lmeSBhIGN1c3RvbSBtZW1vcnkgbGltaXQsIGRlZmluZSBhbm90aGVyIHNpbmdsZXRvbiBoYW5kbGVyXG4gICAgICAgIC8vIHdpdGggdGhpcyBjb25maWd1cmF0aW9uLiBvdGhlcndpc2UsIGl0IHdvbid0IGJlIHBvc3NpYmxlIHRvIHVzZSBtdWx0aXBsZVxuICAgICAgICAvLyBjb25maWd1cmF0aW9ucyBzaW5jZSB3ZSBoYXZlIGEgc2luZ2xldG9uLlxuICAgICAgICBpZiAobWVtb3J5TGltaXQpIHtcbiAgICAgICAgICAgIGlmIChjZGsuVG9rZW4uaXNVbnJlc29sdmVkKG1lbW9yeUxpbWl0KSkge1xuICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2FuXFwndCB1c2UgdG9rZW5zIHdoZW4gc3BlY2lmeWluZyBcIm1lbW9yeUxpbWl0XCIgc2luY2Ugd2UgdXNlIGl0IHRvIGlkZW50aWZ5IHRoZSBzaW5nbGV0b24gY3VzdG9tIHJlc291cmNlIGhhbmRsZXInKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIHV1aWQgKz0gYC0ke21lbW9yeUxpbWl0LnRvU3RyaW5nKCl9TWlCYDtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4gdXVpZDtcbiAgICB9XG59XG4vKipcbiAqIFdlIG5lZWQgYSBjdXN0b20gc291cmNlIGhhc2ggY2FsY3VsYXRpb24gc2luY2UgdGhlIGJ1bmRsZS56aXAgZmlsZVxuICogY29udGFpbnMgcHl0aG9uIGRlcGVuZGVuY2llcyBpbnN0YWxsZWQgZHVyaW5nIGJ1aWxkIGFuZCByZXN1bHRzIGluIGFcbiAqIG5vbi1kZXRlcm1pbmlzdGljIGJlaGF2aW9yLlxuICpcbiAqIFNvIHdlIGp1c3QgdGFrZSB0aGUgYHNyYy9gIGRpcmVjdG9yeSBvZiBvdXIgY3VzdG9tIHJlc29ydWNlIGNvZGUuXG4gKi9cbmZ1bmN0aW9uIGNhbGNTb3VyY2VIYXNoKHNyY0Rpcjogc3RyaW5nKTogc3RyaW5nIHtcbiAgICBjb25zdCBzaGEgPSBjcnlwdG8uY3JlYXRlSGFzaCgnc2hhMjU2Jyk7XG4gICAgZm9yIChjb25zdCBmaWxlIG9mIGZzLnJlYWRkaXJTeW5jKHNyY0RpcikpIHtcbiAgICAgICAgY29uc3QgZGF0YSA9IGZzLnJlYWRGaWxlU3luYyhwYXRoLmpvaW4oc3JjRGlyLCBmaWxlKSk7XG4gICAgICAgIHNoYS51cGRhdGUoYDxmaWxlIG5hbWU9JHtmaWxlfT5gKTtcbiAgICAgICAgc2hhLnVwZGF0ZShkYXRhKTtcbiAgICAgICAgc2hhLnVwZGF0ZSgnPC9maWxlPicpO1xuICAgIH1cbiAgICByZXR1cm4gc2hhLmRpZ2VzdCgnaGV4Jyk7XG59XG4vKipcbiAqIE1ldGFkYXRhXG4gKi9cbmZ1bmN0aW9uIG1hcFVzZXJNZXRhZGF0YShtZXRhZGF0YTogVXNlckRlZmluZWRPYmplY3RNZXRhZGF0YSkge1xuICAgIGNvbnN0IG1hcEtleSA9IChrZXk6IHN0cmluZykgPT4ga2V5LnRvTG93ZXJDYXNlKCkuc3RhcnRzV2l0aCgneC1hbXpuLW1ldGEtJylcbiAgICAgICAgPyBrZXkudG9Mb3dlckNhc2UoKVxuICAgICAgICA6IGB4LWFtem4tbWV0YS0ke2tleS50b0xvd2VyQ2FzZSgpfWA7XG4gICAgcmV0dXJuIE9iamVjdC5rZXlzKG1ldGFkYXRhKS5yZWR1Y2UoKG8sIGtleSkgPT4gKHsgLi4ubywgW21hcEtleShrZXkpXTogbWV0YWRhdGFba2V5XSB9KSwge30pO1xufVxuZnVuY3Rpb24gbWFwU3lzdGVtTWV0YWRhdGEobWV0YWRhdGE6IEJ1Y2tldERlcGxveW1lbnRQcm9wcykge1xuICAgIGNvbnN0IHJlczoge1xuICAgICAgICBba2V5OiBzdHJpbmddOiBzdHJpbmc7XG4gICAgfSA9IHt9O1xuICAgIGlmIChtZXRhZGF0YS5jYWNoZUNvbnRyb2wpIHtcbiAgICAgICAgcmVzWydjYWNoZS1jb250cm9sJ10gPSBtZXRhZGF0YS5jYWNoZUNvbnRyb2wubWFwKGMgPT4gYy52YWx1ZSkuam9pbignLCAnKTtcbiAgICB9XG4gICAgaWYgKG1ldGFkYXRhLmV4cGlyZXMpIHtcbiAgICAgICAgcmVzLmV4cGlyZXMgPSBtZXRhZGF0YS5leHBpcmVzLnZhbHVlO1xuICAgIH1cbiAgICBpZiAobWV0YWRhdGEuY29udGVudERpc3Bvc2l0aW9uKSB7XG4gICAgICAgIHJlc1snY29udGVudC1kaXNwb3NpdGlvbiddID0gbWV0YWRhdGEuY29udGVudERpc3Bvc2l0aW9uO1xuICAgIH1cbiAgICBpZiAobWV0YWRhdGEuY29udGVudEVuY29kaW5nKSB7XG4gICAgICAgIHJlc1snY29udGVudC1lbmNvZGluZyddID0gbWV0YWRhdGEuY29udGVudEVuY29kaW5nO1xuICAgIH1cbiAgICBpZiAobWV0YWRhdGEuY29udGVudExhbmd1YWdlKSB7XG4gICAgICAgIHJlc1snY29udGVudC1sYW5ndWFnZSddID0gbWV0YWRhdGEuY29udGVudExhbmd1YWdlO1xuICAgIH1cbiAgICBpZiAobWV0YWRhdGEuY29udGVudFR5cGUpIHtcbiAgICAgICAgcmVzWydjb250ZW50LXR5cGUnXSA9IG1ldGFkYXRhLmNvbnRlbnRUeXBlO1xuICAgIH1cbiAgICBpZiAobWV0YWRhdGEuc2VydmVyU2lkZUVuY3J5cHRpb24pIHtcbiAgICAgICAgcmVzLnNzZSA9IG1ldGFkYXRhLnNlcnZlclNpZGVFbmNyeXB0aW9uO1xuICAgIH1cbiAgICBpZiAobWV0YWRhdGEuc3RvcmFnZUNsYXNzKSB7XG4gICAgICAgIHJlc1snc3RvcmFnZS1jbGFzcyddID0gbWV0YWRhdGEuc3RvcmFnZUNsYXNzO1xuICAgIH1cbiAgICBpZiAobWV0YWRhdGEud2Vic2l0ZVJlZGlyZWN0TG9jYXRpb24pIHtcbiAgICAgICAgcmVzWyd3ZWJzaXRlLXJlZGlyZWN0J10gPSBtZXRhZGF0YS53ZWJzaXRlUmVkaXJlY3RMb2NhdGlvbjtcbiAgICB9XG4gICAgaWYgKG1ldGFkYXRhLnNlcnZlclNpZGVFbmNyeXB0aW9uQXdzS21zS2V5SWQpIHtcbiAgICAgICAgcmVzWydzc2Uta21zLWtleS1pZCddID0gbWV0YWRhdGEuc2VydmVyU2lkZUVuY3J5cHRpb25Bd3NLbXNLZXlJZDtcbiAgICB9XG4gICAgaWYgKG1ldGFkYXRhLnNlcnZlclNpZGVFbmNyeXB0aW9uQ3VzdG9tZXJBbGdvcml0aG0pIHtcbiAgICAgICAgcmVzWydzc2UtYy1jb3B5LXNvdXJjZSddID0gbWV0YWRhdGEuc2VydmVyU2lkZUVuY3J5cHRpb25DdXN0b21lckFsZ29yaXRobTtcbiAgICB9XG4gICAgcmV0dXJuIE9iamVjdC5rZXlzKHJlcykubGVuZ3RoID09PSAwID8gdW5kZWZpbmVkIDogcmVzO1xufVxuLyoqXG4gKiBVc2VkIGZvciBIVFRQIGNhY2hlLWNvbnRyb2wgaGVhZGVyLCB3aGljaCBpbmZsdWVuY2VzIGRvd25zdHJlYW0gY2FjaGVzLlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAqL1xuZXhwb3J0IGNsYXNzIENhY2hlQ29udHJvbCB7XG4gICAgcHVibGljIHN0YXRpYyBtdXN0UmV2YWxpZGF0ZSgpIHsgcmV0dXJuIG5ldyBDYWNoZUNvbnRyb2woJ211c3QtcmV2YWxpZGF0ZScpOyB9XG4gICAgcHVibGljIHN0YXRpYyBub0NhY2hlKCkgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbCgnbm8tY2FjaGUnKTsgfVxuICAgIHB1YmxpYyBzdGF0aWMgbm9UcmFuc2Zvcm0oKSB7IHJldHVybiBuZXcgQ2FjaGVDb250cm9sKCduby10cmFuc2Zvcm0nKTsgfVxuICAgIHB1YmxpYyBzdGF0aWMgc2V0UHVibGljKCkgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbCgncHVibGljJyk7IH1cbiAgICBwdWJsaWMgc3RhdGljIHNldFByaXZhdGUoKSB7IHJldHVybiBuZXcgQ2FjaGVDb250cm9sKCdwcml2YXRlJyk7IH1cbiAgICBwdWJsaWMgc3RhdGljIHByb3h5UmV2YWxpZGF0ZSgpIHsgcmV0dXJuIG5ldyBDYWNoZUNvbnRyb2woJ3Byb3h5LXJldmFsaWRhdGUnKTsgfVxuICAgIHB1YmxpYyBzdGF0aWMgbWF4QWdlKHQ6IGNkay5EdXJhdGlvbikgeyByZXR1cm4gbmV3IENhY2hlQ29udHJvbChgbWF4LWFnZT0ke3QudG9TZWNvbmRzKCl9YCk7IH1cbiAgICBwdWJsaWMgc3RhdGljIHNNYXhBZ2UodDogY2RrLkR1cmF0aW9uKSB7IHJldHVybiBuZXcgQ2FjaGVDb250cm9sKGBzLW1heGFnZT0ke3QudG9TZWNvbmRzKCl9YCk7IH1cbiAgICBwdWJsaWMgc3RhdGljIGZyb21TdHJpbmcoczogc3RyaW5nKSB7IHJldHVybiBuZXcgQ2FjaGVDb250cm9sKHMpOyB9XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgdmFsdWU6IGFueSkgeyB9XG59XG4vKipcbiAqIEluZGljYXRlcyB3aGV0aGVyIHNlcnZlci1zaWRlIGVuY3J5cHRpb24gaXMgZW5hYmxlZCBmb3IgdGhlIG9iamVjdCwgYW5kIHdoZXRoZXIgdGhhdCBlbmNyeXB0aW9uIGlzXG4gKiBmcm9tIHRoZSBBV1MgS2V5IE1hbmFnZW1lbnQgU2VydmljZSAoQVdTIEtNUykgb3IgZnJvbSBBbWF6b24gUzMgbWFuYWdlZCBlbmNyeXB0aW9uIChTU0UtUzMpLlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAqL1xuZXhwb3J0IGVudW0gU2VydmVyU2lkZUVuY3J5cHRpb24ge1xuICAgIEFFU18yNTYgPSAnQUVTMjU2JyxcbiAgICBBV1NfS01TID0gJ2F3czprbXMnXG59XG4vKipcbiAqIFN0b3JhZ2UgY2xhc3MgdXNlZCBmb3Igc3RvcmluZyB0aGUgb2JqZWN0LlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAqL1xuZXhwb3J0IGVudW0gU3RvcmFnZUNsYXNzIHtcbiAgICBTVEFOREFSRCA9ICdTVEFOREFSRCcsXG4gICAgUkVEVUNFRF9SRURVTkRBTkNZID0gJ1JFRFVDRURfUkVEVU5EQU5DWScsXG4gICAgU1RBTkRBUkRfSUEgPSAnU1RBTkRBUkRfSUEnLFxuICAgIE9ORVpPTkVfSUEgPSAnT05FWk9ORV9JQScsXG4gICAgSU5URUxMSUdFTlRfVElFUklORyA9ICdJTlRFTExJR0VOVF9USUVSSU5HJyxcbiAgICBHTEFDSUVSID0gJ0dMQUNJRVInLFxuICAgIERFRVBfQVJDSElWRSA9ICdERUVQX0FSQ0hJVkUnXG59XG4vKipcbiAqIFVzZWQgZm9yIEhUVFAgZXhwaXJlcyBoZWFkZXIsIHdoaWNoIGluZmx1ZW5jZXMgZG93bnN0cmVhbSBjYWNoZXMuIERvZXMgTk9UIGluZmx1ZW5jZSBkZWxldGlvbiBvZiB0aGUgb2JqZWN0LlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjU3lzTWV0YWRhdGFcbiAqL1xuZXhwb3J0IGNsYXNzIEV4cGlyZXMge1xuICAgIC8qKlxuICAgICAqIEV4cGlyZSBhdCB0aGUgc3BlY2lmaWVkIGRhdGVcbiAgICAgKiBAcGFyYW0gZCBkYXRlIHRvIGV4cGlyZSBhdFxuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgYXREYXRlKGQ6IERhdGUpIHsgcmV0dXJuIG5ldyBFeHBpcmVzKGQudG9VVENTdHJpbmcoKSk7IH1cbiAgICAvKipcbiAgICAgKiBFeHBpcmUgYXQgdGhlIHNwZWNpZmllZCB0aW1lc3RhbXBcbiAgICAgKiBAcGFyYW0gdCB0aW1lc3RhbXAgaW4gdW5peCBtaWxsaXNlY29uZHNcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGF0VGltZXN0YW1wKHQ6IG51bWJlcikgeyByZXR1cm4gRXhwaXJlcy5hdERhdGUobmV3IERhdGUodCkpOyB9XG4gICAgLyoqXG4gICAgICogRXhwaXJlIG9uY2UgdGhlIHNwZWNpZmllZCBkdXJhdGlvbiBoYXMgcGFzc2VkIHNpbmNlIGRlcGxveW1lbnQgdGltZVxuICAgICAqIEBwYXJhbSB0IHRoZSBkdXJhdGlvbiB0byB3YWl0IGJlZm9yZSBleHBpcmluZ1xuICAgICAqL1xuICAgIHB1YmxpYyBzdGF0aWMgYWZ0ZXIodDogY2RrLkR1cmF0aW9uKSB7IHJldHVybiBFeHBpcmVzLmF0RGF0ZShuZXcgRGF0ZShub3cgKyB0LnRvTWlsbGlzZWNvbmRzKCkpKTsgfVxuICAgIHB1YmxpYyBzdGF0aWMgZnJvbVN0cmluZyhzOiBzdHJpbmcpIHsgcmV0dXJuIG5ldyBFeHBpcmVzKHMpOyB9XG4gICAgcHJpdmF0ZSBjb25zdHJ1Y3RvcihwdWJsaWMgcmVhZG9ubHkgdmFsdWU6IGFueSkgeyB9XG59XG5leHBvcnQgaW50ZXJmYWNlIFVzZXJEZWZpbmVkT2JqZWN0TWV0YWRhdGEge1xuICAgIC8qKlxuICAgICAqIEFyYml0cmFyeSBtZXRhZGF0YSBrZXktdmFsdWVzXG4gICAgICogS2V5cyBtdXN0IGJlZ2luIHdpdGggYHgtYW16bi1tZXRhLWAgKHdpbGwgYmUgYWRkZWQgYXV0b21hdGljYWxseSBpZiBub3QgcHJvdmlkZWQpXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vQW1hem9uUzMvbGF0ZXN0L2Rldi9Vc2luZ01ldGFkYXRhLmh0bWwjVXNlck1ldGFkYXRhXG4gICAgICovXG4gICAgcmVhZG9ubHkgW2tleTogc3RyaW5nXTogc3RyaW5nO1xufVxuIl19