"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cxapi = require("../../cx-api"); // Automatically re-written from '@aws-cdk/cx-api'
const fs = require("fs");
const path = require("path");
const compat_1 = require("./compat");
const ARCHIVE_EXTENSIONS = ['.zip', '.jar'];
/**
 * An asset represents a local file or directory, which is automatically uploaded to S3
 * and then can be referenced within a CDK application.
 */
class Asset extends cdk.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        // stage the asset source (conditionally).
        const staging = new cdk.AssetStaging(this, 'Stage', {
            ...props,
            sourcePath: path.resolve(props.path),
            follow: compat_1.toSymlinkFollow(props.follow),
            assetHash: (_a = props.assetHash) !== null && _a !== void 0 ? _a : props.sourceHash,
        });
        this.assetHash = staging.assetHash;
        this.sourceHash = this.assetHash;
        this.assetPath = staging.stagedPath;
        const packaging = determinePackaging(staging.sourcePath);
        // sets isZipArchive based on the type of packaging and file extension
        this.isZipArchive = packaging === cdk.FileAssetPackaging.ZIP_DIRECTORY
            ? true
            : ARCHIVE_EXTENSIONS.some(ext => staging.sourcePath.toLowerCase().endsWith(ext));
        const stack = cdk.Stack.of(this);
        const location = stack.synthesizer.addFileAsset({
            packaging,
            sourceHash: this.sourceHash,
            fileName: staging.stagedPath,
        });
        this.s3BucketName = location.bucketName;
        this.s3ObjectKey = location.objectKey;
        this.s3ObjectUrl = location.s3ObjectUrl;
        this.httpUrl = location.httpUrl;
        this.s3Url = location.httpUrl; // for backwards compatibility
        this.bucket = s3.Bucket.fromBucketName(this, 'AssetBucket', this.s3BucketName);
        for (const reader of ((_b = props.readers) !== null && _b !== void 0 ? _b : [])) {
            this.grantRead(reader);
        }
    }
    /**
     * Adds CloudFormation template metadata to the specified resource with
     * information that indicates which resource property is mapped to this local
     * asset. This can be used by tools such as SAM CLI to provide local
     * experience such as local invocation and debugging of Lambda functions.
     *
     * Asset metadata will only be included if the stack is synthesized with the
     * "aws:cdk:enable-asset-metadata" context key defined, which is the default
     * behavior when synthesizing via the CDK Toolkit.
     *
     * @see https://github.com/aws/aws-cdk/issues/1432
     *
     * @param resource The CloudFormation resource which is using this asset [disable-awslint:ref-via-interface]
     * @param resourceProperty The property name where this asset is referenced
     * (e.g. "Code" for AWS::Lambda::Function)
     */
    addResourceMetadata(resource, resourceProperty) {
        if (!this.node.tryGetContext(cxapi.ASSET_RESOURCE_METADATA_ENABLED_CONTEXT)) {
            return; // not enabled
        }
        // tell tools such as SAM CLI that the "Code" property of this resource
        // points to a local path in order to enable local invocation of this function.
        resource.cfnOptions.metadata = resource.cfnOptions.metadata || {};
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PATH_KEY] = this.assetPath;
        resource.cfnOptions.metadata[cxapi.ASSET_RESOURCE_METADATA_PROPERTY_KEY] = resourceProperty;
    }
    /**
     * Grants read permissions to the principal on the assets bucket.
     */
    grantRead(grantee) {
        // we give permissions on all files in the bucket since we don't want to
        // accidentally revoke permission on old versions when deploying a new
        // version (for example, when using Lambda traffic shifting).
        this.bucket.grantRead(grantee);
    }
}
exports.Asset = Asset;
function determinePackaging(assetPath) {
    if (!fs.existsSync(assetPath)) {
        throw new Error(`Cannot find asset at ${assetPath}`);
    }
    if (fs.statSync(assetPath).isDirectory()) {
        return cdk.FileAssetPackaging.ZIP_DIRECTORY;
    }
    if (fs.statSync(assetPath).isFile()) {
        return cdk.FileAssetPackaging.FILE;
    }
    throw new Error(`Asset ${assetPath} is expected to be either a directory or a regular file`);
}
//# sourceMappingURL=data:application/json;base64,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