"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const version_1 = require("./private/version");
/**
 * A database cluster engine. Provides mapping to the serverless application
 * used for secret rotation.
 */
class DatabaseClusterEngine {
    // tslint:disable-next-line max-line-length
    constructor(name, singleUserRotationApplication, multiUserRotationApplication, parameterGroupFamilies) {
        this.name = name;
        this.singleUserRotationApplication = singleUserRotationApplication;
        this.multiUserRotationApplication = multiUserRotationApplication;
        this.parameterGroupFamilies = parameterGroupFamilies;
    }
    /**
     * Get the latest parameter group family for this engine. Latest is determined using semver on the engine major version.
     * When `engineVersion` is specified, return the parameter group family corresponding to that engine version.
     * Return undefined if no parameter group family is defined for this engine or for the requested `engineVersion`.
     */
    parameterGroupFamily(engineVersion) {
        if (this.parameterGroupFamilies === undefined) {
            return undefined;
        }
        if (engineVersion) {
            const family = this.parameterGroupFamilies.find(x => engineVersion.startsWith(x.engineMajorVersion));
            if (family) {
                return family.parameterGroupFamily;
            }
        }
        else if (this.parameterGroupFamilies.length > 0) {
            const sorted = this.parameterGroupFamilies.slice().sort((a, b) => {
                return version_1.compare(a.engineMajorVersion, b.engineMajorVersion);
            }).reverse();
            return sorted[0].parameterGroupFamily;
        }
        return undefined;
    }
}
exports.DatabaseClusterEngine = DatabaseClusterEngine;
/* tslint:disable max-line-length */
DatabaseClusterEngine.AURORA = new DatabaseClusterEngine('aurora', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'aurora5.6' },
]);
DatabaseClusterEngine.AURORA_MYSQL = new DatabaseClusterEngine('aurora-mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.7', parameterGroupFamily: 'aurora-mysql5.7' },
]);
DatabaseClusterEngine.AURORA_POSTGRESQL = new DatabaseClusterEngine('aurora-postgresql', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.6', parameterGroupFamily: 'aurora-postgresql9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'aurora-postgresql10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'aurora-postgresql11' },
]);
//# sourceMappingURL=data:application/json;base64,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