"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const rds_generated_1 = require("./rds.generated");
/**
 * An option group
 */
class OptionGroup extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        /**
         * The connections object for the options.
         */
        this.optionConnections = {};
        const optionGroup = new rds_generated_1.CfnOptionGroup(this, 'Resource', {
            engineName: props.engine.name,
            majorEngineVersion: props.majorEngineVersion,
            optionGroupDescription: props.description || `Option group for ${props.engine.name} ${props.majorEngineVersion}`,
            optionConfigurations: this.renderConfigurations(props.configurations),
        });
        this.optionGroupName = optionGroup.ref;
    }
    /**
     * Import an existing option group.
     */
    static fromOptionGroupName(scope, id, optionGroupName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.optionGroupName = optionGroupName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Renders the option configurations specifications.
     */
    renderConfigurations(configurations) {
        const configs = [];
        for (const config of configurations) {
            let configuration = {
                optionName: config.name,
                optionSettings: config.settings && Object.entries(config.settings).map(([name, value]) => ({ name, value })),
                optionVersion: config.version,
            };
            if (config.port) {
                if (!config.vpc) {
                    throw new Error('`port` and `vpc` must be specified together.');
                }
                const securityGroup = new ec2.SecurityGroup(this, `SecurityGroup${config.name}`, {
                    description: `Security group for ${config.name} option`,
                    vpc: config.vpc,
                });
                this.optionConnections[config.name] = new ec2.Connections({
                    securityGroups: [securityGroup],
                    defaultPort: ec2.Port.tcp(config.port),
                });
                configuration = {
                    ...configuration,
                    port: config.port,
                    vpcSecurityGroupMemberships: [securityGroup.securityGroupId],
                };
            }
            configs.push(configuration);
        }
        return configs;
    }
}
exports.OptionGroup = OptionGroup;
//# sourceMappingURL=data:application/json;base64,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