"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const lambda = require("../../aws-lambda"); // Automatically re-written from '@aws-cdk/aws-lambda'
const secretsmanager = require("../../aws-secretsmanager"); // Automatically re-written from '@aws-cdk/aws-secretsmanager'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const database_secret_1 = require("./database-secret");
const endpoint_1 = require("./endpoint");
const props_1 = require("./props");
const rds_generated_1 = require("./rds.generated");
/**
 * A new or imported database instance.
 */
class DatabaseInstanceBase extends core_1.Resource {
    /**
     * Import an existing database instance.
     */
    static fromDatabaseInstanceAttributes(scope, id, attrs) {
        class Import extends DatabaseInstanceBase {
            constructor() {
                super(...arguments);
                this.defaultPort = ec2.Port.tcp(attrs.port);
                this.connections = new ec2.Connections({
                    securityGroups: attrs.securityGroups,
                    defaultPort: this.defaultPort,
                });
                this.instanceIdentifier = attrs.instanceIdentifier;
                this.dbInstanceEndpointAddress = attrs.instanceEndpointAddress;
                this.dbInstanceEndpointPort = attrs.port.toString();
                this.instanceEndpoint = new endpoint_1.Endpoint(attrs.instanceEndpointAddress, attrs.port);
            }
        }
        return new Import(scope, id);
    }
    /**
     * Defines a CloudWatch event rule which triggers for instance events. Use
     * `rule.addEventPattern(pattern)` to specify a filter.
     */
    onEvent(id, options = {}) {
        const rule = new events.Rule(this, id, options);
        rule.addEventPattern({
            source: ['aws.rds'],
            resources: [this.instanceArn],
        });
        rule.addTarget(options.target);
        return rule;
    }
    /**
     * The instance arn.
     */
    get instanceArn() {
        return core_1.Stack.of(this).formatArn({
            service: 'rds',
            resource: 'db',
            sep: ':',
            resourceName: this.instanceIdentifier,
        });
    }
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget() {
        return {
            targetId: this.instanceIdentifier,
            targetType: secretsmanager.AttachmentTargetType.RDS_DB_INSTANCE,
        };
    }
}
exports.DatabaseInstanceBase = DatabaseInstanceBase;
/**
 * A database instance engine. Provides mapping to DatabaseEngine used for
 * secret rotation.
 */
class DatabaseInstanceEngine extends props_1.DatabaseClusterEngine {
    constructor() {
        super(...arguments);
        /* tslint:enable max-line-length */
        /** To make it a compile-time error to pass a DatabaseClusterEngine where a DatabaseInstanceEngine is expected. */
        this.isDatabaseInstanceEngine = true;
    }
}
exports.DatabaseInstanceEngine = DatabaseInstanceEngine;
/* tslint:disable max-line-length */
DatabaseInstanceEngine.MARIADB = new DatabaseInstanceEngine('mariadb', secretsmanager.SecretRotationApplication.MARIADB_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MARIADB_ROTATION_MULTI_USER, [
    { engineMajorVersion: '10.0', parameterGroupFamily: 'mariadb10.0' },
    { engineMajorVersion: '10.1', parameterGroupFamily: 'mariadb10.1' },
    { engineMajorVersion: '10.2', parameterGroupFamily: 'mariadb10.2' },
    { engineMajorVersion: '10.3', parameterGroupFamily: 'mariadb10.3' },
]);
DatabaseInstanceEngine.MYSQL = new DatabaseInstanceEngine('mysql', secretsmanager.SecretRotationApplication.MYSQL_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.MYSQL_ROTATION_MULTI_USER, [
    { engineMajorVersion: '5.6', parameterGroupFamily: 'mysql5.6' },
    { engineMajorVersion: '5.7', parameterGroupFamily: 'mysql5.7' },
    { engineMajorVersion: '8.0', parameterGroupFamily: 'mysql8.0' },
]);
DatabaseInstanceEngine.ORACLE_EE = new DatabaseInstanceEngine('oracle-ee', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-ee-11.2' },
    { engineMajorVersion: '12.1', parameterGroupFamily: 'oracle-ee-12.1' },
    { engineMajorVersion: '12.2', parameterGroupFamily: 'oracle-ee-12.2' },
    { engineMajorVersion: '18', parameterGroupFamily: 'oracle-ee-18' },
    { engineMajorVersion: '19', parameterGroupFamily: 'oracle-ee-19' },
]);
DatabaseInstanceEngine.ORACLE_SE2 = new DatabaseInstanceEngine('oracle-se2', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '12.1', parameterGroupFamily: 'oracle-se2-12.1' },
    { engineMajorVersion: '12.2', parameterGroupFamily: 'oracle-se2-12.2' },
    { engineMajorVersion: '18', parameterGroupFamily: 'oracle-se2-18' },
    { engineMajorVersion: '19', parameterGroupFamily: 'oracle-se2-19' },
]);
DatabaseInstanceEngine.ORACLE_SE1 = new DatabaseInstanceEngine('oracle-se1', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-se1-11.2' },
]);
DatabaseInstanceEngine.ORACLE_SE = new DatabaseInstanceEngine('oracle-se', secretsmanager.SecretRotationApplication.ORACLE_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.ORACLE_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11.2', parameterGroupFamily: 'oracle-se-11.2' },
]);
DatabaseInstanceEngine.POSTGRES = new DatabaseInstanceEngine('postgres', secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.POSTGRES_ROTATION_MULTI_USER, [
    { engineMajorVersion: '9.3', parameterGroupFamily: 'postgres9.3' },
    { engineMajorVersion: '9.4', parameterGroupFamily: 'postgres9.4' },
    { engineMajorVersion: '9.5', parameterGroupFamily: 'postgres9.5' },
    { engineMajorVersion: '9.6', parameterGroupFamily: 'postgres9.6' },
    { engineMajorVersion: '10', parameterGroupFamily: 'postgres10' },
    { engineMajorVersion: '11', parameterGroupFamily: 'postgres11' },
]);
DatabaseInstanceEngine.SQL_SERVER_EE = new DatabaseInstanceEngine('sqlserver-ee', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-ee-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-ee-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-ee-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-ee-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_SE = new DatabaseInstanceEngine('sqlserver-se', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-se-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-se-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-se-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-se-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_EX = new DatabaseInstanceEngine('sqlserver-ex', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-ex-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-ex-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-ex-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-ex-14.0' },
]);
DatabaseInstanceEngine.SQL_SERVER_WEB = new DatabaseInstanceEngine('sqlserver-web', secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_SINGLE_USER, secretsmanager.SecretRotationApplication.SQLSERVER_ROTATION_MULTI_USER, [
    { engineMajorVersion: '11', parameterGroupFamily: 'sqlserver-web-11.0' },
    { engineMajorVersion: '12', parameterGroupFamily: 'sqlserver-web-12.0' },
    { engineMajorVersion: '13', parameterGroupFamily: 'sqlserver-web-13.0' },
    { engineMajorVersion: '14', parameterGroupFamily: 'sqlserver-web-14.0' },
]);
/**
 * The license model.
 */
var LicenseModel;
(function (LicenseModel) {
    /**
     * License included.
     */
    LicenseModel["LICENSE_INCLUDED"] = "license-included";
    /**
     * Bring your own licencse.
     */
    LicenseModel["BRING_YOUR_OWN_LICENSE"] = "bring-your-own-license";
    /**
     * General public license.
     */
    LicenseModel["GENERAL_PUBLIC_LICENSE"] = "general-public-license";
})(LicenseModel = exports.LicenseModel || (exports.LicenseModel = {}));
/**
 * The type of storage.
 */
var StorageType;
(function (StorageType) {
    /**
     * Standard.
     */
    StorageType["STANDARD"] = "standard";
    /**
     * General purpose (SSD).
     */
    StorageType["GP2"] = "gp2";
    /**
     * Provisioned IOPS (SSD).
     */
    StorageType["IO1"] = "io1";
})(StorageType = exports.StorageType || (exports.StorageType = {}));
/**
 * The retention period for Performance Insight.
 */
var PerformanceInsightRetention;
(function (PerformanceInsightRetention) {
    /**
     * Default retention period of 7 days.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["DEFAULT"] = 7] = "DEFAULT";
    /**
     * Long term retention period of 2 years.
     */
    PerformanceInsightRetention[PerformanceInsightRetention["LONG_TERM"] = 731] = "LONG_TERM";
})(PerformanceInsightRetention = exports.PerformanceInsightRetention || (exports.PerformanceInsightRetention = {}));
/**
 * A new database instance.
 */
class DatabaseInstanceNew extends DatabaseInstanceBase {
    constructor(scope, id, props) {
        super(scope, id);
        this.vpc = props.vpc;
        this.vpcPlacement = props.vpcPlacement;
        const { subnetIds } = props.vpc.selectSubnets(props.vpcPlacement);
        const subnetGroup = new rds_generated_1.CfnDBSubnetGroup(this, 'SubnetGroup', {
            dbSubnetGroupDescription: `Subnet group for ${this.node.id} database`,
            subnetIds,
        });
        const securityGroups = props.securityGroups || [new ec2.SecurityGroup(this, 'SecurityGroup', {
                description: `Security group for ${this.node.id} database`,
                vpc: props.vpc,
            })];
        this.connections = new ec2.Connections({
            securityGroups,
            defaultPort: ec2.Port.tcp(core_1.Lazy.numberValue({ produce: () => this.instanceEndpoint.port })),
        });
        let monitoringRole;
        if (props.monitoringInterval && props.monitoringInterval.toSeconds()) {
            monitoringRole = props.monitoringRole || new iam.Role(this, 'MonitoringRole', {
                assumedBy: new iam.ServicePrincipal('monitoring.rds.amazonaws.com'),
                managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AmazonRDSEnhancedMonitoringRole')],
            });
        }
        const deletionProtection = props.deletionProtection !== undefined ? props.deletionProtection : true;
        const storageType = props.storageType || StorageType.GP2;
        const iops = storageType === StorageType.IO1 ? (props.iops || 1000) : undefined;
        this.cloudwatchLogsExports = props.cloudwatchLogsExports;
        this.cloudwatchLogsRetention = props.cloudwatchLogsRetention;
        this.cloudwatchLogsRetentionRole = props.cloudwatchLogsRetentionRole;
        this.newCfnProps = {
            autoMinorVersionUpgrade: props.autoMinorVersionUpgrade,
            availabilityZone: props.multiAz ? undefined : props.availabilityZone,
            backupRetentionPeriod: props.backupRetention ? props.backupRetention.toDays() : undefined,
            copyTagsToSnapshot: props.copyTagsToSnapshot !== undefined ? props.copyTagsToSnapshot : true,
            dbInstanceClass: `db.${props.instanceClass}`,
            dbInstanceIdentifier: props.instanceIdentifier,
            dbSubnetGroupName: subnetGroup.ref,
            deleteAutomatedBackups: props.deleteAutomatedBackups,
            deletionProtection,
            enableCloudwatchLogsExports: this.cloudwatchLogsExports,
            enableIamDatabaseAuthentication: props.iamAuthentication,
            enablePerformanceInsights: props.enablePerformanceInsights,
            iops,
            monitoringInterval: props.monitoringInterval && props.monitoringInterval.toSeconds(),
            monitoringRoleArn: monitoringRole && monitoringRole.roleArn,
            multiAz: props.multiAz,
            optionGroupName: props.optionGroup && props.optionGroup.optionGroupName,
            performanceInsightsKmsKeyId: props.enablePerformanceInsights
                ? props.performanceInsightEncryptionKey && props.performanceInsightEncryptionKey.keyArn
                : undefined,
            performanceInsightsRetentionPeriod: props.enablePerformanceInsights
                ? (props.performanceInsightRetention || PerformanceInsightRetention.DEFAULT)
                : undefined,
            port: props.port ? props.port.toString() : undefined,
            preferredBackupWindow: props.preferredBackupWindow,
            preferredMaintenanceWindow: props.preferredMaintenanceWindow,
            processorFeatures: props.processorFeatures && renderProcessorFeatures(props.processorFeatures),
            publiclyAccessible: props.vpcPlacement && props.vpcPlacement.subnetType === ec2.SubnetType.PUBLIC,
            storageType,
            vpcSecurityGroups: securityGroups.map(s => s.securityGroupId),
            maxAllocatedStorage: props.maxAllocatedStorage,
        };
    }
    setLogRetention() {
        if (this.cloudwatchLogsExports && this.cloudwatchLogsRetention) {
            for (const log of this.cloudwatchLogsExports) {
                new lambda.LogRetention(this, `LogRetention${log}`, {
                    logGroupName: `/aws/rds/instance/${this.instanceIdentifier}/${log}`,
                    retention: this.cloudwatchLogsRetention,
                    role: this.cloudwatchLogsRetentionRole,
                });
            }
        }
    }
}
/**
 * A new source database instance (not a read replica)
 */
class DatabaseInstanceSource extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.singleUserRotationApplication = props.engine.singleUserRotationApplication;
        this.multiUserRotationApplication = props.engine.multiUserRotationApplication;
        const timezoneSupport = [DatabaseInstanceEngine.SQL_SERVER_EE, DatabaseInstanceEngine.SQL_SERVER_EX,
            DatabaseInstanceEngine.SQL_SERVER_SE, DatabaseInstanceEngine.SQL_SERVER_WEB];
        if (props.timezone && !timezoneSupport.includes(props.engine)) {
            throw new Error(`timezone property can be configured only for Microsoft SQL Server, not ${props.engine.name}`);
        }
        this.sourceCfnProps = {
            ...this.newCfnProps,
            allocatedStorage: props.allocatedStorage ? props.allocatedStorage.toString() : '100',
            allowMajorVersionUpgrade: props.allowMajorVersionUpgrade,
            dbName: props.databaseName,
            dbParameterGroupName: props.parameterGroup && props.parameterGroup.parameterGroupName,
            engine: props.engine.name,
            engineVersion: props.engineVersion,
            licenseModel: props.licenseModel,
            timezone: props.timezone,
        };
    }
    /**
     * Adds the single user rotation of the master password to this instance.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter) {
        if (!this.secret) {
            throw new Error('Cannot add single user rotation for an instance without secret.');
        }
        const id = 'RotationSingleUser';
        const existing = this.node.tryFindChild(id);
        if (existing) {
            throw new Error('A single user rotation was already added to this instance.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: this.secret,
            automaticallyAfter,
            application: this.singleUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
    /**
     * Adds the multi user rotation to this instance.
     */
    addRotationMultiUser(id, options) {
        if (!this.secret) {
            throw new Error('Cannot add multi user rotation for an instance without secret.');
        }
        return new secretsmanager.SecretRotation(this, id, {
            secret: options.secret,
            masterSecret: this.secret,
            automaticallyAfter: options.automaticallyAfter,
            application: this.multiUserRotationApplication,
            vpc: this.vpc,
            vpcSubnets: this.vpcPlacement,
            target: this,
        });
    }
}
/**
 * A database instance
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstance extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (!props.masterUserPassword) {
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            characterSetName: props.characterSetName,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            masterUsername: secret ? secret.secretValueFromJson('username').toString() : props.masterUsername,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstance = DatabaseInstance;
/**
 * A database instance restored from a snapshot.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceFromSnapshot extends DatabaseInstanceSource {
    constructor(scope, id, props) {
        super(scope, id, props);
        let secret;
        if (props.generateMasterUserPassword) {
            if (!props.masterUsername) { // We need the master username to include it in the generated secret
                throw new Error('`masterUsername` must be specified when `generateMasterUserPassword` is set to true.');
            }
            if (props.masterUserPassword) {
                throw new Error('Cannot specify `masterUserPassword` when `generateMasterUserPassword` is set to true.');
            }
            secret = new database_secret_1.DatabaseSecret(this, 'Secret', {
                username: props.masterUsername,
                encryptionKey: props.masterUserPasswordEncryptionKey,
            });
        }
        else {
            if (props.masterUsername) { // It's not possible to change the master username of a RDS instance
                throw new Error('Cannot specify `masterUsername` when `generateMasterUserPassword` is set to false.');
            }
        }
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.sourceCfnProps,
            dbSnapshotIdentifier: props.snapshotIdentifier,
            masterUserPassword: secret
                ? secret.secretValueFromJson('password').toString()
                : props.masterUserPassword && props.masterUserPassword.toString(),
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        if (secret) {
            this.secret = secret.attach(this);
        }
        this.setLogRetention();
    }
}
exports.DatabaseInstanceFromSnapshot = DatabaseInstanceFromSnapshot;
/**
 * A read replica database instance.
 *
 * @resource AWS::RDS::DBInstance
 */
class DatabaseInstanceReadReplica extends DatabaseInstanceNew {
    constructor(scope, id, props) {
        super(scope, id, props);
        const instance = new rds_generated_1.CfnDBInstance(this, 'Resource', {
            ...this.newCfnProps,
            // this must be ARN, not ID, because of https://github.com/terraform-providers/terraform-provider-aws/issues/528#issuecomment-391169012
            sourceDbInstanceIdentifier: props.sourceDatabaseInstance.instanceArn,
            kmsKeyId: props.storageEncryptionKey && props.storageEncryptionKey.keyArn,
            storageEncrypted: props.storageEncryptionKey ? true : props.storageEncrypted,
        });
        this.instanceIdentifier = instance.ref;
        this.dbInstanceEndpointAddress = instance.attrEndpointAddress;
        this.dbInstanceEndpointPort = instance.attrEndpointPort;
        // create a number token that represents the port of the instance
        const portAttribute = core_1.Token.asNumber(instance.attrEndpointPort);
        this.instanceEndpoint = new endpoint_1.Endpoint(instance.attrEndpointAddress, portAttribute);
        applyInstanceDeletionPolicy(instance, props.removalPolicy);
        this.setLogRetention();
    }
}
exports.DatabaseInstanceReadReplica = DatabaseInstanceReadReplica;
/**
 * Renders the processor features specifications
 *
 * @param features the processor features
 */
function renderProcessorFeatures(features) {
    const featuresList = Object.entries(features).map(([name, value]) => ({ name, value: value.toString() }));
    return featuresList.length === 0 ? undefined : featuresList;
}
function applyInstanceDeletionPolicy(cfnDbInstance, removalPolicy) {
    if (!removalPolicy) {
        // the default DeletionPolicy is 'Snapshot', which is fine,
        // but we should also make it 'Snapshot' for UpdateReplace policy
        cfnDbInstance.cfnOptions.updateReplacePolicy = core_1.CfnDeletionPolicy.SNAPSHOT;
    }
    else {
        // just apply whatever removal policy the customer explicitly provided
        cfnDbInstance.applyRemovalPolicy(removalPolicy);
    }
}
//# sourceMappingURL=data:application/json;base64,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