import * as ec2 from "../../aws-ec2";
import { IRole } from "../../aws-iam";
import * as kms from "../../aws-kms";
import * as s3 from "../../aws-s3";
import * as secretsmanager from "../../aws-secretsmanager";
import { Construct, Duration, RemovalPolicy, Resource } from "../../core";
import { DatabaseClusterAttributes, IDatabaseCluster } from './cluster-ref';
import { Endpoint } from './endpoint';
import { IParameterGroup } from './parameter-group';
import { BackupProps, DatabaseClusterEngine, InstanceProps, Login, RotationMultiUserOptions } from './props';
/**
 * Properties for a new database cluster
 */
export interface DatabaseClusterProps {
    /**
     * What kind of database to start
     */
    readonly engine: DatabaseClusterEngine;
    /**
     * What version of the database to start
     *
     * @default - The default for the engine is used.
     */
    readonly engineVersion?: string;
    /**
     * How many replicas/instances to create
     *
     * Has to be at least 1.
     *
     * @default 2
     */
    readonly instances?: number;
    /**
     * Settings for the individual instances that are launched
     */
    readonly instanceProps: InstanceProps;
    /**
     * Username and password for the administrative user
     */
    readonly masterUser: Login;
    /**
     * Backup settings
     *
     * @default - Backup retention period for automated backups is 1 day.
     * Backup preferred window is set to a 30-minute window selected at random from an
     * 8-hour block of time for each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_WorkingWithAutomatedBackups.html#USER_WorkingWithAutomatedBackups.BackupWindow
     */
    readonly backup?: BackupProps;
    /**
     * What port to listen on
     *
     * @default - The default for the engine is used.
     */
    readonly port?: number;
    /**
     * An optional identifier for the cluster
     *
     * @default - A name is automatically generated.
     */
    readonly clusterIdentifier?: string;
    /**
     * Base identifier for instances
     *
     * Every replica is named by appending the replica number to this string, 1-based.
     *
     * @default - clusterIdentifier is used with the word "Instance" appended.
     * If clusterIdentifier is not provided, the identifier is automatically generated.
     */
    readonly instanceIdentifierBase?: string;
    /**
     * Name of a database which is automatically created inside the cluster
     *
     * @default - Database is not created in cluster.
     */
    readonly defaultDatabaseName?: string;
    /**
     * Whether to enable storage encryption.
     *
     * @default - true if storageEncryptionKey is provided, false otherwise
     */
    readonly storageEncrypted?: boolean;
    /**
     * The KMS key for storage encryption.
     * If specified, {@link storageEncrypted} will be set to `true`.
     *
     * @default - if storageEncrypted is true then the default master key, no key otherwise
     */
    readonly storageEncryptionKey?: kms.IKey;
    /**
     * A preferred maintenance window day/time range. Should be specified as a range ddd:hh24:mi-ddd:hh24:mi (24H Clock UTC).
     *
     * Example: 'Sun:23:45-Mon:00:15'
     *
     * @default - 30-minute window selected at random from an 8-hour block of time for
     * each AWS Region, occurring on a random day of the week.
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_UpgradeDBInstance.Maintenance.html#Concepts.DBMaintenance
     */
    readonly preferredMaintenanceWindow?: string;
    /**
     * Additional parameters to pass to the database engine
     *
     * @default - No parameter group.
     */
    readonly parameterGroup?: IParameterGroup;
    /**
     * The removal policy to apply when the cluster and its instances are removed
     * from the stack or replaced during an update.
     *
     * @default - RemovalPolicy.SNAPSHOT (remove the cluster and instances, but retain a snapshot of the data)
     */
    readonly removalPolicy?: RemovalPolicy;
    /**
     * The interval, in seconds, between points when Amazon RDS collects enhanced
     * monitoring metrics for the DB instances.
     *
     * @default no enhanced monitoring
     */
    readonly monitoringInterval?: Duration;
    /**
     * Role that will be used to manage DB instances monitoring.
     *
     * @default - A role is automatically created for you
     */
    readonly monitoringRole?: IRole;
    /**
     * Role that will be associated with this DB cluster to enable S3 import.
     * This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ImportBuckets` is used.
     *
     * For MySQL:
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.LoadFromS3.html
     *
     * For PostgreSQL:
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
     *
     * @default - New role is created if `s3ImportBuckets` is set, no role is defined otherwise
     */
    readonly s3ImportRole?: IRole;
    /**
     * S3 buckets that you want to load data from. This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ImportRole` is used.
     *
     * For MySQL:
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.LoadFromS3.html
     *
     * For PostgreSQL:
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraPostgreSQL.Migrating.html
     *
     * @default - None
     */
    readonly s3ImportBuckets?: s3.IBucket[];
    /**
     * Role that will be associated with this DB cluster to enable S3 export.
     * This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ExportBuckets` is used.
     *
     * For MySQL:
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.SaveIntoS3.html
     *
     * For PostgreSQL:
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
     *
     * @default - New role is created if `s3ExportBuckets` is set, no role is defined otherwise
     */
    readonly s3ExportRole?: IRole;
    /**
     * S3 buckets that you want to load data into. This feature is only supported by the Aurora database engine.
     *
     * This property must not be used if `s3ExportRole` is used.
     *
     * For MySQL:
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/AuroraMySQL.Integrating.SaveIntoS3.html
     *
     * For PostgreSQL:
     * @see https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/postgresql-s3-export.html
     *
     * @default - None
     */
    readonly s3ExportBuckets?: s3.IBucket[];
}
/**
 * A new or imported clustered database.
 */
declare abstract class DatabaseClusterBase extends Resource implements IDatabaseCluster {
    /**
     * Identifier of the cluster
     */
    abstract readonly clusterIdentifier: string;
    /**
     * Identifiers of the replicas
     */
    abstract readonly instanceIdentifiers: string[];
    /**
     * The endpoint to use for read/write operations
     */
    abstract readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     */
    abstract readonly clusterReadEndpoint: Endpoint;
    /**
     * Endpoints which address each individual replica.
     */
    abstract readonly instanceEndpoints: Endpoint[];
    /**
     * Access to the network connections
     */
    abstract readonly connections: ec2.Connections;
    /**
     * Security group identifier of this database
     */
    abstract readonly securityGroupId: string;
    /**
     * Renders the secret attachment target specifications.
     */
    asSecretAttachmentTarget(): secretsmanager.SecretAttachmentTargetProps;
}
/**
 * Create a clustered database with a given number of instances.
 *
 * @resource AWS::RDS::DBCluster
 */
export declare class DatabaseCluster extends DatabaseClusterBase {
    /**
     * Import an existing DatabaseCluster from properties
     */
    static fromDatabaseClusterAttributes(scope: Construct, id: string, attrs: DatabaseClusterAttributes): IDatabaseCluster;
    /**
     * Identifier of the cluster
     */
    readonly clusterIdentifier: string;
    /**
     * Identifiers of the replicas
     */
    readonly instanceIdentifiers: string[];
    /**
     * The endpoint to use for read/write operations
     */
    readonly clusterEndpoint: Endpoint;
    /**
     * Endpoint to use for load-balanced read-only operations.
     */
    readonly clusterReadEndpoint: Endpoint;
    /**
     * Endpoints which address each individual replica.
     */
    readonly instanceEndpoints: Endpoint[];
    /**
     * Access to the network connections
     */
    readonly connections: ec2.Connections;
    /**
     * Security group identifier of this database
     */
    readonly securityGroupId: string;
    /**
     * The secret attached to this cluster
     */
    readonly secret?: secretsmanager.ISecret;
    private readonly singleUserRotationApplication;
    private readonly multiUserRotationApplication;
    /**
     * The VPC where the DB subnet group is created.
     */
    private readonly vpc;
    /**
     * The subnets used by the DB subnet group.
     *
     * @default - the Vpc default strategy if not specified.
     */
    private readonly vpcSubnets?;
    constructor(scope: Construct, id: string, props: DatabaseClusterProps);
    /**
     * Adds the single user rotation of the master password to this cluster.
     *
     * @param [automaticallyAfter=Duration.days(30)] Specifies the number of days after the previous rotation
     * before Secrets Manager triggers the next automatic rotation.
     */
    addRotationSingleUser(automaticallyAfter?: Duration): secretsmanager.SecretRotation;
    /**
     * Adds the multi user rotation to this cluster.
     */
    addRotationMultiUser(id: string, options: RotationMultiUserOptions): secretsmanager.SecretRotation;
}
export {};
