"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const path = require("path");
const code_1 = require("./code");
const runtime_1 = require("./runtime");
const singleton_lambda_1 = require("./singleton-lambda");
/**
 * Creates a custom resource to control the retention policy of a CloudWatch Logs
 * log group. The log group is created if it doesn't already exist. The policy
 * is removed when `retentionDays` is `undefined` or equal to `Infinity`.
 */
class LogRetention extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        // Custom resource provider
        const provider = new singleton_lambda_1.SingletonFunction(this, 'Provider', {
            code: code_1.Code.fromAsset(path.join(__dirname, 'log-retention-provider')),
            runtime: runtime_1.Runtime.NODEJS_10_X,
            handler: 'index.handler',
            uuid: 'aae0aa3c-5b4d-4f87-b02d-85b201efdd8a',
            lambdaPurpose: 'LogRetention',
            role: props.role,
        });
        // Duplicate statements will be deduplicated by `PolicyDocument`
        provider.addToRolePolicy(new iam.PolicyStatement({
            actions: ['logs:PutRetentionPolicy', 'logs:DeleteRetentionPolicy'],
            // We need '*' here because we will also put a retention policy on
            // the log group of the provider function. Referencing it's name
            // creates a CF circular dependency.
            resources: ['*'],
        }));
        // Need to use a CfnResource here to prevent lerna dependency cycles
        // @aws-cdk/aws-cloudformation -> @aws-cdk/aws-lambda -> @aws-cdk/aws-cloudformation
        const resource = new cdk.CfnResource(this, 'Resource', {
            type: 'Custom::LogRetention',
            properties: {
                ServiceToken: provider.functionArn,
                LogGroupName: props.logGroupName,
                RetentionInDays: props.retention === logs.RetentionDays.INFINITE ? undefined : props.retention,
            },
        });
        const logGroupName = resource.getAtt('LogGroupName').toString();
        // Append ':*' at the end of the ARN to match with how CloudFormation does this for LogGroup ARNs
        // See https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-resource-logs-loggroup.html#aws-resource-logs-loggroup-return-values
        this.logGroupArn = cdk.Stack.of(this).formatArn({
            service: 'logs',
            resource: 'log-group',
            resourceName: `${logGroupName}:*`,
            sep: ':',
        });
    }
}
exports.LogRetention = LogRetention;
//# sourceMappingURL=data:application/json;base64,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