"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const function_base_1 = require("./function-base");
const lambda_version_1 = require("./lambda-version");
const lambda_generated_1 = require("./lambda.generated");
/**
 * A new alias to a particular version of a Lambda function.
 */
class Alias extends function_base_1.QualifiedFunctionBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.aliasName,
        });
        this.canCreatePermissions = true;
        this.lambda = props.version.lambda;
        this.aliasName = this.physicalName;
        this.version = props.version;
        const alias = new lambda_generated_1.CfnAlias(this, 'Resource', {
            name: this.aliasName,
            description: props.description,
            functionName: this.version.lambda.functionName,
            functionVersion: props.version.version,
            routingConfig: this.determineRoutingConfig(props),
            provisionedConcurrencyConfig: this.determineProvisionedConcurrency(props),
        });
        this.functionArn = this.getResourceArnAttribute(alias.ref, {
            service: 'lambda',
            resource: 'function',
            resourceName: `${this.lambda.functionName}:${this.physicalName}`,
            sep: ':',
        });
        this.qualifier = lambda_version_1.extractQualifierFromArn(alias.ref);
        if (props.onFailure || props.onSuccess || props.maxEventAge || props.retryAttempts !== undefined) {
            this.configureAsyncInvoke({
                onFailure: props.onFailure,
                onSuccess: props.onSuccess,
                maxEventAge: props.maxEventAge,
                retryAttempts: props.retryAttempts,
            });
        }
        // ARN parsing splits on `:`, so we can only get the function's name from the ARN as resourceName...
        // And we're parsing it out (instead of using the underlying function directly) in order to have use of it incur
        // an implicit dependency on the resource.
        this.functionName = `${this.stack.parseArn(this.functionArn, ':').resourceName}:${this.aliasName}`;
    }
    static fromAliasAttributes(scope, id, attrs) {
        class Imported extends function_base_1.QualifiedFunctionBase {
            constructor() {
                super(...arguments);
                this.aliasName = attrs.aliasName;
                this.version = attrs.aliasVersion;
                this.lambda = attrs.aliasVersion.lambda;
                this.functionArn = `${attrs.aliasVersion.lambda.functionArn}:${attrs.aliasName}`;
                this.functionName = `${attrs.aliasVersion.lambda.functionName}:${attrs.aliasName}`;
                this.grantPrincipal = attrs.aliasVersion.grantPrincipal;
                this.role = attrs.aliasVersion.role;
                this.canCreatePermissions = false;
                this.qualifier = attrs.aliasName;
            }
        }
        return new Imported(scope, id);
    }
    get grantPrincipal() {
        return this.version.grantPrincipal;
    }
    get role() {
        return this.version.role;
    }
    metric(metricName, props = {}) {
        // Metrics on Aliases need the "bare" function name, and the alias' ARN, this differes from the base behavior.
        return super.metric(metricName, {
            dimensions: {
                FunctionName: this.lambda.functionName,
                // construct the name from the underlying lambda so that alarms on an alias
                // don't cause a circular dependency with CodeDeploy
                // see: https://github.com/aws/aws-cdk/issues/2231
                Resource: `${this.lambda.functionName}:${this.aliasName}`,
            },
            ...props,
        });
    }
    /**
     * Calculate the routingConfig parameter from the input props
     */
    determineRoutingConfig(props) {
        if (!props.additionalVersions || props.additionalVersions.length === 0) {
            return undefined;
        }
        this.validateAdditionalWeights(props.additionalVersions);
        return {
            additionalVersionWeights: props.additionalVersions.map(vw => {
                return {
                    functionVersion: vw.version.version,
                    functionWeight: vw.weight,
                };
            }),
        };
    }
    /**
     * Validate that the additional version weights make sense
     *
     * We validate that they are positive and add up to something <= 1.
     */
    validateAdditionalWeights(weights) {
        const total = weights.map(w => {
            if (w.weight < 0 || w.weight > 1) {
                throw new Error(`Additional version weight must be between 0 and 1, got: ${w.weight}`);
            }
            return w.weight;
        }).reduce((a, x) => a + x);
        if (total > 1) {
            throw new Error(`Sum of additional version weights must not exceed 1, got: ${total}`);
        }
    }
    /**
     * Validate that the provisionedConcurrentExecutions makes sense
     *
     * Member must have value greater than or equal to 1
     */
    determineProvisionedConcurrency(props) {
        if (!props.provisionedConcurrentExecutions) {
            return undefined;
        }
        if (props.provisionedConcurrentExecutions <= 0) {
            throw new Error('provisionedConcurrentExecutions must have value greater than or equal to 1');
        }
        return { provisionedConcurrentExecutions: props.provisionedConcurrentExecutions };
    }
}
exports.Alias = Alias;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYWxpYXMuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJhbGlhcy50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOztBQUdBLG1EQUFtRTtBQUNuRSxxREFBcUU7QUFDckUseURBQThDO0FBbUU5Qzs7R0FFRztBQUNILE1BQWEsS0FBTSxTQUFRLHFDQUFxQjtJQXVDNUMsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxLQUFpQjtRQUN2RCxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRTtZQUNiLFlBQVksRUFBRSxLQUFLLENBQUMsU0FBUztTQUNoQyxDQUFDLENBQUM7UUFKWSx5QkFBb0IsR0FBWSxJQUFJLENBQUM7UUFLcEQsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDLE1BQU0sQ0FBQztRQUNuQyxJQUFJLENBQUMsU0FBUyxHQUFHLElBQUksQ0FBQyxZQUFZLENBQUM7UUFDbkMsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLE1BQU0sS0FBSyxHQUFHLElBQUksMkJBQVEsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3pDLElBQUksRUFBRSxJQUFJLENBQUMsU0FBUztZQUNwQixXQUFXLEVBQUUsS0FBSyxDQUFDLFdBQVc7WUFDOUIsWUFBWSxFQUFFLElBQUksQ0FBQyxPQUFPLENBQUMsTUFBTSxDQUFDLFlBQVk7WUFDOUMsZUFBZSxFQUFFLEtBQUssQ0FBQyxPQUFPLENBQUMsT0FBTztZQUN0QyxhQUFhLEVBQUUsSUFBSSxDQUFDLHNCQUFzQixDQUFDLEtBQUssQ0FBQztZQUNqRCw0QkFBNEIsRUFBRSxJQUFJLENBQUMsK0JBQStCLENBQUMsS0FBSyxDQUFDO1NBQzVFLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxXQUFXLEdBQUcsSUFBSSxDQUFDLHVCQUF1QixDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUU7WUFDdkQsT0FBTyxFQUFFLFFBQVE7WUFDakIsUUFBUSxFQUFFLFVBQVU7WUFDcEIsWUFBWSxFQUFFLEdBQUcsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZLElBQUksSUFBSSxDQUFDLFlBQVksRUFBRTtZQUNoRSxHQUFHLEVBQUUsR0FBRztTQUNYLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxTQUFTLEdBQUcsd0NBQXVCLENBQUMsS0FBSyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3BELElBQUksS0FBSyxDQUFDLFNBQVMsSUFBSSxLQUFLLENBQUMsU0FBUyxJQUFJLEtBQUssQ0FBQyxXQUFXLElBQUksS0FBSyxDQUFDLGFBQWEsS0FBSyxTQUFTLEVBQUU7WUFDOUYsSUFBSSxDQUFDLG9CQUFvQixDQUFDO2dCQUN0QixTQUFTLEVBQUUsS0FBSyxDQUFDLFNBQVM7Z0JBQzFCLFNBQVMsRUFBRSxLQUFLLENBQUMsU0FBUztnQkFDMUIsV0FBVyxFQUFFLEtBQUssQ0FBQyxXQUFXO2dCQUM5QixhQUFhLEVBQUUsS0FBSyxDQUFDLGFBQWE7YUFDckMsQ0FBQyxDQUFDO1NBQ047UUFDRCxvR0FBb0c7UUFDcEcsZ0hBQWdIO1FBQ2hILDBDQUEwQztRQUMxQyxJQUFJLENBQUMsWUFBWSxHQUFHLEdBQUcsSUFBSSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFdBQVcsRUFBRSxHQUFHLENBQUMsQ0FBQyxZQUFhLElBQUksSUFBSSxDQUFDLFNBQVMsRUFBRSxDQUFDO0lBQ3hHLENBQUM7SUF4RU0sTUFBTSxDQUFDLG1CQUFtQixDQUFDLEtBQWdCLEVBQUUsRUFBVSxFQUFFLEtBQXNCO1FBQ2xGLE1BQU0sUUFBUyxTQUFRLHFDQUFxQjtZQUE1Qzs7Z0JBQ29CLGNBQVMsR0FBRyxLQUFLLENBQUMsU0FBUyxDQUFDO2dCQUM1QixZQUFPLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQztnQkFDN0IsV0FBTSxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDO2dCQUNuQyxnQkFBVyxHQUFHLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxNQUFNLENBQUMsV0FBVyxJQUFJLEtBQUssQ0FBQyxTQUFTLEVBQUUsQ0FBQztnQkFDNUUsaUJBQVksR0FBRyxHQUFHLEtBQUssQ0FBQyxZQUFZLENBQUMsTUFBTSxDQUFDLFlBQVksSUFBSSxLQUFLLENBQUMsU0FBUyxFQUFFLENBQUM7Z0JBQzlFLG1CQUFjLEdBQUcsS0FBSyxDQUFDLFlBQVksQ0FBQyxjQUFjLENBQUM7Z0JBQ25ELFNBQUksR0FBRyxLQUFLLENBQUMsWUFBWSxDQUFDLElBQUksQ0FBQztnQkFDNUIseUJBQW9CLEdBQUcsS0FBSyxDQUFDO2dCQUM3QixjQUFTLEdBQUcsS0FBSyxDQUFDLFNBQVMsQ0FBQztZQUNuRCxDQUFDO1NBQUE7UUFDRCxPQUFPLElBQUksUUFBUSxDQUFDLEtBQUssRUFBRSxFQUFFLENBQUMsQ0FBQztJQUNuQyxDQUFDO0lBNERELElBQVcsY0FBYztRQUNyQixPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDO0lBQ3ZDLENBQUM7SUFDRCxJQUFXLElBQUk7UUFDWCxPQUFPLElBQUksQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDO0lBQzdCLENBQUM7SUFDTSxNQUFNLENBQUMsVUFBa0IsRUFBRSxRQUFrQyxFQUFFO1FBQ2xFLDhHQUE4RztRQUM5RyxPQUFPLEtBQUssQ0FBQyxNQUFNLENBQUMsVUFBVSxFQUFFO1lBQzVCLFVBQVUsRUFBRTtnQkFDUixZQUFZLEVBQUUsSUFBSSxDQUFDLE1BQU0sQ0FBQyxZQUFZO2dCQUN0QywyRUFBMkU7Z0JBQzNFLG9EQUFvRDtnQkFDcEQsa0RBQWtEO2dCQUNsRCxRQUFRLEVBQUUsR0FBRyxJQUFJLENBQUMsTUFBTSxDQUFDLFlBQVksSUFBSSxJQUFJLENBQUMsU0FBUyxFQUFFO2FBQzVEO1lBQ0QsR0FBRyxLQUFLO1NBQ1gsQ0FBQyxDQUFDO0lBQ1AsQ0FBQztJQUNEOztPQUVHO0lBQ0ssc0JBQXNCLENBQUMsS0FBaUI7UUFDNUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxrQkFBa0IsSUFBSSxLQUFLLENBQUMsa0JBQWtCLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNwRSxPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELElBQUksQ0FBQyx5QkFBeUIsQ0FBQyxLQUFLLENBQUMsa0JBQWtCLENBQUMsQ0FBQztRQUN6RCxPQUFPO1lBQ0gsd0JBQXdCLEVBQUUsS0FBSyxDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsRUFBRTtnQkFDeEQsT0FBTztvQkFDSCxlQUFlLEVBQUUsRUFBRSxDQUFDLE9BQU8sQ0FBQyxPQUFPO29CQUNuQyxjQUFjLEVBQUUsRUFBRSxDQUFDLE1BQU07aUJBQzVCLENBQUM7WUFDTixDQUFDLENBQUM7U0FDTCxDQUFDO0lBQ04sQ0FBQztJQUNEOzs7O09BSUc7SUFDSyx5QkFBeUIsQ0FBQyxPQUF3QjtRQUN0RCxNQUFNLEtBQUssR0FBRyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFO1lBQzFCLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQzlCLE1BQU0sSUFBSSxLQUFLLENBQUMsMkRBQTJELENBQUMsQ0FBQyxNQUFNLEVBQUUsQ0FBQyxDQUFDO2FBQzFGO1lBQ0QsT0FBTyxDQUFDLENBQUMsTUFBTSxDQUFDO1FBQ3BCLENBQUMsQ0FBQyxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDLEVBQUUsRUFBRSxDQUFDLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQztRQUMzQixJQUFJLEtBQUssR0FBRyxDQUFDLEVBQUU7WUFDWCxNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxLQUFLLEVBQUUsQ0FBQyxDQUFDO1NBQ3pGO0lBQ0wsQ0FBQztJQUNEOzs7O09BSUc7SUFDSywrQkFBK0IsQ0FBQyxLQUFpQjtRQUNyRCxJQUFJLENBQUMsS0FBSyxDQUFDLCtCQUErQixFQUFFO1lBQ3hDLE9BQU8sU0FBUyxDQUFDO1NBQ3BCO1FBQ0QsSUFBSSxLQUFLLENBQUMsK0JBQStCLElBQUksQ0FBQyxFQUFFO1lBQzVDLE1BQU0sSUFBSSxLQUFLLENBQUMsNEVBQTRFLENBQUMsQ0FBQztTQUNqRztRQUNELE9BQU8sRUFBRSwrQkFBK0IsRUFBRSxLQUFLLENBQUMsK0JBQStCLEVBQUUsQ0FBQztJQUN0RixDQUFDO0NBQ0o7QUE1SUQsc0JBNElDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2xvdWR3YXRjaCBmcm9tIFwiLi4vLi4vYXdzLWNsb3Vkd2F0Y2hcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1jbG91ZHdhdGNoJ1xuaW1wb3J0IHsgQ29uc3RydWN0IH0gZnJvbSBcIi4uLy4uL2NvcmVcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2NvcmUnXG5pbXBvcnQgeyBFdmVudEludm9rZUNvbmZpZ09wdGlvbnMgfSBmcm9tICcuL2V2ZW50LWludm9rZS1jb25maWcnO1xuaW1wb3J0IHsgSUZ1bmN0aW9uLCBRdWFsaWZpZWRGdW5jdGlvbkJhc2UgfSBmcm9tICcuL2Z1bmN0aW9uLWJhc2UnO1xuaW1wb3J0IHsgZXh0cmFjdFF1YWxpZmllckZyb21Bcm4sIElWZXJzaW9uIH0gZnJvbSAnLi9sYW1iZGEtdmVyc2lvbic7XG5pbXBvcnQgeyBDZm5BbGlhcyB9IGZyb20gJy4vbGFtYmRhLmdlbmVyYXRlZCc7XG5leHBvcnQgaW50ZXJmYWNlIElBbGlhcyBleHRlbmRzIElGdW5jdGlvbiB7XG4gICAgLyoqXG4gICAgICogTmFtZSBvZiB0aGlzIGFsaWFzLlxuICAgICAqXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFsaWFzTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIFRoZSB1bmRlcmx5aW5nIExhbWJkYSBmdW5jdGlvbiB2ZXJzaW9uLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHZlcnNpb246IElWZXJzaW9uO1xufVxuLyoqXG4gKiBPcHRpb25zIGZvciBgbGFtYmRhLkFsaWFzYC5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBBbGlhc09wdGlvbnMgZXh0ZW5kcyBFdmVudEludm9rZUNvbmZpZ09wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIERlc2NyaXB0aW9uIGZvciB0aGUgYWxpYXNcbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IE5vIGRlc2NyaXB0aW9uXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVzY3JpcHRpb24/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogQWRkaXRpb25hbCB2ZXJzaW9ucyB3aXRoIGluZGl2aWR1YWwgd2VpZ2h0cyB0aGlzIGFsaWFzIHBvaW50cyB0b1xuICAgICAqXG4gICAgICogSW5kaXZpZHVhbCBhZGRpdGlvbmFsIHZlcnNpb24gd2VpZ2h0cyBzcGVjaWZpZWQgaGVyZSBzaG91bGQgYWRkIHVwIHRvXG4gICAgICogKGxlc3MgdGhhbikgb25lLiBBbGwgcmVtYWluaW5nIHdlaWdodCBpcyByb3V0ZWQgdG8gdGhlIGRlZmF1bHRcbiAgICAgKiB2ZXJzaW9uLlxuICAgICAqXG4gICAgICogRm9yIGV4YW1wbGUsIHRoZSBjb25maWcgaXNcbiAgICAgKlxuICAgICAqICAgIHZlcnNpb246IFwiMVwiXG4gICAgICogICAgYWRkaXRpb25hbFZlcnNpb25zOiBbeyB2ZXJzaW9uOiBcIjJcIiwgd2VpZ2h0OiAwLjA1IH1dXG4gICAgICpcbiAgICAgKiBUaGVuIDUlIG9mIHRyYWZmaWMgd2lsbCBiZSByb3V0ZWQgdG8gZnVuY3Rpb24gdmVyc2lvbiAyLCB3aGlsZVxuICAgICAqIHRoZSByZW1haW5pbmcgOTUlIG9mIHRyYWZmaWMgd2lsbCBiZSByb3V0ZWQgdG8gZnVuY3Rpb24gdmVyc2lvbiAxLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgTm8gYWRkaXRpb25hbCB2ZXJzaW9uc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGFkZGl0aW9uYWxWZXJzaW9ucz86IFZlcnNpb25XZWlnaHRbXTtcbiAgICAvKipcbiAgICAgKiBTcGVjaWZpZXMgYSBwcm92aXNpb25lZCBjb25jdXJyZW5jeSBjb25maWd1cmF0aW9uIGZvciBhIGZ1bmN0aW9uJ3MgYWxpYXMuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCBObyBwcm92aXNpb25lZCBjb25jdXJyZW5jeVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHByb3Zpc2lvbmVkQ29uY3VycmVudEV4ZWN1dGlvbnM/OiBudW1iZXI7XG59XG4vKipcbiAqIFByb3BlcnRpZXMgZm9yIGEgbmV3IExhbWJkYSBhbGlhc1xuICovXG5leHBvcnQgaW50ZXJmYWNlIEFsaWFzUHJvcHMgZXh0ZW5kcyBBbGlhc09wdGlvbnMge1xuICAgIC8qKlxuICAgICAqIE5hbWUgb2YgdGhpcyBhbGlhc1xuICAgICAqL1xuICAgIHJlYWRvbmx5IGFsaWFzTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEZ1bmN0aW9uIHZlcnNpb24gdGhpcyBhbGlhcyByZWZlcnMgdG9cbiAgICAgKlxuICAgICAqIFVzZSBsYW1iZGEuYWRkVmVyc2lvbigpIHRvIG9idGFpbiBhIG5ldyBsYW1iZGEgdmVyc2lvbiB0byByZWZlciB0by5cbiAgICAgKi9cbiAgICByZWFkb25seSB2ZXJzaW9uOiBJVmVyc2lvbjtcbn1cbmV4cG9ydCBpbnRlcmZhY2UgQWxpYXNBdHRyaWJ1dGVzIHtcbiAgICByZWFkb25seSBhbGlhc05hbWU6IHN0cmluZztcbiAgICByZWFkb25seSBhbGlhc1ZlcnNpb246IElWZXJzaW9uO1xufVxuLyoqXG4gKiBBIG5ldyBhbGlhcyB0byBhIHBhcnRpY3VsYXIgdmVyc2lvbiBvZiBhIExhbWJkYSBmdW5jdGlvbi5cbiAqL1xuZXhwb3J0IGNsYXNzIEFsaWFzIGV4dGVuZHMgUXVhbGlmaWVkRnVuY3Rpb25CYXNlIGltcGxlbWVudHMgSUFsaWFzIHtcbiAgICBwdWJsaWMgc3RhdGljIGZyb21BbGlhc0F0dHJpYnV0ZXMoc2NvcGU6IENvbnN0cnVjdCwgaWQ6IHN0cmluZywgYXR0cnM6IEFsaWFzQXR0cmlidXRlcyk6IElBbGlhcyB7XG4gICAgICAgIGNsYXNzIEltcG9ydGVkIGV4dGVuZHMgUXVhbGlmaWVkRnVuY3Rpb25CYXNlIGltcGxlbWVudHMgSUFsaWFzIHtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBhbGlhc05hbWUgPSBhdHRycy5hbGlhc05hbWU7XG4gICAgICAgICAgICBwdWJsaWMgcmVhZG9ubHkgdmVyc2lvbiA9IGF0dHJzLmFsaWFzVmVyc2lvbjtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBsYW1iZGEgPSBhdHRycy5hbGlhc1ZlcnNpb24ubGFtYmRhO1xuICAgICAgICAgICAgcHVibGljIHJlYWRvbmx5IGZ1bmN0aW9uQXJuID0gYCR7YXR0cnMuYWxpYXNWZXJzaW9uLmxhbWJkYS5mdW5jdGlvbkFybn06JHthdHRycy5hbGlhc05hbWV9YDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBmdW5jdGlvbk5hbWUgPSBgJHthdHRycy5hbGlhc1ZlcnNpb24ubGFtYmRhLmZ1bmN0aW9uTmFtZX06JHthdHRycy5hbGlhc05hbWV9YDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSBncmFudFByaW5jaXBhbCA9IGF0dHJzLmFsaWFzVmVyc2lvbi5ncmFudFByaW5jaXBhbDtcbiAgICAgICAgICAgIHB1YmxpYyByZWFkb25seSByb2xlID0gYXR0cnMuYWxpYXNWZXJzaW9uLnJvbGU7XG4gICAgICAgICAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgY2FuQ3JlYXRlUGVybWlzc2lvbnMgPSBmYWxzZTtcbiAgICAgICAgICAgIHByb3RlY3RlZCByZWFkb25seSBxdWFsaWZpZXIgPSBhdHRycy5hbGlhc05hbWU7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIG5ldyBJbXBvcnRlZChzY29wZSwgaWQpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBOYW1lIG9mIHRoaXMgYWxpYXMuXG4gICAgICpcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGFsaWFzTmFtZTogc3RyaW5nO1xuICAgIC8qKlxuICAgICAqIEFSTiBvZiB0aGlzIGFsaWFzXG4gICAgICpcbiAgICAgKiBVc2VkIHRvIGJlIGFibGUgdG8gdXNlIEFsaWFzIGluIHBsYWNlIG9mIGEgcmVndWxhciBMYW1iZGEuIExhbWJkYSBhY2NlcHRzXG4gICAgICogQVJOcyBldmVyeXdoZXJlIGl0IGFjY2VwdHMgZnVuY3Rpb24gbmFtZXMuXG4gICAgICovXG4gICAgcHVibGljIHJlYWRvbmx5IGZ1bmN0aW9uTmFtZTogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSBsYW1iZGE6IElGdW5jdGlvbjtcbiAgICBwdWJsaWMgcmVhZG9ubHkgdmVyc2lvbjogSVZlcnNpb247XG4gICAgLyoqXG4gICAgICogQVJOIG9mIHRoaXMgYWxpYXNcbiAgICAgKlxuICAgICAqIFVzZWQgdG8gYmUgYWJsZSB0byB1c2UgQWxpYXMgaW4gcGxhY2Ugb2YgYSByZWd1bGFyIExhbWJkYS4gTGFtYmRhIGFjY2VwdHNcbiAgICAgKiBBUk5zIGV2ZXJ5d2hlcmUgaXQgYWNjZXB0cyBmdW5jdGlvbiBuYW1lcy5cbiAgICAgKi9cbiAgICBwdWJsaWMgcmVhZG9ubHkgZnVuY3Rpb25Bcm46IHN0cmluZztcbiAgICBwcm90ZWN0ZWQgcmVhZG9ubHkgcXVhbGlmaWVyOiBzdHJpbmc7XG4gICAgcHJvdGVjdGVkIHJlYWRvbmx5IGNhbkNyZWF0ZVBlcm1pc3Npb25zOiBib29sZWFuID0gdHJ1ZTtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQWxpYXNQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHtcbiAgICAgICAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuYWxpYXNOYW1lLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5sYW1iZGEgPSBwcm9wcy52ZXJzaW9uLmxhbWJkYTtcbiAgICAgICAgdGhpcy5hbGlhc05hbWUgPSB0aGlzLnBoeXNpY2FsTmFtZTtcbiAgICAgICAgdGhpcy52ZXJzaW9uID0gcHJvcHMudmVyc2lvbjtcbiAgICAgICAgY29uc3QgYWxpYXMgPSBuZXcgQ2ZuQWxpYXModGhpcywgJ1Jlc291cmNlJywge1xuICAgICAgICAgICAgbmFtZTogdGhpcy5hbGlhc05hbWUsXG4gICAgICAgICAgICBkZXNjcmlwdGlvbjogcHJvcHMuZGVzY3JpcHRpb24sXG4gICAgICAgICAgICBmdW5jdGlvbk5hbWU6IHRoaXMudmVyc2lvbi5sYW1iZGEuZnVuY3Rpb25OYW1lLFxuICAgICAgICAgICAgZnVuY3Rpb25WZXJzaW9uOiBwcm9wcy52ZXJzaW9uLnZlcnNpb24sXG4gICAgICAgICAgICByb3V0aW5nQ29uZmlnOiB0aGlzLmRldGVybWluZVJvdXRpbmdDb25maWcocHJvcHMpLFxuICAgICAgICAgICAgcHJvdmlzaW9uZWRDb25jdXJyZW5jeUNvbmZpZzogdGhpcy5kZXRlcm1pbmVQcm92aXNpb25lZENvbmN1cnJlbmN5KHByb3BzKSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuZnVuY3Rpb25Bcm4gPSB0aGlzLmdldFJlc291cmNlQXJuQXR0cmlidXRlKGFsaWFzLnJlZiwge1xuICAgICAgICAgICAgc2VydmljZTogJ2xhbWJkYScsXG4gICAgICAgICAgICByZXNvdXJjZTogJ2Z1bmN0aW9uJyxcbiAgICAgICAgICAgIHJlc291cmNlTmFtZTogYCR7dGhpcy5sYW1iZGEuZnVuY3Rpb25OYW1lfToke3RoaXMucGh5c2ljYWxOYW1lfWAsXG4gICAgICAgICAgICBzZXA6ICc6JyxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMucXVhbGlmaWVyID0gZXh0cmFjdFF1YWxpZmllckZyb21Bcm4oYWxpYXMucmVmKTtcbiAgICAgICAgaWYgKHByb3BzLm9uRmFpbHVyZSB8fCBwcm9wcy5vblN1Y2Nlc3MgfHwgcHJvcHMubWF4RXZlbnRBZ2UgfHwgcHJvcHMucmV0cnlBdHRlbXB0cyAhPT0gdW5kZWZpbmVkKSB7XG4gICAgICAgICAgICB0aGlzLmNvbmZpZ3VyZUFzeW5jSW52b2tlKHtcbiAgICAgICAgICAgICAgICBvbkZhaWx1cmU6IHByb3BzLm9uRmFpbHVyZSxcbiAgICAgICAgICAgICAgICBvblN1Y2Nlc3M6IHByb3BzLm9uU3VjY2VzcyxcbiAgICAgICAgICAgICAgICBtYXhFdmVudEFnZTogcHJvcHMubWF4RXZlbnRBZ2UsXG4gICAgICAgICAgICAgICAgcmV0cnlBdHRlbXB0czogcHJvcHMucmV0cnlBdHRlbXB0cyxcbiAgICAgICAgICAgIH0pO1xuICAgICAgICB9XG4gICAgICAgIC8vIEFSTiBwYXJzaW5nIHNwbGl0cyBvbiBgOmAsIHNvIHdlIGNhbiBvbmx5IGdldCB0aGUgZnVuY3Rpb24ncyBuYW1lIGZyb20gdGhlIEFSTiBhcyByZXNvdXJjZU5hbWUuLi5cbiAgICAgICAgLy8gQW5kIHdlJ3JlIHBhcnNpbmcgaXQgb3V0IChpbnN0ZWFkIG9mIHVzaW5nIHRoZSB1bmRlcmx5aW5nIGZ1bmN0aW9uIGRpcmVjdGx5KSBpbiBvcmRlciB0byBoYXZlIHVzZSBvZiBpdCBpbmN1clxuICAgICAgICAvLyBhbiBpbXBsaWNpdCBkZXBlbmRlbmN5IG9uIHRoZSByZXNvdXJjZS5cbiAgICAgICAgdGhpcy5mdW5jdGlvbk5hbWUgPSBgJHt0aGlzLnN0YWNrLnBhcnNlQXJuKHRoaXMuZnVuY3Rpb25Bcm4sICc6JykucmVzb3VyY2VOYW1lIX06JHt0aGlzLmFsaWFzTmFtZX1gO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IGdyYW50UHJpbmNpcGFsKCkge1xuICAgICAgICByZXR1cm4gdGhpcy52ZXJzaW9uLmdyYW50UHJpbmNpcGFsO1xuICAgIH1cbiAgICBwdWJsaWMgZ2V0IHJvbGUoKSB7XG4gICAgICAgIHJldHVybiB0aGlzLnZlcnNpb24ucm9sZTtcbiAgICB9XG4gICAgcHVibGljIG1ldHJpYyhtZXRyaWNOYW1lOiBzdHJpbmcsIHByb3BzOiBjbG91ZHdhdGNoLk1ldHJpY09wdGlvbnMgPSB7fSk6IGNsb3Vkd2F0Y2guTWV0cmljIHtcbiAgICAgICAgLy8gTWV0cmljcyBvbiBBbGlhc2VzIG5lZWQgdGhlIFwiYmFyZVwiIGZ1bmN0aW9uIG5hbWUsIGFuZCB0aGUgYWxpYXMnIEFSTiwgdGhpcyBkaWZmZXJlcyBmcm9tIHRoZSBiYXNlIGJlaGF2aW9yLlxuICAgICAgICByZXR1cm4gc3VwZXIubWV0cmljKG1ldHJpY05hbWUsIHtcbiAgICAgICAgICAgIGRpbWVuc2lvbnM6IHtcbiAgICAgICAgICAgICAgICBGdW5jdGlvbk5hbWU6IHRoaXMubGFtYmRhLmZ1bmN0aW9uTmFtZSxcbiAgICAgICAgICAgICAgICAvLyBjb25zdHJ1Y3QgdGhlIG5hbWUgZnJvbSB0aGUgdW5kZXJseWluZyBsYW1iZGEgc28gdGhhdCBhbGFybXMgb24gYW4gYWxpYXNcbiAgICAgICAgICAgICAgICAvLyBkb24ndCBjYXVzZSBhIGNpcmN1bGFyIGRlcGVuZGVuY3kgd2l0aCBDb2RlRGVwbG95XG4gICAgICAgICAgICAgICAgLy8gc2VlOiBodHRwczovL2dpdGh1Yi5jb20vYXdzL2F3cy1jZGsvaXNzdWVzLzIyMzFcbiAgICAgICAgICAgICAgICBSZXNvdXJjZTogYCR7dGhpcy5sYW1iZGEuZnVuY3Rpb25OYW1lfToke3RoaXMuYWxpYXNOYW1lfWAsXG4gICAgICAgICAgICB9LFxuICAgICAgICAgICAgLi4ucHJvcHMsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBDYWxjdWxhdGUgdGhlIHJvdXRpbmdDb25maWcgcGFyYW1ldGVyIGZyb20gdGhlIGlucHV0IHByb3BzXG4gICAgICovXG4gICAgcHJpdmF0ZSBkZXRlcm1pbmVSb3V0aW5nQ29uZmlnKHByb3BzOiBBbGlhc1Byb3BzKSB7XG4gICAgICAgIGlmICghcHJvcHMuYWRkaXRpb25hbFZlcnNpb25zIHx8IHByb3BzLmFkZGl0aW9uYWxWZXJzaW9ucy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICAgICAgdGhpcy52YWxpZGF0ZUFkZGl0aW9uYWxXZWlnaHRzKHByb3BzLmFkZGl0aW9uYWxWZXJzaW9ucyk7XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBhZGRpdGlvbmFsVmVyc2lvbldlaWdodHM6IHByb3BzLmFkZGl0aW9uYWxWZXJzaW9ucy5tYXAodncgPT4ge1xuICAgICAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgICAgIGZ1bmN0aW9uVmVyc2lvbjogdncudmVyc2lvbi52ZXJzaW9uLFxuICAgICAgICAgICAgICAgICAgICBmdW5jdGlvbldlaWdodDogdncud2VpZ2h0LFxuICAgICAgICAgICAgICAgIH07XG4gICAgICAgICAgICB9KSxcbiAgICAgICAgfTtcbiAgICB9XG4gICAgLyoqXG4gICAgICogVmFsaWRhdGUgdGhhdCB0aGUgYWRkaXRpb25hbCB2ZXJzaW9uIHdlaWdodHMgbWFrZSBzZW5zZVxuICAgICAqXG4gICAgICogV2UgdmFsaWRhdGUgdGhhdCB0aGV5IGFyZSBwb3NpdGl2ZSBhbmQgYWRkIHVwIHRvIHNvbWV0aGluZyA8PSAxLlxuICAgICAqL1xuICAgIHByaXZhdGUgdmFsaWRhdGVBZGRpdGlvbmFsV2VpZ2h0cyh3ZWlnaHRzOiBWZXJzaW9uV2VpZ2h0W10pIHtcbiAgICAgICAgY29uc3QgdG90YWwgPSB3ZWlnaHRzLm1hcCh3ID0+IHtcbiAgICAgICAgICAgIGlmICh3LndlaWdodCA8IDAgfHwgdy53ZWlnaHQgPiAxKSB7XG4gICAgICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKGBBZGRpdGlvbmFsIHZlcnNpb24gd2VpZ2h0IG11c3QgYmUgYmV0d2VlbiAwIGFuZCAxLCBnb3Q6ICR7dy53ZWlnaHR9YCk7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgICByZXR1cm4gdy53ZWlnaHQ7XG4gICAgICAgIH0pLnJlZHVjZSgoYSwgeCkgPT4gYSArIHgpO1xuICAgICAgICBpZiAodG90YWwgPiAxKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFN1bSBvZiBhZGRpdGlvbmFsIHZlcnNpb24gd2VpZ2h0cyBtdXN0IG5vdCBleGNlZWQgMSwgZ290OiAke3RvdGFsfWApO1xuICAgICAgICB9XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFZhbGlkYXRlIHRoYXQgdGhlIHByb3Zpc2lvbmVkQ29uY3VycmVudEV4ZWN1dGlvbnMgbWFrZXMgc2Vuc2VcbiAgICAgKlxuICAgICAqIE1lbWJlciBtdXN0IGhhdmUgdmFsdWUgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIDFcbiAgICAgKi9cbiAgICBwcml2YXRlIGRldGVybWluZVByb3Zpc2lvbmVkQ29uY3VycmVuY3kocHJvcHM6IEFsaWFzUHJvcHMpOiBDZm5BbGlhcy5Qcm92aXNpb25lZENvbmN1cnJlbmN5Q29uZmlndXJhdGlvblByb3BlcnR5IHwgdW5kZWZpbmVkIHtcbiAgICAgICAgaWYgKCFwcm9wcy5wcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIGlmIChwcm9wcy5wcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zIDw9IDApIHtcbiAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcigncHJvdmlzaW9uZWRDb25jdXJyZW50RXhlY3V0aW9ucyBtdXN0IGhhdmUgdmFsdWUgZ3JlYXRlciB0aGFuIG9yIGVxdWFsIHRvIDEnKTtcbiAgICAgICAgfVxuICAgICAgICByZXR1cm4geyBwcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zOiBwcm9wcy5wcm92aXNpb25lZENvbmN1cnJlbnRFeGVjdXRpb25zIH07XG4gICAgfVxufVxuLyoqXG4gKiBBIHZlcnNpb24vd2VpZ2h0IHBhaXIgZm9yIHJvdXRpbmcgdHJhZmZpYyB0byBMYW1iZGEgZnVuY3Rpb25zXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgVmVyc2lvbldlaWdodCB7XG4gICAgLyoqXG4gICAgICogVGhlIHZlcnNpb24gdG8gcm91dGUgdHJhZmZpYyB0b1xuICAgICAqL1xuICAgIHJlYWRvbmx5IHZlcnNpb246IElWZXJzaW9uO1xuICAgIC8qKlxuICAgICAqIEhvdyBtdWNoIHdlaWdodCB0byBhc3NpZ24gdG8gdGhpcyB2ZXJzaW9uICgwLi4xKVxuICAgICAqL1xuICAgIHJlYWRvbmx5IHdlaWdodDogbnVtYmVyO1xufVxuIl19